/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amp.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.amp.AmpAsyncClient;
import software.amazon.awssdk.services.amp.model.DescribeAnomalyDetectorRequest;
import software.amazon.awssdk.services.amp.model.DescribeAnomalyDetectorResponse;
import software.amazon.awssdk.services.amp.model.DescribeScraperRequest;
import software.amazon.awssdk.services.amp.model.DescribeScraperResponse;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface AmpAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorActive} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeAnomalyDetectorRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorActive(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorActive} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAnomalyDetectorRequest#builder()}
     *
     * @param describeAnomalyDetectorRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorActive(
            Consumer<DescribeAnomalyDetectorRequest.Builder> describeAnomalyDetectorRequest) {
        return waitUntilAnomalyDetectorActive(DescribeAnomalyDetectorRequest.builder()
                .applyMutation(describeAnomalyDetectorRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorActive} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeAnomalyDetectorRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorActive(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorActive} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAnomalyDetectorRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorActive(
            Consumer<DescribeAnomalyDetectorRequest.Builder> describeAnomalyDetectorRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAnomalyDetectorActive(
                DescribeAnomalyDetectorRequest.builder().applyMutation(describeAnomalyDetectorRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeAnomalyDetectorRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorDeleted(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeAnomalyDetectorRequest#builder()}
     *
     * @param describeAnomalyDetectorRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorDeleted(
            Consumer<DescribeAnomalyDetectorRequest.Builder> describeAnomalyDetectorRequest) {
        return waitUntilAnomalyDetectorDeleted(DescribeAnomalyDetectorRequest.builder()
                .applyMutation(describeAnomalyDetectorRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeAnomalyDetectorRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorDeleted(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeAnomalyDetector} API until the desired condition
     * {@code AnomalyDetectorDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeAnomalyDetectorRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeAnomalyDetectorResponse>> waitUntilAnomalyDetectorDeleted(
            Consumer<DescribeAnomalyDetectorRequest.Builder> describeAnomalyDetectorRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilAnomalyDetectorDeleted(
                DescribeAnomalyDetectorRequest.builder().applyMutation(describeAnomalyDetectorRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeScraperRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            DescribeScraperRequest describeScraperRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeScraperRequest#builder()}
     *
     * @param describeScraperRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            Consumer<DescribeScraperRequest.Builder> describeScraperRequest) {
        return waitUntilScraperActive(DescribeScraperRequest.builder().applyMutation(describeScraperRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeScraperRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            DescribeScraperRequest describeScraperRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeScraperRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperActive(
            Consumer<DescribeScraperRequest.Builder> describeScraperRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilScraperActive(DescribeScraperRequest.builder().applyMutation(describeScraperRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeScraperRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            DescribeScraperRequest describeScraperRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeScraperRequest#builder()}
     *
     * @param describeScraperRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            Consumer<DescribeScraperRequest.Builder> describeScraperRequest) {
        return waitUntilScraperDeleted(DescribeScraperRequest.builder().applyMutation(describeScraperRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeScraperRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            DescribeScraperRequest describeScraperRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeScraper} API until the desired condition {@code ScraperDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeScraperRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeScraperResponse>> waitUntilScraperDeleted(
            Consumer<DescribeScraperRequest.Builder> describeScraperRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilScraperDeleted(DescribeScraperRequest.builder().applyMutation(describeScraperRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeWorkspaceRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            DescribeWorkspaceRequest describeWorkspaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeWorkspaceRequest#builder()}
     *
     * @param describeWorkspaceRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            Consumer<DescribeWorkspaceRequest.Builder> describeWorkspaceRequest) {
        return waitUntilWorkspaceActive(DescribeWorkspaceRequest.builder().applyMutation(describeWorkspaceRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeWorkspaceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            DescribeWorkspaceRequest describeWorkspaceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeWorkspaceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceActive(
            Consumer<DescribeWorkspaceRequest.Builder> describeWorkspaceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilWorkspaceActive(DescribeWorkspaceRequest.builder().applyMutation(describeWorkspaceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeWorkspaceRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            DescribeWorkspaceRequest describeWorkspaceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeWorkspaceRequest#builder()}
     *
     * @param describeWorkspaceRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            Consumer<DescribeWorkspaceRequest.Builder> describeWorkspaceRequest) {
        return waitUntilWorkspaceDeleted(DescribeWorkspaceRequest.builder().applyMutation(describeWorkspaceRequest).build());
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeWorkspaceRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            DescribeWorkspaceRequest describeWorkspaceRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link AmpAsyncClient#describeWorkspace} API until the desired condition {@code WorkspaceDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeWorkspaceRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeWorkspaceResponse>> waitUntilWorkspaceDeleted(
            Consumer<DescribeWorkspaceRequest.Builder> describeWorkspaceRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilWorkspaceDeleted(DescribeWorkspaceRequest.builder().applyMutation(describeWorkspaceRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link AmpAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultAmpAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link AmpAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link AmpAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link AmpAsyncWaiter}
     */
    static AmpAsyncWaiter create() {
        return DefaultAmpAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link AmpAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(AmpAsyncClient client);

        /**
         * Builds an instance of {@link AmpAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link AmpAsyncWaiter}
         */
        AmpAsyncWaiter build();
    }
}
