/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplify;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplify.model.AmplifyException;
import software.amazon.awssdk.services.amplify.model.BadRequestException;
import software.amazon.awssdk.services.amplify.model.CreateAppRequest;
import software.amazon.awssdk.services.amplify.model.CreateAppResponse;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.CreateBranchRequest;
import software.amazon.awssdk.services.amplify.model.CreateBranchResponse;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.CreateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.CreateWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DeleteAppRequest;
import software.amazon.awssdk.services.amplify.model.DeleteAppResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBranchRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBranchResponse;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.DeleteJobRequest;
import software.amazon.awssdk.services.amplify.model.DeleteJobResponse;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DependentServiceFailureException;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsRequest;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsResponse;
import software.amazon.awssdk.services.amplify.model.GetAppRequest;
import software.amazon.awssdk.services.amplify.model.GetAppResponse;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlRequest;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlResponse;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.GetBranchRequest;
import software.amazon.awssdk.services.amplify.model.GetBranchResponse;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.GetJobRequest;
import software.amazon.awssdk.services.amplify.model.GetJobResponse;
import software.amazon.awssdk.services.amplify.model.GetWebhookRequest;
import software.amazon.awssdk.services.amplify.model.GetWebhookResponse;
import software.amazon.awssdk.services.amplify.model.InternalFailureException;
import software.amazon.awssdk.services.amplify.model.LimitExceededException;
import software.amazon.awssdk.services.amplify.model.ListAppsRequest;
import software.amazon.awssdk.services.amplify.model.ListAppsResponse;
import software.amazon.awssdk.services.amplify.model.ListArtifactsRequest;
import software.amazon.awssdk.services.amplify.model.ListArtifactsResponse;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsRequest;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsResponse;
import software.amazon.awssdk.services.amplify.model.ListBranchesRequest;
import software.amazon.awssdk.services.amplify.model.ListBranchesResponse;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsRequest;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsResponse;
import software.amazon.awssdk.services.amplify.model.ListJobsRequest;
import software.amazon.awssdk.services.amplify.model.ListJobsResponse;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.amplify.model.ListWebhooksRequest;
import software.amazon.awssdk.services.amplify.model.ListWebhooksResponse;
import software.amazon.awssdk.services.amplify.model.NotFoundException;
import software.amazon.awssdk.services.amplify.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amplify.model.StartDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.StartDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.StartJobRequest;
import software.amazon.awssdk.services.amplify.model.StartJobResponse;
import software.amazon.awssdk.services.amplify.model.StopJobRequest;
import software.amazon.awssdk.services.amplify.model.StopJobResponse;
import software.amazon.awssdk.services.amplify.model.TagResourceRequest;
import software.amazon.awssdk.services.amplify.model.TagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UnauthorizedException;
import software.amazon.awssdk.services.amplify.model.UntagResourceRequest;
import software.amazon.awssdk.services.amplify.model.UntagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UpdateAppRequest;
import software.amazon.awssdk.services.amplify.model.UpdateAppResponse;
import software.amazon.awssdk.services.amplify.model.UpdateBranchRequest;
import software.amazon.awssdk.services.amplify.model.UpdateBranchResponse;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookResponse;
import software.amazon.awssdk.services.amplify.transform.CreateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GenerateAccessLogsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetArtifactUrlRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListArtifactsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBackendEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBranchesRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListDomainAssociationsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListWebhooksRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StopJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateWebhookRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmplifyAsyncClient}.
 *
 * @see AmplifyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyAsyncClient implements AmplifyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmplifyAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new Amplify app.
     * </p>
     *
     * @param createAppRequest
     *        The request structure used to create apps in Amplify.
     * @return A Java Future containing the result of the CreateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppResponse> createApp(CreateAppRequest createAppRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppRequest, CreateAppResponse>().withOperationName("CreateApp")
                            .withMarshaller(new CreateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAppRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAppRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new backend environment for an Amplify app.
     * </p>
     *
     * @param createBackendEnvironmentRequest
     *        The request structure for the backend environment create request.
     * @return A Java Future containing the result of the CreateBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateBackendEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendEnvironmentResponse> createBackendEnvironment(
            CreateBackendEnvironmentRequest createBackendEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendEnvironmentRequest, CreateBackendEnvironmentResponse>()
                            .withOperationName("CreateBackendEnvironment")
                            .withMarshaller(new CreateBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createBackendEnvironmentRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new branch for an Amplify app.
     * </p>
     *
     * @param createBranchRequest
     *        The request structure for the create branch request.
     * @return A Java Future containing the result of the CreateBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBranchResponse> createBranch(CreateBranchRequest createBranchRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBranchRequest, CreateBranchResponse>()
                            .withOperationName("CreateBranch").withMarshaller(new CreateBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBranchRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createBranchRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBranchRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a deployment for a manually deployed Amplify app. Manually deployed apps are not connected to a
     * repository.
     * </p>
     *
     * @param createDeploymentRequest
     *        The request structure for the create a new deployment request.
     * @return A Java Future containing the result of the CreateDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentResponse> createDeployment(CreateDeploymentRequest createDeploymentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                            .withOperationName("CreateDeployment")
                            .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app. This action associates a custom domain with the Amplify app
     * </p>
     *
     * @param createDomainAssociationRequest
     *        The request structure for the create domain association request.
     * @return A Java Future containing the result of the CreateDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainAssociationResponse> createDomainAssociation(
            CreateDomainAssociationRequest createDomainAssociationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDomainAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainAssociationRequest, CreateDomainAssociationResponse>()
                            .withOperationName("CreateDomainAssociation")
                            .withMarshaller(new CreateDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDomainAssociationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainAssociationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new webhook on an Amplify app.
     * </p>
     *
     * @param createWebhookRequest
     *        The request structure for the create webhook request.
     * @return A Java Future containing the result of the CreateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWebhookResponse> createWebhook(CreateWebhookRequest createWebhookRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWebhookRequest, CreateWebhookResponse>()
                            .withOperationName("CreateWebhook")
                            .withMarshaller(new CreateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWebhookRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createWebhookRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebhookRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amplify app specified by an app ID.
     * </p>
     *
     * @param deleteAppRequest
     *        Describes the request structure for the delete app request.
     * @return A Java Future containing the result of the DeleteApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppResponse> deleteApp(DeleteAppRequest deleteAppRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>().withOperationName("DeleteApp")
                            .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAppRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAppRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a backend environment for an Amplify app.
     * </p>
     *
     * @param deleteBackendEnvironmentRequest
     *        The request structure for the delete backend environment request.
     * @return A Java Future containing the result of the DeleteBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteBackendEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendEnvironmentResponse> deleteBackendEnvironment(
            DeleteBackendEnvironmentRequest deleteBackendEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBackendEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendEnvironmentRequest, DeleteBackendEnvironmentResponse>()
                            .withOperationName("DeleteBackendEnvironment")
                            .withMarshaller(new DeleteBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackendEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBackendEnvironmentRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a branch for an Amplify app.
     * </p>
     *
     * @param deleteBranchRequest
     *        The request structure for the delete branch request.
     * @return A Java Future containing the result of the DeleteBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBranchResponse> deleteBranch(DeleteBranchRequest deleteBranchRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBranchRequest, DeleteBranchResponse>()
                            .withOperationName("DeleteBranch").withMarshaller(new DeleteBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBranchRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBranchRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBranchRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a domain association for an Amplify app.
     * </p>
     *
     * @param deleteDomainAssociationRequest
     *        The request structure for the delete domain association request.
     * @return A Java Future containing the result of the DeleteDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainAssociationResponse> deleteDomainAssociation(
            DeleteDomainAssociationRequest deleteDomainAssociationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDomainAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainAssociationRequest, DeleteDomainAssociationResponse>()
                            .withOperationName("DeleteDomainAssociation")
                            .withMarshaller(new DeleteDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDomainAssociationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainAssociationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a job for a branch of an Amplify app.
     * </p>
     *
     * @param deleteJobRequest
     *        The request structure for the delete job request.
     * @return A Java Future containing the result of the DeleteJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobResponse> deleteJob(DeleteJobRequest deleteJobRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>().withOperationName("DeleteJob")
                            .withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a webhook.
     * </p>
     *
     * @param deleteWebhookRequest
     *        The request structure for the delete webhook request.
     * @return A Java Future containing the result of the DeleteWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWebhookResponse> deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                            .withOperationName("DeleteWebhook")
                            .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWebhookRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteWebhookRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWebhookRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the website access logs for a specific time range using a presigned URL.
     * </p>
     *
     * @param generateAccessLogsRequest
     *        The request structure for the generate access logs request.
     * @return A Java Future containing the result of the GenerateAccessLogs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GenerateAccessLogs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GenerateAccessLogs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateAccessLogsResponse> generateAccessLogs(GenerateAccessLogsRequest generateAccessLogsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateAccessLogs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateAccessLogsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateAccessLogsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateAccessLogsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateAccessLogsRequest, GenerateAccessLogsResponse>()
                            .withOperationName("GenerateAccessLogs")
                            .withMarshaller(new GenerateAccessLogsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(generateAccessLogsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = generateAccessLogsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GenerateAccessLogsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateAccessLogsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing Amplify app by appID.
     * </p>
     *
     * @param getAppRequest
     *        The request structure for the get app request.
     * @return A Java Future containing the result of the GetApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppResponse> getApp(GetAppRequest getAppRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppRequest, GetAppResponse>().withOperationName("GetApp")
                            .withMarshaller(new GetAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAppRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the artifact info that corresponds to an artifact id.
     * </p>
     *
     * @param getArtifactUrlRequest
     *        Returns the request structure for the get artifact request.
     * @return A Java Future containing the result of the GetArtifactUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetArtifactUrl
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetArtifactUrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetArtifactUrlResponse> getArtifactUrl(GetArtifactUrlRequest getArtifactUrlRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArtifactUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetArtifactUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetArtifactUrlResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetArtifactUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetArtifactUrlRequest, GetArtifactUrlResponse>()
                            .withOperationName("GetArtifactUrl")
                            .withMarshaller(new GetArtifactUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getArtifactUrlRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getArtifactUrlRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetArtifactUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getArtifactUrlRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a backend environment for an Amplify app.
     * </p>
     *
     * @param getBackendEnvironmentRequest
     *        The request structure for the get backend environment request.
     * @return A Java Future containing the result of the GetBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetBackendEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBackendEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendEnvironmentResponse> getBackendEnvironment(
            GetBackendEnvironmentRequest getBackendEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBackendEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendEnvironmentRequest, GetBackendEnvironmentResponse>()
                            .withOperationName("GetBackendEnvironment")
                            .withMarshaller(new GetBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBackendEnvironmentRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a branch for an Amplify app.
     * </p>
     *
     * @param getBranchRequest
     *        The request structure for the get branch request.
     * @return A Java Future containing the result of the GetBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBranchResponse> getBranch(GetBranchRequest getBranchRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBranchRequest, GetBranchResponse>().withOperationName("GetBranch")
                            .withMarshaller(new GetBranchRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBranchRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBranchRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBranchRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the domain information for an Amplify app.
     * </p>
     *
     * @param getDomainAssociationRequest
     *        The request structure for the get domain association request.
     * @return A Java Future containing the result of the GetDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetDomainAssociation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainAssociationResponse> getDomainAssociation(
            GetDomainAssociationRequest getDomainAssociationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainAssociationRequest, GetDomainAssociationResponse>()
                            .withOperationName("GetDomainAssociation")
                            .withMarshaller(new GetDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDomainAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDomainAssociationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainAssociationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a job for a branch of an Amplify app.
     * </p>
     *
     * @param getJobRequest
     *        The request structure for the get job request.
     * @return A Java Future containing the result of the GetJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobResponse> getJob(GetJobRequest getJobRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                            .withMarshaller(new GetJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the webhook information that corresponds to a specified webhook ID.
     * </p>
     *
     * @param getWebhookRequest
     *        The request structure for the get webhook request.
     * @return A Java Future containing the result of the GetWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetWebhookResponse> getWebhook(GetWebhookRequest getWebhookRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWebhookRequest, GetWebhookResponse>().withOperationName("GetWebhook")
                            .withMarshaller(new GetWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWebhookRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getWebhookRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWebhookRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the existing Amplify apps.
     * </p>
     *
     * @param listAppsRequest
     *        The request structure for the list apps request.
     * @return A Java Future containing the result of the ListApps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListApps
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsResponse> listApps(ListAppsRequest listAppsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>().withOperationName("ListApps")
                            .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAppsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAppsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAppsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of artifacts for a specified app, branch, and job.
     * </p>
     *
     * @param listArtifactsRequest
     *        Describes the request structure for the list artifacts request.
     * @return A Java Future containing the result of the ListArtifacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListArtifacts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListArtifacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListArtifactsResponse> listArtifacts(ListArtifactsRequest listArtifactsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArtifacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListArtifactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListArtifactsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListArtifactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListArtifactsRequest, ListArtifactsResponse>()
                            .withOperationName("ListArtifacts")
                            .withMarshaller(new ListArtifactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listArtifactsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listArtifactsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListArtifactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArtifactsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the backend environments for an Amplify app.
     * </p>
     *
     * @param listBackendEnvironmentsRequest
     *        The request structure for the list backend environments request.
     * @return A Java Future containing the result of the ListBackendEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListBackendEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBackendEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBackendEnvironmentsResponse> listBackendEnvironments(
            ListBackendEnvironmentsRequest listBackendEnvironmentsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackendEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBackendEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBackendEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBackendEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBackendEnvironmentsRequest, ListBackendEnvironmentsResponse>()
                            .withOperationName("ListBackendEnvironments")
                            .withMarshaller(new ListBackendEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBackendEnvironmentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listBackendEnvironmentsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListBackendEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackendEnvironmentsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the branches of an Amplify app.
     * </p>
     *
     * @param listBranchesRequest
     *        The request structure for the list branches request.
     * @return A Java Future containing the result of the ListBranches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListBranches
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBranches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBranchesResponse> listBranches(ListBranchesRequest listBranchesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBranches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBranchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBranchesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBranchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBranchesRequest, ListBranchesResponse>()
                            .withOperationName("ListBranches").withMarshaller(new ListBranchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBranchesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listBranchesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListBranchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBranchesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the domain associations for an Amplify app.
     * </p>
     *
     * @param listDomainAssociationsRequest
     *        The request structure for the list domain associations request.
     * @return A Java Future containing the result of the ListDomainAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListDomainAssociations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListDomainAssociations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainAssociationsResponse> listDomainAssociations(
            ListDomainAssociationsRequest listDomainAssociationsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDomainAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainAssociationsRequest, ListDomainAssociationsResponse>()
                            .withOperationName("ListDomainAssociations")
                            .withMarshaller(new ListDomainAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainAssociationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDomainAssociationsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListDomainAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainAssociationsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the jobs for a branch of an Amplify app.
     * </p>
     *
     * @param listJobsRequest
     *        The request structure for the list jobs request.
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listJobsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request structure to use to list tags for a resource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of webhooks for an Amplify app.
     * </p>
     *
     * @param listWebhooksRequest
     *        The request structure for the list webhooks request.
     * @return A Java Future containing the result of the ListWebhooks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListWebhooks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListWebhooks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebhooksResponse> listWebhooks(ListWebhooksRequest listWebhooksRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebhooks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebhooksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWebhooksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWebhooksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebhooksRequest, ListWebhooksResponse>()
                            .withOperationName("ListWebhooks").withMarshaller(new ListWebhooksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWebhooksRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listWebhooksRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListWebhooksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWebhooksRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a deployment for a manually deployed app. Manually deployed apps are not connected to a repository.
     * </p>
     *
     * @param startDeploymentRequest
     *        The request structure for the start a deployment request.
     * @return A Java Future containing the result of the StartDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StartDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeploymentResponse> startDeployment(StartDeploymentRequest startDeploymentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                            .withOperationName("StartDeployment")
                            .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new job for a branch of an Amplify app.
     * </p>
     *
     * @param startJobRequest
     *        The request structure for the start job request.
     * @return A Java Future containing the result of the StartJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StartJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobResponse> startJob(StartJobRequest startJobRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>().withOperationName("StartJob")
                            .withMarshaller(new StartJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(startJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a job that is in progress for a branch of an Amplify app.
     * </p>
     *
     * @param stopJobRequest
     *        The request structure for the stop job request.
     * @return A Java Future containing the result of the StopJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StopJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StopJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopJobResponse> stopJob(StopJobRequest stopJobRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopJobRequest, StopJobResponse>().withOperationName("StopJob")
                            .withMarshaller(new StopJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags the resource with a tag key and value.
     * </p>
     *
     * @param tagResourceRequest
     *        The request structure to tag a resource with a tag key and value.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags a resource with a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param untagResourceRequest
     *        The request structure for the untag resource request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Amplify app.
     * </p>
     *
     * @param updateAppRequest
     *        The request structure for the update app request.
     * @return A Java Future containing the result of the UpdateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateApp
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppResponse> updateApp(UpdateAppRequest updateAppRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAppResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppRequest, UpdateAppResponse>().withOperationName("UpdateApp")
                            .withMarshaller(new UpdateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAppRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateAppRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a branch for an Amplify app.
     * </p>
     *
     * @param updateBranchRequest
     *        The request structure for the update branch request.
     * @return A Java Future containing the result of the UpdateBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateBranch
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBranchResponse> updateBranch(UpdateBranchRequest updateBranchRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBranchResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBranchRequest, UpdateBranchResponse>()
                            .withOperationName("UpdateBranch").withMarshaller(new UpdateBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBranchRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateBranchRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBranchRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app.
     * </p>
     *
     * @param updateDomainAssociationRequest
     *        The request structure for the update domain association request.
     * @return A Java Future containing the result of the UpdateDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateDomainAssociation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainAssociationResponse> updateDomainAssociation(
            UpdateDomainAssociationRequest updateDomainAssociationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainAssociationRequest, UpdateDomainAssociationResponse>()
                            .withOperationName("UpdateDomainAssociation")
                            .withMarshaller(new UpdateDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDomainAssociationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDomainAssociationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainAssociationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a webhook.
     * </p>
     *
     * @param updateWebhookRequest
     *        The request structure for the update webhook request.
     * @return A Java Future containing the result of the UpdateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateWebhook
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWebhookResponse> updateWebhook(UpdateWebhookRequest updateWebhookRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateWebhookResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWebhookRequest, UpdateWebhookResponse>()
                            .withOperationName("UpdateWebhook")
                            .withMarshaller(new UpdateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWebhookRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateWebhookRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWebhookRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                                .exceptionBuilderSupplier(DependentServiceFailureException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
