/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplify;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplify.model.AmplifyException;
import software.amazon.awssdk.services.amplify.model.BadRequestException;
import software.amazon.awssdk.services.amplify.model.CreateAppRequest;
import software.amazon.awssdk.services.amplify.model.CreateAppResponse;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.CreateBranchRequest;
import software.amazon.awssdk.services.amplify.model.CreateBranchResponse;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.CreateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.CreateWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DeleteAppRequest;
import software.amazon.awssdk.services.amplify.model.DeleteAppResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBranchRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBranchResponse;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.DeleteJobRequest;
import software.amazon.awssdk.services.amplify.model.DeleteJobResponse;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DependentServiceFailureException;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsRequest;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsResponse;
import software.amazon.awssdk.services.amplify.model.GetAppRequest;
import software.amazon.awssdk.services.amplify.model.GetAppResponse;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlRequest;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlResponse;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.GetBranchRequest;
import software.amazon.awssdk.services.amplify.model.GetBranchResponse;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.GetJobRequest;
import software.amazon.awssdk.services.amplify.model.GetJobResponse;
import software.amazon.awssdk.services.amplify.model.GetWebhookRequest;
import software.amazon.awssdk.services.amplify.model.GetWebhookResponse;
import software.amazon.awssdk.services.amplify.model.InternalFailureException;
import software.amazon.awssdk.services.amplify.model.LimitExceededException;
import software.amazon.awssdk.services.amplify.model.ListAppsRequest;
import software.amazon.awssdk.services.amplify.model.ListAppsResponse;
import software.amazon.awssdk.services.amplify.model.ListArtifactsRequest;
import software.amazon.awssdk.services.amplify.model.ListArtifactsResponse;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsRequest;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsResponse;
import software.amazon.awssdk.services.amplify.model.ListBranchesRequest;
import software.amazon.awssdk.services.amplify.model.ListBranchesResponse;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsRequest;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsResponse;
import software.amazon.awssdk.services.amplify.model.ListJobsRequest;
import software.amazon.awssdk.services.amplify.model.ListJobsResponse;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.amplify.model.ListWebhooksRequest;
import software.amazon.awssdk.services.amplify.model.ListWebhooksResponse;
import software.amazon.awssdk.services.amplify.model.NotFoundException;
import software.amazon.awssdk.services.amplify.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amplify.model.StartDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.StartDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.StartJobRequest;
import software.amazon.awssdk.services.amplify.model.StartJobResponse;
import software.amazon.awssdk.services.amplify.model.StopJobRequest;
import software.amazon.awssdk.services.amplify.model.StopJobResponse;
import software.amazon.awssdk.services.amplify.model.TagResourceRequest;
import software.amazon.awssdk.services.amplify.model.TagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UnauthorizedException;
import software.amazon.awssdk.services.amplify.model.UntagResourceRequest;
import software.amazon.awssdk.services.amplify.model.UntagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UpdateAppRequest;
import software.amazon.awssdk.services.amplify.model.UpdateAppResponse;
import software.amazon.awssdk.services.amplify.model.UpdateBranchRequest;
import software.amazon.awssdk.services.amplify.model.UpdateBranchResponse;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookResponse;
import software.amazon.awssdk.services.amplify.transform.CreateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GenerateAccessLogsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetArtifactUrlRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListArtifactsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBackendEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBranchesRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListDomainAssociationsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListWebhooksRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StopJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateWebhookRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AmplifyClient}.
 *
 * @see AmplifyClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyClient implements AmplifyClient {
    private static final Logger log = Logger.loggerFor(DefaultAmplifyClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new Amplify app.
     * </p>
     *
     * @param createAppRequest
     *        The request structure used to create apps in Amplify.
     * @return Result of the CreateApp operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAppResponse createApp(CreateAppRequest createAppRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, LimitExceededException, DependentServiceFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApp");

            return clientHandler.execute(new ClientExecutionParams<CreateAppRequest, CreateAppResponse>()
                    .withOperationName("CreateApp").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAppRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new backend environment for an Amplify app.
     * </p>
     *
     * @param createBackendEnvironmentRequest
     *        The request structure for the backend environment create request.
     * @return Result of the CreateBackendEnvironment operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateBackendEnvironmentResponse createBackendEnvironment(
            CreateBackendEnvironmentRequest createBackendEnvironmentRequest) throws BadRequestException, UnauthorizedException,
            NotFoundException, InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBackendEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendEnvironment");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateBackendEnvironmentRequest, CreateBackendEnvironmentResponse>()
                            .withOperationName("CreateBackendEnvironment").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createBackendEnvironmentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateBackendEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new branch for an Amplify app.
     * </p>
     *
     * @param createBranchRequest
     *        The request structure for the create branch request.
     * @return Result of the CreateBranch operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBranchResponse createBranch(CreateBranchRequest createBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, LimitExceededException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBranch");

            return clientHandler.execute(new ClientExecutionParams<CreateBranchRequest, CreateBranchResponse>()
                    .withOperationName("CreateBranch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBranchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBranchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a deployment for a manually deployed Amplify app. Manually deployed apps are not connected to a
     * repository.
     * </p>
     *
     * @param createDeploymentRequest
     *        The request structure for the create a new deployment request.
     * @return Result of the CreateDeployment operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDeploymentResponse createDeployment(CreateDeploymentRequest createDeploymentRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");

            return clientHandler.execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                    .withOperationName("CreateDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app. This action associates a custom domain with the Amplify app
     * </p>
     *
     * @param createDomainAssociationRequest
     *        The request structure for the create domain association request.
     * @return Result of the CreateDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDomainAssociationResponse createDomainAssociation(CreateDomainAssociationRequest createDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            LimitExceededException, DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomainAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDomainAssociationRequest, CreateDomainAssociationResponse>()
                            .withOperationName("CreateDomainAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDomainAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDomainAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new webhook on an Amplify app.
     * </p>
     *
     * @param createWebhookRequest
     *        The request structure for the create webhook request.
     * @return Result of the CreateWebhook operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.CreateWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWebhookResponse createWebhook(CreateWebhookRequest createWebhookRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, LimitExceededException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebhook");

            return clientHandler.execute(new ClientExecutionParams<CreateWebhookRequest, CreateWebhookResponse>()
                    .withOperationName("CreateWebhook").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWebhookRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWebhookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amplify app specified by an app ID.
     * </p>
     *
     * @param deleteAppRequest
     *        Describes the request structure for the delete app request.
     * @return Result of the DeleteApp operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAppResponse deleteApp(DeleteAppRequest deleteAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, DependentServiceFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApp");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>()
                    .withOperationName("DeleteApp").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a backend environment for an Amplify app.
     * </p>
     *
     * @param deleteBackendEnvironmentRequest
     *        The request structure for the delete backend environment request.
     * @return Result of the DeleteBackendEnvironment operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteBackendEnvironmentResponse deleteBackendEnvironment(
            DeleteBackendEnvironmentRequest deleteBackendEnvironmentRequest) throws BadRequestException, UnauthorizedException,
            NotFoundException, InternalFailureException, DependentServiceFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBackendEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendEnvironment");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendEnvironmentRequest, DeleteBackendEnvironmentResponse>()
                            .withOperationName("DeleteBackendEnvironment").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackendEnvironmentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteBackendEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a branch for an Amplify app.
     * </p>
     *
     * @param deleteBranchRequest
     *        The request structure for the delete branch request.
     * @return Result of the DeleteBranch operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBranchResponse deleteBranch(DeleteBranchRequest deleteBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, DependentServiceFailureException,
            AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBranch");

            return clientHandler.execute(new ClientExecutionParams<DeleteBranchRequest, DeleteBranchResponse>()
                    .withOperationName("DeleteBranch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBranchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBranchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a domain association for an Amplify app.
     * </p>
     *
     * @param deleteDomainAssociationRequest
     *        The request structure for the delete domain association request.
     * @return Result of the DeleteDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDomainAssociationResponse deleteDomainAssociation(DeleteDomainAssociationRequest deleteDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomainAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainAssociationRequest, DeleteDomainAssociationResponse>()
                            .withOperationName("DeleteDomainAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDomainAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDomainAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a job for a branch of an Amplify app.
     * </p>
     *
     * @param deleteJobRequest
     *        The request structure for the delete job request.
     * @return Result of the DeleteJob operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteJobResponse deleteJob(DeleteJobRequest deleteJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");

            return clientHandler.execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>()
                    .withOperationName("DeleteJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a webhook.
     * </p>
     *
     * @param deleteWebhookRequest
     *        The request structure for the delete webhook request.
     * @return Result of the DeleteWebhook operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.DeleteWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteWebhookResponse deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWebhook");

            return clientHandler.execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                    .withOperationName("DeleteWebhook").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWebhookRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the website access logs for a specific time range using a presigned URL.
     * </p>
     *
     * @param generateAccessLogsRequest
     *        The request structure for the generate access logs request.
     * @return Result of the GenerateAccessLogs operation returned by the service.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GenerateAccessLogs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GenerateAccessLogs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GenerateAccessLogsResponse generateAccessLogs(GenerateAccessLogsRequest generateAccessLogsRequest)
            throws NotFoundException, BadRequestException, UnauthorizedException, InternalFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateAccessLogsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GenerateAccessLogsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateAccessLogsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateAccessLogs");

            return clientHandler.execute(new ClientExecutionParams<GenerateAccessLogsRequest, GenerateAccessLogsResponse>()
                    .withOperationName("GenerateAccessLogs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(generateAccessLogsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GenerateAccessLogsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an existing Amplify app by appID.
     * </p>
     *
     * @param getAppRequest
     *        The request structure for the get app request.
     * @return Result of the GetApp operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAppResponse getApp(GetAppRequest getAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApp");

            return clientHandler.execute(new ClientExecutionParams<GetAppRequest, GetAppResponse>().withOperationName("GetApp")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getAppRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the artifact info that corresponds to an artifact id.
     * </p>
     *
     * @param getArtifactUrlRequest
     *        Returns the request structure for the get artifact request.
     * @return Result of the GetArtifactUrl operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetArtifactUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetArtifactUrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetArtifactUrlResponse getArtifactUrl(GetArtifactUrlRequest getArtifactUrlRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetArtifactUrlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetArtifactUrlResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getArtifactUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArtifactUrl");

            return clientHandler.execute(new ClientExecutionParams<GetArtifactUrlRequest, GetArtifactUrlResponse>()
                    .withOperationName("GetArtifactUrl").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getArtifactUrlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetArtifactUrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a backend environment for an Amplify app.
     * </p>
     *
     * @param getBackendEnvironmentRequest
     *        The request structure for the get backend environment request.
     * @return Result of the GetBackendEnvironment operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBackendEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetBackendEnvironmentResponse getBackendEnvironment(GetBackendEnvironmentRequest getBackendEnvironmentRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackendEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendEnvironment");

            return clientHandler.execute(new ClientExecutionParams<GetBackendEnvironmentRequest, GetBackendEnvironmentResponse>()
                    .withOperationName("GetBackendEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBackendEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBackendEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a branch for an Amplify app.
     * </p>
     *
     * @param getBranchRequest
     *        The request structure for the get branch request.
     * @return Result of the GetBranch operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBranchResponse getBranch(GetBranchRequest getBranchRequest) throws BadRequestException, UnauthorizedException,
            NotFoundException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBranch");

            return clientHandler.execute(new ClientExecutionParams<GetBranchRequest, GetBranchResponse>()
                    .withOperationName("GetBranch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBranchRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetBranchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the domain information for an Amplify app.
     * </p>
     *
     * @param getDomainAssociationRequest
     *        The request structure for the get domain association request.
     * @return Result of the GetDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetDomainAssociation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetDomainAssociationResponse getDomainAssociation(GetDomainAssociationRequest getDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainAssociation");

            return clientHandler.execute(new ClientExecutionParams<GetDomainAssociationRequest, GetDomainAssociationResponse>()
                    .withOperationName("GetDomainAssociation").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDomainAssociationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDomainAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a job for a branch of an Amplify app.
     * </p>
     *
     * @param getJobRequest
     *        The request structure for the get job request.
     * @return Result of the GetJob operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetJobResponse getJob(GetJobRequest getJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");

            return clientHandler.execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the webhook information that corresponds to a specified webhook ID.
     * </p>
     *
     * @param getWebhookRequest
     *        The request structure for the get webhook request.
     * @return Result of the GetWebhook operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.GetWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetWebhookResponse getWebhook(GetWebhookRequest getWebhookRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWebhook");

            return clientHandler
                    .execute(new ClientExecutionParams<GetWebhookRequest, GetWebhookResponse>().withOperationName("GetWebhook")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getWebhookRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetWebhookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the existing Amplify apps.
     * </p>
     *
     * @param listAppsRequest
     *        The request structure for the list apps request.
     * @return Result of the ListApps operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListApps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAppsResponse listApps(ListAppsRequest listAppsRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApps");

            return clientHandler.execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>()
                    .withOperationName("ListApps").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAppsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListAppsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of artifacts for a specified app, branch, and job.
     * </p>
     *
     * @param listArtifactsRequest
     *        Describes the request structure for the list artifacts request.
     * @return Result of the ListArtifacts operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListArtifacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListArtifacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListArtifactsResponse listArtifacts(ListArtifactsRequest listArtifactsRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListArtifactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListArtifactsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArtifactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArtifacts");

            return clientHandler.execute(new ClientExecutionParams<ListArtifactsRequest, ListArtifactsResponse>()
                    .withOperationName("ListArtifacts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listArtifactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListArtifactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the backend environments for an Amplify app.
     * </p>
     *
     * @param listBackendEnvironmentsRequest
     *        The request structure for the list backend environments request.
     * @return Result of the ListBackendEnvironments operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListBackendEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBackendEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBackendEnvironmentsResponse listBackendEnvironments(ListBackendEnvironmentsRequest listBackendEnvironmentsRequest)
            throws BadRequestException, UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackendEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBackendEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackendEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackendEnvironments");

            return clientHandler
                    .execute(new ClientExecutionParams<ListBackendEnvironmentsRequest, ListBackendEnvironmentsResponse>()
                            .withOperationName("ListBackendEnvironments").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listBackendEnvironmentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListBackendEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the branches of an Amplify app.
     * </p>
     *
     * @param listBranchesRequest
     *        The request structure for the list branches request.
     * @return Result of the ListBranches operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListBranches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBranches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBranchesResponse listBranches(ListBranchesRequest listBranchesRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBranchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBranchesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBranchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBranches");

            return clientHandler.execute(new ClientExecutionParams<ListBranchesRequest, ListBranchesResponse>()
                    .withOperationName("ListBranches").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBranchesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBranchesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the domain associations for an Amplify app.
     * </p>
     *
     * @param listDomainAssociationsRequest
     *        The request structure for the list domain associations request.
     * @return Result of the ListDomainAssociations operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListDomainAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListDomainAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDomainAssociationsResponse listDomainAssociations(ListDomainAssociationsRequest listDomainAssociationsRequest)
            throws BadRequestException, UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDomainAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDomainAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDomainAssociationsRequest, ListDomainAssociationsResponse>()
                            .withOperationName("ListDomainAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDomainAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDomainAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the jobs for a branch of an Amplify app.
     * </p>
     *
     * @param listJobsRequest
     *        The request structure for the list jobs request.
     * @return Result of the ListJobs operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request structure to use to list tags for a resource.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws ResourceNotFoundException
     *         An operation failed due to a non-existent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalFailureException, BadRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of webhooks for an Amplify app.
     * </p>
     *
     * @param listWebhooksRequest
     *        The request structure for the list webhooks request.
     * @return Result of the ListWebhooks operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.ListWebhooks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListWebhooks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWebhooksResponse listWebhooks(ListWebhooksRequest listWebhooksRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWebhooksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWebhooksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWebhooksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebhooks");

            return clientHandler.execute(new ClientExecutionParams<ListWebhooksRequest, ListWebhooksResponse>()
                    .withOperationName("ListWebhooks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWebhooksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWebhooksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a deployment for a manually deployed app. Manually deployed apps are not connected to a repository.
     * </p>
     *
     * @param startDeploymentRequest
     *        The request structure for the start a deployment request.
     * @return Result of the StartDeployment operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartDeploymentResponse startDeployment(StartDeploymentRequest startDeploymentRequest) throws BadRequestException,
            UnauthorizedException, InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException,
            SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");

            return clientHandler.execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                    .withOperationName("StartDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a new job for a branch of an Amplify app.
     * </p>
     *
     * @param startJobRequest
     *        The request structure for the start job request.
     * @return Result of the StartJob operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.StartJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartJobResponse startJob(StartJobRequest startJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJob");

            return clientHandler.execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>()
                    .withOperationName("StartJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new StartJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a job that is in progress for a branch of an Amplify app.
     * </p>
     *
     * @param stopJobRequest
     *        The request structure for the stop job request.
     * @return Result of the StopJob operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws LimitExceededException
     *         A resource could not be created because service quotas were exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.StopJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StopJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopJobResponse stopJob(StopJobRequest stopJobRequest) throws BadRequestException, UnauthorizedException,
            InternalFailureException, NotFoundException, LimitExceededException, AwsServiceException, SdkClientException,
            AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopJob");

            return clientHandler.execute(new ClientExecutionParams<StopJobRequest, StopJobResponse>()
                    .withOperationName("StopJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new StopJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags the resource with a tag key and value.
     * </p>
     *
     * @param tagResourceRequest
     *        The request structure to tag a resource with a tag key and value.
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws ResourceNotFoundException
     *         An operation failed due to a non-existent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalFailureException,
            BadRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untags a resource with a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param untagResourceRequest
     *        The request structure for the untag resource request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws ResourceNotFoundException
     *         An operation failed due to a non-existent resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalFailureException,
            BadRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Amplify app.
     * </p>
     *
     * @param updateAppRequest
     *        The request structure for the update app request.
     * @return Result of the UpdateApp operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAppResponse updateApp(UpdateAppRequest updateAppRequest) throws BadRequestException, NotFoundException,
            UnauthorizedException, InternalFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAppResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApp");

            return clientHandler.execute(new ClientExecutionParams<UpdateAppRequest, UpdateAppResponse>()
                    .withOperationName("UpdateApp").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAppRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new UpdateAppRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a branch for an Amplify app.
     * </p>
     *
     * @param updateBranchRequest
     *        The request structure for the update branch request.
     * @return Result of the UpdateBranch operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBranchResponse updateBranch(UpdateBranchRequest updateBranchRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, DependentServiceFailureException,
            AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBranchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBranch");

            return clientHandler.execute(new ClientExecutionParams<UpdateBranchRequest, UpdateBranchResponse>()
                    .withOperationName("UpdateBranch").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateBranchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBranchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app.
     * </p>
     *
     * @param updateDomainAssociationRequest
     *        The request structure for the update domain association request.
     * @return Result of the UpdateDomainAssociation operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDomainAssociationResponse updateDomainAssociation(UpdateDomainAssociationRequest updateDomainAssociationRequest)
            throws BadRequestException, UnauthorizedException, NotFoundException, InternalFailureException,
            DependentServiceFailureException, AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDomainAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainAssociationRequest, UpdateDomainAssociationResponse>()
                            .withOperationName("UpdateDomainAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDomainAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDomainAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a webhook.
     * </p>
     *
     * @param updateWebhookRequest
     *        The request structure for the update webhook request.
     * @return Result of the UpdateWebhook operation returned by the service.
     * @throws BadRequestException
     *         A request contains unexpected data.
     * @throws UnauthorizedException
     *         An operation failed due to a lack of access.
     * @throws NotFoundException
     *         An entity was not found during an operation.
     * @throws InternalFailureException
     *         The service failed to perform an operation due to an internal issue.
     * @throws DependentServiceFailureException
     *         An operation failed because a dependent service threw an exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AmplifyException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AmplifyClient.UpdateWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateWebhookResponse updateWebhook(UpdateWebhookRequest updateWebhookRequest) throws BadRequestException,
            UnauthorizedException, NotFoundException, InternalFailureException, DependentServiceFailureException,
            AwsServiceException, SdkClientException, AmplifyException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWebhookResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWebhook");

            return clientHandler.execute(new ClientExecutionParams<UpdateWebhookRequest, UpdateWebhookResponse>()
                    .withOperationName("UpdateWebhook").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWebhookRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWebhookRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                                .exceptionBuilderSupplier(DependentServiceFailureException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
