/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplify;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.amplify.internal.AmplifyServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.amplify.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.amplify.model.AmplifyException;
import software.amazon.awssdk.services.amplify.model.BadRequestException;
import software.amazon.awssdk.services.amplify.model.CreateAppRequest;
import software.amazon.awssdk.services.amplify.model.CreateAppResponse;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.CreateBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.CreateBranchRequest;
import software.amazon.awssdk.services.amplify.model.CreateBranchResponse;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.CreateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.CreateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.CreateWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DeleteAppRequest;
import software.amazon.awssdk.services.amplify.model.DeleteAppResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.DeleteBranchRequest;
import software.amazon.awssdk.services.amplify.model.DeleteBranchResponse;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.DeleteDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.DeleteJobRequest;
import software.amazon.awssdk.services.amplify.model.DeleteJobResponse;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookRequest;
import software.amazon.awssdk.services.amplify.model.DeleteWebhookResponse;
import software.amazon.awssdk.services.amplify.model.DependentServiceFailureException;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsRequest;
import software.amazon.awssdk.services.amplify.model.GenerateAccessLogsResponse;
import software.amazon.awssdk.services.amplify.model.GetAppRequest;
import software.amazon.awssdk.services.amplify.model.GetAppResponse;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlRequest;
import software.amazon.awssdk.services.amplify.model.GetArtifactUrlResponse;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentRequest;
import software.amazon.awssdk.services.amplify.model.GetBackendEnvironmentResponse;
import software.amazon.awssdk.services.amplify.model.GetBranchRequest;
import software.amazon.awssdk.services.amplify.model.GetBranchResponse;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.GetDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.GetJobRequest;
import software.amazon.awssdk.services.amplify.model.GetJobResponse;
import software.amazon.awssdk.services.amplify.model.GetWebhookRequest;
import software.amazon.awssdk.services.amplify.model.GetWebhookResponse;
import software.amazon.awssdk.services.amplify.model.InternalFailureException;
import software.amazon.awssdk.services.amplify.model.LimitExceededException;
import software.amazon.awssdk.services.amplify.model.ListAppsRequest;
import software.amazon.awssdk.services.amplify.model.ListAppsResponse;
import software.amazon.awssdk.services.amplify.model.ListArtifactsRequest;
import software.amazon.awssdk.services.amplify.model.ListArtifactsResponse;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsRequest;
import software.amazon.awssdk.services.amplify.model.ListBackendEnvironmentsResponse;
import software.amazon.awssdk.services.amplify.model.ListBranchesRequest;
import software.amazon.awssdk.services.amplify.model.ListBranchesResponse;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsRequest;
import software.amazon.awssdk.services.amplify.model.ListDomainAssociationsResponse;
import software.amazon.awssdk.services.amplify.model.ListJobsRequest;
import software.amazon.awssdk.services.amplify.model.ListJobsResponse;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.amplify.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.amplify.model.ListWebhooksRequest;
import software.amazon.awssdk.services.amplify.model.ListWebhooksResponse;
import software.amazon.awssdk.services.amplify.model.NotFoundException;
import software.amazon.awssdk.services.amplify.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amplify.model.StartDeploymentRequest;
import software.amazon.awssdk.services.amplify.model.StartDeploymentResponse;
import software.amazon.awssdk.services.amplify.model.StartJobRequest;
import software.amazon.awssdk.services.amplify.model.StartJobResponse;
import software.amazon.awssdk.services.amplify.model.StopJobRequest;
import software.amazon.awssdk.services.amplify.model.StopJobResponse;
import software.amazon.awssdk.services.amplify.model.TagResourceRequest;
import software.amazon.awssdk.services.amplify.model.TagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UnauthorizedException;
import software.amazon.awssdk.services.amplify.model.UntagResourceRequest;
import software.amazon.awssdk.services.amplify.model.UntagResourceResponse;
import software.amazon.awssdk.services.amplify.model.UpdateAppRequest;
import software.amazon.awssdk.services.amplify.model.UpdateAppResponse;
import software.amazon.awssdk.services.amplify.model.UpdateBranchRequest;
import software.amazon.awssdk.services.amplify.model.UpdateBranchResponse;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationRequest;
import software.amazon.awssdk.services.amplify.model.UpdateDomainAssociationResponse;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookRequest;
import software.amazon.awssdk.services.amplify.model.UpdateWebhookResponse;
import software.amazon.awssdk.services.amplify.transform.CreateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.CreateWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.DeleteWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GenerateAccessLogsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetArtifactUrlRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBackendEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.GetWebhookRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListAppsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListArtifactsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBackendEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListBranchesRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListDomainAssociationsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.ListWebhooksRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StartJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.StopJobRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateAppRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateBranchRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateDomainAssociationRequestMarshaller;
import software.amazon.awssdk.services.amplify.transform.UpdateWebhookRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmplifyAsyncClient}.
 *
 * @see AmplifyAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyAsyncClient implements AmplifyAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmplifyAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Amplify" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new Amplify app.
     * </p>
     *
     * @param createAppRequest
     *        The request structure used to create apps in Amplify.
     * @return A Java Future containing the result of the CreateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppResponse> createApp(CreateAppRequest createAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppRequest, CreateAppResponse>().withOperationName("CreateApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAppRequest));
            CompletableFuture<CreateAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new backend environment for an Amplify app.
     * </p>
     * <p>
     * This API is available only to Amplify Gen 1 applications where the backend is created using Amplify Studio or the
     * Amplify command line interface (CLI). This API isn’t available to Amplify Gen 2 applications. When you deploy an
     * application with Amplify Gen 2, you provision the app's backend infrastructure using Typescript code.
     * </p>
     *
     * @param createBackendEnvironmentRequest
     *        The request structure for the backend environment create request.
     * @return A Java Future containing the result of the CreateBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendEnvironmentResponse> createBackendEnvironment(
            CreateBackendEnvironmentRequest createBackendEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBackendEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendEnvironmentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendEnvironmentRequest, CreateBackendEnvironmentResponse>()
                            .withOperationName("CreateBackendEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBackendEnvironmentRequest));
            CompletableFuture<CreateBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new branch for an Amplify app.
     * </p>
     *
     * @param createBranchRequest
     *        The request structure for the create branch request.
     * @return A Java Future containing the result of the CreateBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBranchResponse> createBranch(CreateBranchRequest createBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBranchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBranchRequest, CreateBranchResponse>()
                            .withOperationName("CreateBranch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBranchRequest));
            CompletableFuture<CreateBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a deployment for a manually deployed Amplify app. Manually deployed apps are not connected to a Git
     * repository.
     * </p>
     * <p>
     * The maximum duration between the <code>CreateDeployment</code> call and the <code>StartDeployment</code> call
     * cannot exceed 8 hours. If the duration exceeds 8 hours, the <code>StartDeployment</code> call and the associated
     * <code>Job</code> will fail.
     * </p>
     *
     * @param createDeploymentRequest
     *        The request structure for the create a new deployment request.
     * @return A Java Future containing the result of the CreateDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentResponse> createDeployment(CreateDeploymentRequest createDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                            .withOperationName("CreateDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDeploymentRequest));
            CompletableFuture<CreateDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app. This action associates a custom domain with the Amplify app
     * </p>
     *
     * @param createDomainAssociationRequest
     *        The request structure for the create domain association request.
     * @return A Java Future containing the result of the CreateDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainAssociationResponse> createDomainAssociation(
            CreateDomainAssociationRequest createDomainAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDomainAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDomainAssociationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainAssociationRequest, CreateDomainAssociationResponse>()
                            .withOperationName("CreateDomainAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDomainAssociationRequest));
            CompletableFuture<CreateDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new webhook on an Amplify app.
     * </p>
     *
     * @param createWebhookRequest
     *        The request structure for the create webhook request.
     * @return A Java Future containing the result of the CreateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.CreateWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/CreateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWebhookResponse> createWebhook(CreateWebhookRequest createWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateWebhookResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWebhookRequest, CreateWebhookResponse>()
                            .withOperationName("CreateWebhook").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWebhookRequest));
            CompletableFuture<CreateWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amplify app specified by an app ID.
     * </p>
     *
     * @param deleteAppRequest
     *        Describes the request structure for the delete app request.
     * @return A Java Future containing the result of the DeleteApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppResponse> deleteApp(DeleteAppRequest deleteAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppRequest, DeleteAppResponse>().withOperationName("DeleteApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppRequest));
            CompletableFuture<DeleteAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a backend environment for an Amplify app.
     * </p>
     * <p>
     * This API is available only to Amplify Gen 1 applications where the backend is created using Amplify Studio or the
     * Amplify command line interface (CLI). This API isn’t available to Amplify Gen 2 applications. When you deploy an
     * application with Amplify Gen 2, you provision the app's backend infrastructure using Typescript code.
     * </p>
     *
     * @param deleteBackendEnvironmentRequest
     *        The request structure for the delete backend environment request.
     * @return A Java Future containing the result of the DeleteBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBackendEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendEnvironmentResponse> deleteBackendEnvironment(
            DeleteBackendEnvironmentRequest deleteBackendEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackendEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBackendEnvironmentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendEnvironmentRequest, DeleteBackendEnvironmentResponse>()
                            .withOperationName("DeleteBackendEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBackendEnvironmentRequest));
            CompletableFuture<DeleteBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a branch for an Amplify app.
     * </p>
     *
     * @param deleteBranchRequest
     *        The request structure for the delete branch request.
     * @return A Java Future containing the result of the DeleteBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBranchResponse> deleteBranch(DeleteBranchRequest deleteBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBranchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBranchRequest, DeleteBranchResponse>()
                            .withOperationName("DeleteBranch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBranchRequest));
            CompletableFuture<DeleteBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a domain association for an Amplify app.
     * </p>
     *
     * @param deleteDomainAssociationRequest
     *        The request structure for the delete domain association request.
     * @return A Java Future containing the result of the DeleteDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainAssociationResponse> deleteDomainAssociation(
            DeleteDomainAssociationRequest deleteDomainAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDomainAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDomainAssociationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainAssociationRequest, DeleteDomainAssociationResponse>()
                            .withOperationName("DeleteDomainAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDomainAssociationRequest));
            CompletableFuture<DeleteDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a job for a branch of an Amplify app.
     * </p>
     *
     * @param deleteJobRequest
     *        The request structure for the delete job request.
     * @return A Java Future containing the result of the DeleteJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobResponse> deleteJob(DeleteJobRequest deleteJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobRequest, DeleteJobResponse>().withOperationName("DeleteJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteJobRequest));
            CompletableFuture<DeleteJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a webhook.
     * </p>
     *
     * @param deleteWebhookRequest
     *        The request structure for the delete webhook request.
     * @return A Java Future containing the result of the DeleteWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.DeleteWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/DeleteWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWebhookResponse> deleteWebhook(DeleteWebhookRequest deleteWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWebhookResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWebhookRequest, DeleteWebhookResponse>()
                            .withOperationName("DeleteWebhook").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWebhookRequest));
            CompletableFuture<DeleteWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the website access logs for a specific time range using a presigned URL.
     * </p>
     *
     * @param generateAccessLogsRequest
     *        The request structure for the generate access logs request.
     * @return A Java Future containing the result of the GenerateAccessLogs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GenerateAccessLogs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GenerateAccessLogs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateAccessLogsResponse> generateAccessLogs(GenerateAccessLogsRequest generateAccessLogsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateAccessLogsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateAccessLogsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateAccessLogs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateAccessLogsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateAccessLogsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GenerateAccessLogsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateAccessLogsRequest, GenerateAccessLogsResponse>()
                            .withOperationName("GenerateAccessLogs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GenerateAccessLogsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(generateAccessLogsRequest));
            CompletableFuture<GenerateAccessLogsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing Amplify app specified by an app ID.
     * </p>
     *
     * @param getAppRequest
     *        The request structure for the get app request.
     * @return A Java Future containing the result of the GetApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppResponse> getApp(GetAppRequest getAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppRequest, GetAppResponse>().withOperationName("GetApp")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetAppRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAppRequest));
            CompletableFuture<GetAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the artifact info that corresponds to an artifact id.
     * </p>
     *
     * @param getArtifactUrlRequest
     *        Returns the request structure for the get artifact request.
     * @return A Java Future containing the result of the GetArtifactUrl operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetArtifactUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetArtifactUrl" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetArtifactUrlResponse> getArtifactUrl(GetArtifactUrlRequest getArtifactUrlRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getArtifactUrlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getArtifactUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArtifactUrl");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetArtifactUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetArtifactUrlResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetArtifactUrlResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetArtifactUrlRequest, GetArtifactUrlResponse>()
                            .withOperationName("GetArtifactUrl").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetArtifactUrlRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getArtifactUrlRequest));
            CompletableFuture<GetArtifactUrlResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a backend environment for an Amplify app.
     * </p>
     * <p>
     * This API is available only to Amplify Gen 1 applications where the backend is created using Amplify Studio or the
     * Amplify command line interface (CLI). This API isn’t available to Amplify Gen 2 applications. When you deploy an
     * application with Amplify Gen 2, you provision the app's backend infrastructure using Typescript code.
     * </p>
     *
     * @param getBackendEnvironmentRequest
     *        The request structure for the get backend environment request.
     * @return A Java Future containing the result of the GetBackendEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetBackendEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBackendEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendEnvironmentResponse> getBackendEnvironment(
            GetBackendEnvironmentRequest getBackendEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBackendEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBackendEnvironmentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetBackendEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendEnvironmentRequest, GetBackendEnvironmentResponse>()
                            .withOperationName("GetBackendEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBackendEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBackendEnvironmentRequest));
            CompletableFuture<GetBackendEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a branch for an Amplify app.
     * </p>
     *
     * @param getBranchRequest
     *        The request structure for the get branch request.
     * @return A Java Future containing the result of the GetBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBranchResponse> getBranch(GetBranchRequest getBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBranchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBranchRequest, GetBranchResponse>().withOperationName("GetBranch")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBranchRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBranchRequest));
            CompletableFuture<GetBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the domain information for an Amplify app.
     * </p>
     *
     * @param getDomainAssociationRequest
     *        The request structure for the get domain association request.
     * @return A Java Future containing the result of the GetDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetDomainAssociation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainAssociationResponse> getDomainAssociation(
            GetDomainAssociationRequest getDomainAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDomainAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDomainAssociationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainAssociationRequest, GetDomainAssociationResponse>()
                            .withOperationName("GetDomainAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDomainAssociationRequest));
            CompletableFuture<GetDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a job for a branch of an Amplify app.
     * </p>
     *
     * @param getJobRequest
     *        The request structure for the get job request.
     * @return A Java Future containing the result of the GetJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobResponse> getJob(GetJobRequest getJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobRequest));
            CompletableFuture<GetJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the webhook information that corresponds to a specified webhook ID.
     * </p>
     *
     * @param getWebhookRequest
     *        The request structure for the get webhook request.
     * @return A Java Future containing the result of the GetWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.GetWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/GetWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetWebhookResponse> getWebhook(GetWebhookRequest getWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWebhookResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWebhookRequest, GetWebhookResponse>().withOperationName("GetWebhook")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWebhookRequest));
            CompletableFuture<GetWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the existing Amplify apps.
     * </p>
     *
     * @param listAppsRequest
     *        The request structure for the list apps request.
     * @return A Java Future containing the result of the ListApps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListApps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListApps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppsResponse> listApps(ListAppsRequest listAppsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAppsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAppsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAppsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppsRequest, ListAppsResponse>().withOperationName("ListApps")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAppsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppsRequest));
            CompletableFuture<ListAppsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of end-to-end testing artifacts for a specified app, branch, and job.
     * </p>
     * <p>
     * To return the build artifacts, use the <a
     * href="https://docs.aws.amazon.com/amplify/latest/APIReference/API_GetJob.html">GetJob</a> API.
     * </p>
     * <p>
     * For more information about Amplify testing support, see <a
     * href="https://docs.aws.amazon.com/amplify/latest/userguide/running-tests.html">Setting up end-to-end Cypress
     * tests for your Amplify application</a> in the <i>Amplify Hosting User Guide</i>.
     * </p>
     *
     * @param listArtifactsRequest
     *        Describes the request structure for the list artifacts request.
     * @return A Java Future containing the result of the ListArtifacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListArtifacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListArtifacts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListArtifactsResponse> listArtifacts(ListArtifactsRequest listArtifactsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listArtifactsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listArtifactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListArtifacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListArtifactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListArtifactsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListArtifactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListArtifactsRequest, ListArtifactsResponse>()
                            .withOperationName("ListArtifacts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListArtifactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listArtifactsRequest));
            CompletableFuture<ListArtifactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the backend environments for an Amplify app.
     * </p>
     * <p>
     * This API is available only to Amplify Gen 1 applications where the backend is created using Amplify Studio or the
     * Amplify command line interface (CLI). This API isn’t available to Amplify Gen 2 applications. When you deploy an
     * application with Amplify Gen 2, you provision the app's backend infrastructure using Typescript code.
     * </p>
     *
     * @param listBackendEnvironmentsRequest
     *        The request structure for the list backend environments request.
     * @return A Java Future containing the result of the ListBackendEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListBackendEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBackendEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBackendEnvironmentsResponse> listBackendEnvironments(
            ListBackendEnvironmentsRequest listBackendEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBackendEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackendEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackendEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBackendEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBackendEnvironmentsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListBackendEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBackendEnvironmentsRequest, ListBackendEnvironmentsResponse>()
                            .withOperationName("ListBackendEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBackendEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBackendEnvironmentsRequest));
            CompletableFuture<ListBackendEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the branches of an Amplify app.
     * </p>
     *
     * @param listBranchesRequest
     *        The request structure for the list branches request.
     * @return A Java Future containing the result of the ListBranches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListBranches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListBranches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBranchesResponse> listBranches(ListBranchesRequest listBranchesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBranchesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBranchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBranches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBranchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBranchesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListBranchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBranchesRequest, ListBranchesResponse>()
                            .withOperationName("ListBranches").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBranchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBranchesRequest));
            CompletableFuture<ListBranchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the domain associations for an Amplify app.
     * </p>
     *
     * @param listDomainAssociationsRequest
     *        The request structure for the list domain associations request.
     * @return A Java Future containing the result of the ListDomainAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListDomainAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListDomainAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainAssociationsResponse> listDomainAssociations(
            ListDomainAssociationsRequest listDomainAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDomainAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomainAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDomainAssociationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListDomainAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainAssociationsRequest, ListDomainAssociationsResponse>()
                            .withOperationName("ListDomainAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDomainAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDomainAssociationsRequest));
            CompletableFuture<ListDomainAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the jobs for a branch of an Amplify app.
     * </p>
     *
     * @param listJobsRequest
     *        The request structure for the list jobs request.
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     *        The request structure to use to list tags for a resource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of webhooks for an Amplify app.
     * </p>
     *
     * @param listWebhooksRequest
     *        The request structure for the list webhooks request.
     * @return A Java Future containing the result of the ListWebhooks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.ListWebhooks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/ListWebhooks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWebhooksResponse> listWebhooks(ListWebhooksRequest listWebhooksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWebhooksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWebhooksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWebhooks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWebhooksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWebhooksResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListWebhooksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWebhooksRequest, ListWebhooksResponse>()
                            .withOperationName("ListWebhooks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWebhooksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWebhooksRequest));
            CompletableFuture<ListWebhooksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a deployment for a manually deployed app. Manually deployed apps are not connected to a Git repository.
     * </p>
     * <p>
     * The maximum duration between the <code>CreateDeployment</code> call and the <code>StartDeployment</code> call
     * cannot exceed 8 hours. If the duration exceeds 8 hours, the <code>StartDeployment</code> call and the associated
     * <code>Job</code> will fail.
     * </p>
     *
     * @param startDeploymentRequest
     *        The request structure for the start a deployment request.
     * @return A Java Future containing the result of the StartDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeploymentResponse> startDeployment(StartDeploymentRequest startDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeploymentResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                            .withOperationName("StartDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startDeploymentRequest));
            CompletableFuture<StartDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a new job for a branch of an Amplify app.
     * </p>
     *
     * @param startJobRequest
     *        The request structure for the start job request.
     * @return A Java Future containing the result of the StartJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StartJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StartJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartJobResponse> startJob(StartJobRequest startJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartJobRequest, StartJobResponse>().withOperationName("StartJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(startJobRequest));
            CompletableFuture<StartJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a job that is in progress for a branch of an Amplify app.
     * </p>
     *
     * @param stopJobRequest
     *        The request structure for the stop job request.
     * @return A Java Future containing the result of the StopJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>LimitExceededException A resource could not be created because service quotas were exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.StopJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/StopJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopJobResponse> stopJob(StopJobRequest stopJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopJobResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StopJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopJobRequest, StopJobResponse>().withOperationName("StopJob")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new StopJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopJobRequest));
            CompletableFuture<StopJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags the resource with a tag key and value.
     * </p>
     *
     * @param tagResourceRequest
     *        The request structure to tag a resource with a tag key and value.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Untags a resource with a specified Amazon Resource Name (ARN).
     * </p>
     *
     * @param untagResourceRequest
     *        The request structure for the untag resource request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>ResourceNotFoundException An operation failed due to a non-existent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Amplify app.
     * </p>
     *
     * @param updateAppRequest
     *        The request structure for the update app request.
     * @return A Java Future containing the result of the UpdateApp operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateApp
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateApp" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppResponse> updateApp(UpdateAppRequest updateAppRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAppRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApp");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAppResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAppResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppRequest, UpdateAppResponse>().withOperationName("UpdateApp")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAppRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAppRequest));
            CompletableFuture<UpdateAppResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a branch for an Amplify app.
     * </p>
     *
     * @param updateBranchRequest
     *        The request structure for the update branch request.
     * @return A Java Future containing the result of the UpdateBranch operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateBranch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateBranch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBranchResponse> updateBranch(UpdateBranchRequest updateBranchRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBranchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBranchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBranch");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBranchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBranchResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateBranchResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBranchRequest, UpdateBranchResponse>()
                            .withOperationName("UpdateBranch").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateBranchRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateBranchRequest));
            CompletableFuture<UpdateBranchResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new domain association for an Amplify app.
     * </p>
     *
     * @param updateDomainAssociationRequest
     *        The request structure for the update domain association request.
     * @return A Java Future containing the result of the UpdateDomainAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateDomainAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateDomainAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainAssociationResponse> updateDomainAssociation(
            UpdateDomainAssociationRequest updateDomainAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDomainAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomainAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDomainAssociationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateDomainAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainAssociationRequest, UpdateDomainAssociationResponse>()
                            .withOperationName("UpdateDomainAssociation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDomainAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDomainAssociationRequest));
            CompletableFuture<UpdateDomainAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a webhook.
     * </p>
     *
     * @param updateWebhookRequest
     *        The request structure for the update webhook request.
     * @return A Java Future containing the result of the UpdateWebhook operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException A request contains unexpected data.</li>
     *         <li>UnauthorizedException An operation failed due to a lack of access.</li>
     *         <li>NotFoundException An entity was not found during an operation.</li>
     *         <li>InternalFailureException The service failed to perform an operation due to an internal issue.</li>
     *         <li>DependentServiceFailureException An operation failed because a dependent service threw an exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AmplifyAsyncClient.UpdateWebhook
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplify-2017-07-25/UpdateWebhook" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWebhookResponse> updateWebhook(UpdateWebhookRequest updateWebhookRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWebhookRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWebhookRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Amplify");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWebhook");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWebhookResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateWebhookResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InternalFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalFailureException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalFailureException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(401)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(429)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DependentServiceFailureException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependentServiceFailureException")
                            .httpStatusCode(503).exceptionBuilderSupplier(DependentServiceFailureException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "BadRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(BadRequestException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateWebhookResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWebhookRequest, UpdateWebhookResponse>()
                            .withOperationName("UpdateWebhook").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWebhookRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWebhookRequest));
            CompletableFuture<UpdateWebhookResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AmplifyServiceClientConfiguration serviceClientConfiguration() {
        return new AmplifyServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(AmplifyException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        AmplifyServiceClientConfigurationBuilder serviceConfigBuilder = new AmplifyServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
