/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplifyuibuilder;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplifyuibuilder.model.AmplifyUiBuilderException;
import software.amazon.awssdk.services.amplifyuibuilder.model.AmplifyUiBuilderRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExchangeCodeForTokenRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExchangeCodeForTokenResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportComponentsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportComponentsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportThemesRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportThemesResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.InternalServerException;
import software.amazon.awssdk.services.amplifyuibuilder.model.InvalidParameterException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.RefreshTokenRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.RefreshTokenResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ResourceConflictException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.paginators.ListComponentsPublisher;
import software.amazon.awssdk.services.amplifyuibuilder.paginators.ListThemesPublisher;
import software.amazon.awssdk.services.amplifyuibuilder.transform.CreateComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.CreateThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.DeleteComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.DeleteThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExchangeCodeForTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExportComponentsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExportThemesRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListThemesRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.RefreshTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.UpdateComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.UpdateThemeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmplifyUiBuilderAsyncClient}.
 *
 * @see AmplifyUiBuilderAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyUiBuilderAsyncClient implements AmplifyUiBuilderAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmplifyUiBuilderAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyUiBuilderAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new component for an Amplify app.
     * </p>
     *
     * @param createComponentRequest
     * @return A Java Future containing the result of the CreateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.CreateComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/CreateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComponentResponse> createComponent(CreateComponentRequest createComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComponentRequest, CreateComponentResponse>()
                            .withOperationName("CreateComponent")
                            .withMarshaller(new CreateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createComponentRequest));
            CompletableFuture<CreateComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a theme to apply to the components in an Amplify app.
     * </p>
     *
     * @param createThemeRequest
     * @return A Java Future containing the result of the CreateTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.CreateTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/CreateTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateThemeResponse> createTheme(CreateThemeRequest createThemeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateThemeRequest, CreateThemeResponse>()
                            .withOperationName("CreateTheme").withMarshaller(new CreateThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createThemeRequest));
            CompletableFuture<CreateThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a component from an Amplify app.
     * </p>
     *
     * @param deleteComponentRequest
     * @return A Java Future containing the result of the DeleteComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.DeleteComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/DeleteComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComponentResponse> deleteComponent(DeleteComponentRequest deleteComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComponentRequest, DeleteComponentResponse>()
                            .withOperationName("DeleteComponent")
                            .withMarshaller(new DeleteComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteComponentRequest));
            CompletableFuture<DeleteComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a theme from an Amplify app.
     * </p>
     *
     * @param deleteThemeRequest
     * @return A Java Future containing the result of the DeleteTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.DeleteTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/DeleteTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteThemeResponse> deleteTheme(DeleteThemeRequest deleteThemeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteThemeRequest, DeleteThemeResponse>()
                            .withOperationName("DeleteTheme").withMarshaller(new DeleteThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteThemeRequest));
            CompletableFuture<DeleteThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exchanges an access code for a token.
     * </p>
     *
     * @param exchangeCodeForTokenRequest
     * @return A Java Future containing the result of the ExchangeCodeForToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExchangeCodeForToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExchangeCodeForToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExchangeCodeForTokenResponse> exchangeCodeForToken(
            ExchangeCodeForTokenRequest exchangeCodeForTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exchangeCodeForTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExchangeCodeForToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExchangeCodeForTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExchangeCodeForTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExchangeCodeForTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExchangeCodeForTokenRequest, ExchangeCodeForTokenResponse>()
                            .withOperationName("ExchangeCodeForToken")
                            .withMarshaller(new ExchangeCodeForTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exchangeCodeForTokenRequest));
            CompletableFuture<ExchangeCodeForTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports component configurations to code that is ready to integrate into an Amplify app.
     * </p>
     *
     * @param exportComponentsRequest
     * @return A Java Future containing the result of the ExportComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExportComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExportComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportComponentsResponse> exportComponents(ExportComponentsRequest exportComponentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportComponentsRequest, ExportComponentsResponse>()
                            .withOperationName("ExportComponents")
                            .withMarshaller(new ExportComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportComponentsRequest));
            CompletableFuture<ExportComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports theme configurations to code that is ready to integrate into an Amplify app.
     * </p>
     *
     * @param exportThemesRequest
     * @return A Java Future containing the result of the ExportThemes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExportThemes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExportThemes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportThemesResponse> exportThemes(ExportThemesRequest exportThemesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportThemesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportThemes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportThemesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportThemesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportThemesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportThemesRequest, ExportThemesResponse>()
                            .withOperationName("ExportThemes").withMarshaller(new ExportThemesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportThemesRequest));
            CompletableFuture<ExportThemesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing component for an Amplify app.
     * </p>
     *
     * @param getComponentRequest
     * @return A Java Future containing the result of the GetComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetComponent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentResponse> getComponent(GetComponentRequest getComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                            .withOperationName("GetComponent").withMarshaller(new GetComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getComponentRequest));
            CompletableFuture<GetComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing theme for an Amplify app.
     * </p>
     *
     * @param getThemeRequest
     * @return A Java Future containing the result of the GetTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetTheme" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetThemeResponse> getTheme(GetThemeRequest getThemeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThemeRequest, GetThemeResponse>().withOperationName("GetTheme")
                            .withMarshaller(new GetThemeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getThemeRequest));
            CompletableFuture<GetThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of components for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listComponentsRequest
     * @return A Java Future containing the result of the ListComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentsResponse> listComponents(ListComponentsRequest listComponentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                            .withOperationName("ListComponents")
                            .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listComponentsRequest));
            CompletableFuture<ListComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of components for a specified Amplify app and backend environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listComponents(software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amplifyuibuilder.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amplifyuibuilder.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComponents(software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listComponentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListComponents"
     *      target="_top">AWS API Documentation</a>
     */
    public ListComponentsPublisher listComponentsPaginator(ListComponentsRequest listComponentsRequest) {
        return new ListComponentsPublisher(this, applyPaginatorUserAgent(listComponentsRequest));
    }

    /**
     * <p>
     * Retrieves a list of themes for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listThemesRequest
     * @return A Java Future containing the result of the ListThemes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListThemes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListThemes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListThemesResponse> listThemes(ListThemesRequest listThemesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listThemesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListThemes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListThemesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListThemesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListThemesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListThemesRequest, ListThemesResponse>().withOperationName("ListThemes")
                            .withMarshaller(new ListThemesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listThemesRequest));
            CompletableFuture<ListThemesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of themes for a specified Amplify app and backend environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listThemes(software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amplifyuibuilder.paginators.ListThemesPublisher publisher = client.listThemesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.amplifyuibuilder.paginators.ListThemesPublisher publisher = client.listThemesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listThemes(software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesRequest)} operation.</b>
     * </p>
     *
     * @param listThemesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListThemes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListThemes" target="_top">AWS
     *      API Documentation</a>
     */
    public ListThemesPublisher listThemesPaginator(ListThemesRequest listThemesRequest) {
        return new ListThemesPublisher(this, applyPaginatorUserAgent(listThemesRequest));
    }

    /**
     * <p>
     * Refreshes a previously issued access token that might have expired.
     * </p>
     *
     * @param refreshTokenRequest
     * @return A Java Future containing the result of the RefreshToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.RefreshToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/RefreshToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RefreshTokenResponse> refreshToken(RefreshTokenRequest refreshTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, refreshTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RefreshToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RefreshTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RefreshTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RefreshTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RefreshTokenRequest, RefreshTokenResponse>()
                            .withOperationName("RefreshToken").withMarshaller(new RefreshTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(refreshTokenRequest));
            CompletableFuture<RefreshTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing component.
     * </p>
     *
     * @param updateComponentRequest
     * @return A Java Future containing the result of the UpdateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.UpdateComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/UpdateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComponentResponse> updateComponent(UpdateComponentRequest updateComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComponentRequest, UpdateComponentResponse>()
                            .withOperationName("UpdateComponent")
                            .withMarshaller(new UpdateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateComponentRequest));
            CompletableFuture<UpdateComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing theme.
     * </p>
     *
     * @param updateThemeRequest
     * @return A Java Future containing the result of the UpdateTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.UpdateTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/UpdateTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateThemeResponse> updateTheme(UpdateThemeRequest updateThemeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateThemeRequest, UpdateThemeResponse>()
                            .withOperationName("UpdateTheme").withMarshaller(new UpdateThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateThemeRequest));
            CompletableFuture<UpdateThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyUiBuilderException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AmplifyUiBuilderRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
