/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplifyuibuilder.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration settings for a user interface (UI) element for an Amplify app. A component is configured
 * as a primary, stand-alone UI element. Use <code>ComponentChild</code> to configure an instance of a
 * <code>Component</code>. A <code>ComponentChild</code> instance inherits the configuration of the main
 * <code>Component</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Component implements SdkPojo, Serializable, ToCopyableBuilder<Component.Builder, Component> {
    private static final SdkField<String> APP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("appId")
            .getter(getter(Component::appId)).setter(setter(Builder::appId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appId").build()).build();

    private static final SdkField<Map<String, ComponentBindingPropertiesValue>> BINDING_PROPERTIES_FIELD = SdkField
            .<Map<String, ComponentBindingPropertiesValue>> builder(MarshallingType.MAP)
            .memberName("bindingProperties")
            .getter(getter(Component::bindingProperties))
            .setter(setter(Builder::bindingProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bindingProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ComponentBindingPropertiesValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentBindingPropertiesValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<ComponentChild>> CHILDREN_FIELD = SdkField
            .<List<ComponentChild>> builder(MarshallingType.LIST)
            .memberName("children")
            .getter(getter(Component::children))
            .setter(setter(Builder::children))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("children").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComponentChild> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentChild::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, ComponentDataConfiguration>> COLLECTION_PROPERTIES_FIELD = SdkField
            .<Map<String, ComponentDataConfiguration>> builder(MarshallingType.MAP)
            .memberName("collectionProperties")
            .getter(getter(Component::collectionProperties))
            .setter(setter(Builder::collectionProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("collectionProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ComponentDataConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentDataConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> COMPONENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("componentType").getter(getter(Component::componentType)).setter(setter(Builder::componentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("componentType").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(Component::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("environmentName").getter(getter(Component::environmentName)).setter(setter(Builder::environmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentName").build()).build();

    private static final SdkField<Map<String, ComponentEvent>> EVENTS_FIELD = SdkField
            .<Map<String, ComponentEvent>> builder(MarshallingType.MAP)
            .memberName("events")
            .getter(getter(Component::events))
            .setter(setter(Builder::events))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("events").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ComponentEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(Component::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(Component::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Component::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, Map<String, String>>> OVERRIDES_FIELD = SdkField
            .<Map<String, Map<String, String>>> builder(MarshallingType.MAP)
            .memberName("overrides")
            .getter(getter(Component::overrides))
            .setter(setter(Builder::overrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrides").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Map<String, String>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Map<String, ComponentProperty>> PROPERTIES_FIELD = SdkField
            .<Map<String, ComponentProperty>> builder(MarshallingType.MAP)
            .memberName("properties")
            .getter(getter(Component::properties))
            .setter(setter(Builder::properties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("properties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ComponentProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("schemaVersion").getter(getter(Component::schemaVersion)).setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schemaVersion").build()).build();

    private static final SdkField<String> SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sourceId").getter(getter(Component::sourceId)).setter(setter(Builder::sourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceId").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Component::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<ComponentVariant>> VARIANTS_FIELD = SdkField
            .<List<ComponentVariant>> builder(MarshallingType.LIST)
            .memberName("variants")
            .getter(getter(Component::variants))
            .setter(setter(Builder::variants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComponentVariant> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComponentVariant::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_ID_FIELD,
            BINDING_PROPERTIES_FIELD, CHILDREN_FIELD, COLLECTION_PROPERTIES_FIELD, COMPONENT_TYPE_FIELD, CREATED_AT_FIELD,
            ENVIRONMENT_NAME_FIELD, EVENTS_FIELD, ID_FIELD, MODIFIED_AT_FIELD, NAME_FIELD, OVERRIDES_FIELD, PROPERTIES_FIELD,
            SCHEMA_VERSION_FIELD, SOURCE_ID_FIELD, TAGS_FIELD, VARIANTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String appId;

    private final Map<String, ComponentBindingPropertiesValue> bindingProperties;

    private final List<ComponentChild> children;

    private final Map<String, ComponentDataConfiguration> collectionProperties;

    private final String componentType;

    private final Instant createdAt;

    private final String environmentName;

    private final Map<String, ComponentEvent> events;

    private final String id;

    private final Instant modifiedAt;

    private final String name;

    private final Map<String, Map<String, String>> overrides;

    private final Map<String, ComponentProperty> properties;

    private final String schemaVersion;

    private final String sourceId;

    private final Map<String, String> tags;

    private final List<ComponentVariant> variants;

    private Component(BuilderImpl builder) {
        this.appId = builder.appId;
        this.bindingProperties = builder.bindingProperties;
        this.children = builder.children;
        this.collectionProperties = builder.collectionProperties;
        this.componentType = builder.componentType;
        this.createdAt = builder.createdAt;
        this.environmentName = builder.environmentName;
        this.events = builder.events;
        this.id = builder.id;
        this.modifiedAt = builder.modifiedAt;
        this.name = builder.name;
        this.overrides = builder.overrides;
        this.properties = builder.properties;
        this.schemaVersion = builder.schemaVersion;
        this.sourceId = builder.sourceId;
        this.tags = builder.tags;
        this.variants = builder.variants;
    }

    /**
     * <p>
     * The unique ID of the Amplify app associated with the component.
     * </p>
     * 
     * @return The unique ID of the Amplify app associated with the component.
     */
    public final String appId() {
        return appId;
    }

    /**
     * For responses, this returns true if the service returned a value for the BindingProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBindingProperties() {
        return bindingProperties != null && !(bindingProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The information to connect a component's properties to data at runtime. You can't specify <code>tags</code> as a
     * valid property for <code>bindingProperties</code>.
     * </p>
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBindingProperties} method.
     * </p>
     * 
     * @return The information to connect a component's properties to data at runtime. You can't specify
     *         <code>tags</code> as a valid property for <code>bindingProperties</code>.</p>
     */
    public final Map<String, ComponentBindingPropertiesValue> bindingProperties() {
        return bindingProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the Children property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasChildren() {
        return children != null && !(children instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the component's <code>ComponentChild</code> instances.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildren} method.
     * </p>
     * 
     * @return A list of the component's <code>ComponentChild</code> instances.
     */
    public final List<ComponentChild> children() {
        return children;
    }

    /**
     * For responses, this returns true if the service returned a value for the CollectionProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCollectionProperties() {
        return collectionProperties != null && !(collectionProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The data binding configuration for the component's properties. Use this for a collection component. You can't
     * specify <code>tags</code> as a valid property for <code>collectionProperties</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCollectionProperties} method.
     * </p>
     * 
     * @return The data binding configuration for the component's properties. Use this for a collection component. You
     *         can't specify <code>tags</code> as a valid property for <code>collectionProperties</code>.
     */
    public final Map<String, ComponentDataConfiguration> collectionProperties() {
        return collectionProperties;
    }

    /**
     * <p>
     * The type of the component. This can be an Amplify custom UI component or another custom component.
     * </p>
     * 
     * @return The type of the component. This can be an Amplify custom UI component or another custom component.
     */
    public final String componentType() {
        return componentType;
    }

    /**
     * <p>
     * The time that the component was created.
     * </p>
     * 
     * @return The time that the component was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The name of the backend environment that is a part of the Amplify app.
     * </p>
     * 
     * @return The name of the backend environment that is a part of the Amplify app.
     */
    public final String environmentName() {
        return environmentName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Events property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEvents() {
        return events != null && !(events instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Describes the events that can be raised on the component. Use for the workflow feature in Amplify Studio that
     * allows you to bind events and actions to components.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEvents} method.
     * </p>
     * 
     * @return Describes the events that can be raised on the component. Use for the workflow feature in Amplify Studio
     *         that allows you to bind events and actions to components.
     */
    public final Map<String, ComponentEvent> events() {
        return events;
    }

    /**
     * <p>
     * The unique ID of the component.
     * </p>
     * 
     * @return The unique ID of the component.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The time that the component was modified.
     * </p>
     * 
     * @return The time that the component was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The name of the component.
     * </p>
     * 
     * @return The name of the component.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Overrides property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOverrides() {
        return overrides != null && !(overrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Describes the component's properties that can be overriden in a customized instance of the component. You can't
     * specify <code>tags</code> as a valid property for <code>overrides</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOverrides} method.
     * </p>
     * 
     * @return Describes the component's properties that can be overriden in a customized instance of the component. You
     *         can't specify <code>tags</code> as a valid property for <code>overrides</code>.
     */
    public final Map<String, Map<String, String>> overrides() {
        return overrides;
    }

    /**
     * For responses, this returns true if the service returned a value for the Properties property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProperties() {
        return properties != null && !(properties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Describes the component's properties. You can't specify <code>tags</code> as a valid property for
     * <code>properties</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProperties} method.
     * </p>
     * 
     * @return Describes the component's properties. You can't specify <code>tags</code> as a valid property for
     *         <code>properties</code>.
     */
    public final Map<String, ComponentProperty> properties() {
        return properties;
    }

    /**
     * <p>
     * The schema version of the component when it was imported.
     * </p>
     * 
     * @return The schema version of the component when it was imported.
     */
    public final String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The unique ID of the component in its original source system, such as Figma.
     * </p>
     * 
     * @return The unique ID of the component in its original source system, such as Figma.
     */
    public final String sourceId() {
        return sourceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more key-value pairs to use when tagging the component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return One or more key-value pairs to use when tagging the component.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variants property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariants() {
        return variants != null && !(variants instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the component's variants. A variant is a unique style configuration of a main component.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariants} method.
     * </p>
     * 
     * @return A list of the component's variants. A variant is a unique style configuration of a main component.
     */
    public final List<ComponentVariant> variants() {
        return variants;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appId());
        hashCode = 31 * hashCode + Objects.hashCode(hasBindingProperties() ? bindingProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasChildren() ? children() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCollectionProperties() ? collectionProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(componentType());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(environmentName());
        hashCode = 31 * hashCode + Objects.hashCode(hasEvents() ? events() : null);
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasOverrides() ? overrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasProperties() ? properties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(sourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVariants() ? variants() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Component)) {
            return false;
        }
        Component other = (Component) obj;
        return Objects.equals(appId(), other.appId()) && hasBindingProperties() == other.hasBindingProperties()
                && Objects.equals(bindingProperties(), other.bindingProperties()) && hasChildren() == other.hasChildren()
                && Objects.equals(children(), other.children()) && hasCollectionProperties() == other.hasCollectionProperties()
                && Objects.equals(collectionProperties(), other.collectionProperties())
                && Objects.equals(componentType(), other.componentType()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(environmentName(), other.environmentName()) && hasEvents() == other.hasEvents()
                && Objects.equals(events(), other.events()) && Objects.equals(id(), other.id())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(name(), other.name())
                && hasOverrides() == other.hasOverrides() && Objects.equals(overrides(), other.overrides())
                && hasProperties() == other.hasProperties() && Objects.equals(properties(), other.properties())
                && Objects.equals(schemaVersion(), other.schemaVersion()) && Objects.equals(sourceId(), other.sourceId())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags()) && hasVariants() == other.hasVariants()
                && Objects.equals(variants(), other.variants());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Component").add("AppId", appId())
                .add("BindingProperties", hasBindingProperties() ? bindingProperties() : null)
                .add("Children", hasChildren() ? children() : null)
                .add("CollectionProperties", hasCollectionProperties() ? collectionProperties() : null)
                .add("ComponentType", componentType()).add("CreatedAt", createdAt()).add("EnvironmentName", environmentName())
                .add("Events", hasEvents() ? events() : null).add("Id", id()).add("ModifiedAt", modifiedAt()).add("Name", name())
                .add("Overrides", hasOverrides() ? overrides() : null).add("Properties", hasProperties() ? properties() : null)
                .add("SchemaVersion", schemaVersion()).add("SourceId", sourceId()).add("Tags", hasTags() ? tags() : null)
                .add("Variants", hasVariants() ? variants() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appId":
            return Optional.ofNullable(clazz.cast(appId()));
        case "bindingProperties":
            return Optional.ofNullable(clazz.cast(bindingProperties()));
        case "children":
            return Optional.ofNullable(clazz.cast(children()));
        case "collectionProperties":
            return Optional.ofNullable(clazz.cast(collectionProperties()));
        case "componentType":
            return Optional.ofNullable(clazz.cast(componentType()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "environmentName":
            return Optional.ofNullable(clazz.cast(environmentName()));
        case "events":
            return Optional.ofNullable(clazz.cast(events()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "overrides":
            return Optional.ofNullable(clazz.cast(overrides()));
        case "properties":
            return Optional.ofNullable(clazz.cast(properties()));
        case "schemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "sourceId":
            return Optional.ofNullable(clazz.cast(sourceId()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "variants":
            return Optional.ofNullable(clazz.cast(variants()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Component, T> g) {
        return obj -> g.apply((Component) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Component> {
        /**
         * <p>
         * The unique ID of the Amplify app associated with the component.
         * </p>
         * 
         * @param appId
         *        The unique ID of the Amplify app associated with the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appId(String appId);

        /**
         * <p>
         * The information to connect a component's properties to data at runtime. You can't specify <code>tags</code>
         * as a valid property for <code>bindingProperties</code>.
         * </p>
         * <p/>
         * 
         * @param bindingProperties
         *        The information to connect a component's properties to data at runtime. You can't specify
         *        <code>tags</code> as a valid property for <code>bindingProperties</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bindingProperties(Map<String, ComponentBindingPropertiesValue> bindingProperties);

        /**
         * <p>
         * A list of the component's <code>ComponentChild</code> instances.
         * </p>
         * 
         * @param children
         *        A list of the component's <code>ComponentChild</code> instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder children(Collection<ComponentChild> children);

        /**
         * <p>
         * A list of the component's <code>ComponentChild</code> instances.
         * </p>
         * 
         * @param children
         *        A list of the component's <code>ComponentChild</code> instances.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder children(ComponentChild... children);

        /**
         * <p>
         * A list of the component's <code>ComponentChild</code> instances.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentChild.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentChild#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentChild.Builder#build()} is called
         * immediately and its result is passed to {@link #children(List<ComponentChild>)}.
         * 
         * @param children
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentChild.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #children(java.util.Collection<ComponentChild>)
         */
        Builder children(Consumer<ComponentChild.Builder>... children);

        /**
         * <p>
         * The data binding configuration for the component's properties. Use this for a collection component. You can't
         * specify <code>tags</code> as a valid property for <code>collectionProperties</code>.
         * </p>
         * 
         * @param collectionProperties
         *        The data binding configuration for the component's properties. Use this for a collection component.
         *        You can't specify <code>tags</code> as a valid property for <code>collectionProperties</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collectionProperties(Map<String, ComponentDataConfiguration> collectionProperties);

        /**
         * <p>
         * The type of the component. This can be an Amplify custom UI component or another custom component.
         * </p>
         * 
         * @param componentType
         *        The type of the component. This can be an Amplify custom UI component or another custom component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder componentType(String componentType);

        /**
         * <p>
         * The time that the component was created.
         * </p>
         * 
         * @param createdAt
         *        The time that the component was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The name of the backend environment that is a part of the Amplify app.
         * </p>
         * 
         * @param environmentName
         *        The name of the backend environment that is a part of the Amplify app.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentName(String environmentName);

        /**
         * <p>
         * Describes the events that can be raised on the component. Use for the workflow feature in Amplify Studio that
         * allows you to bind events and actions to components.
         * </p>
         * 
         * @param events
         *        Describes the events that can be raised on the component. Use for the workflow feature in Amplify
         *        Studio that allows you to bind events and actions to components.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Map<String, ComponentEvent> events);

        /**
         * <p>
         * The unique ID of the component.
         * </p>
         * 
         * @param id
         *        The unique ID of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The time that the component was modified.
         * </p>
         * 
         * @param modifiedAt
         *        The time that the component was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The name of the component.
         * </p>
         * 
         * @param name
         *        The name of the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Describes the component's properties that can be overriden in a customized instance of the component. You
         * can't specify <code>tags</code> as a valid property for <code>overrides</code>.
         * </p>
         * 
         * @param overrides
         *        Describes the component's properties that can be overriden in a customized instance of the component.
         *        You can't specify <code>tags</code> as a valid property for <code>overrides</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(Map<String, ? extends Map<String, String>> overrides);

        /**
         * <p>
         * Describes the component's properties. You can't specify <code>tags</code> as a valid property for
         * <code>properties</code>.
         * </p>
         * 
         * @param properties
         *        Describes the component's properties. You can't specify <code>tags</code> as a valid property for
         *        <code>properties</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder properties(Map<String, ComponentProperty> properties);

        /**
         * <p>
         * The schema version of the component when it was imported.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version of the component when it was imported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The unique ID of the component in its original source system, such as Figma.
         * </p>
         * 
         * @param sourceId
         *        The unique ID of the component in its original source system, such as Figma.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceId(String sourceId);

        /**
         * <p>
         * One or more key-value pairs to use when tagging the component.
         * </p>
         * 
         * @param tags
         *        One or more key-value pairs to use when tagging the component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A list of the component's variants. A variant is a unique style configuration of a main component.
         * </p>
         * 
         * @param variants
         *        A list of the component's variants. A variant is a unique style configuration of a main component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variants(Collection<ComponentVariant> variants);

        /**
         * <p>
         * A list of the component's variants. A variant is a unique style configuration of a main component.
         * </p>
         * 
         * @param variants
         *        A list of the component's variants. A variant is a unique style configuration of a main component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variants(ComponentVariant... variants);

        /**
         * <p>
         * A list of the component's variants. A variant is a unique style configuration of a main component.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentVariant.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentVariant#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentVariant.Builder#build()} is called
         * immediately and its result is passed to {@link #variants(List<ComponentVariant>)}.
         * 
         * @param variants
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.amplifyuibuilder.model.ComponentVariant.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variants(java.util.Collection<ComponentVariant>)
         */
        Builder variants(Consumer<ComponentVariant.Builder>... variants);
    }

    static final class BuilderImpl implements Builder {
        private String appId;

        private Map<String, ComponentBindingPropertiesValue> bindingProperties = DefaultSdkAutoConstructMap.getInstance();

        private List<ComponentChild> children = DefaultSdkAutoConstructList.getInstance();

        private Map<String, ComponentDataConfiguration> collectionProperties = DefaultSdkAutoConstructMap.getInstance();

        private String componentType;

        private Instant createdAt;

        private String environmentName;

        private Map<String, ComponentEvent> events = DefaultSdkAutoConstructMap.getInstance();

        private String id;

        private Instant modifiedAt;

        private String name;

        private Map<String, Map<String, String>> overrides = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, ComponentProperty> properties = DefaultSdkAutoConstructMap.getInstance();

        private String schemaVersion;

        private String sourceId;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<ComponentVariant> variants = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Component model) {
            appId(model.appId);
            bindingProperties(model.bindingProperties);
            children(model.children);
            collectionProperties(model.collectionProperties);
            componentType(model.componentType);
            createdAt(model.createdAt);
            environmentName(model.environmentName);
            events(model.events);
            id(model.id);
            modifiedAt(model.modifiedAt);
            name(model.name);
            overrides(model.overrides);
            properties(model.properties);
            schemaVersion(model.schemaVersion);
            sourceId(model.sourceId);
            tags(model.tags);
            variants(model.variants);
        }

        public final String getAppId() {
            return appId;
        }

        public final void setAppId(String appId) {
            this.appId = appId;
        }

        @Override
        public final Builder appId(String appId) {
            this.appId = appId;
            return this;
        }

        public final Map<String, ComponentBindingPropertiesValue.Builder> getBindingProperties() {
            Map<String, ComponentBindingPropertiesValue.Builder> result = ComponentBindingPropertiesCopier
                    .copyToBuilder(this.bindingProperties);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setBindingProperties(Map<String, ComponentBindingPropertiesValue.BuilderImpl> bindingProperties) {
            this.bindingProperties = ComponentBindingPropertiesCopier.copyFromBuilder(bindingProperties);
        }

        @Override
        public final Builder bindingProperties(Map<String, ComponentBindingPropertiesValue> bindingProperties) {
            this.bindingProperties = ComponentBindingPropertiesCopier.copy(bindingProperties);
            return this;
        }

        public final List<ComponentChild.Builder> getChildren() {
            List<ComponentChild.Builder> result = ComponentChildListCopier.copyToBuilder(this.children);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setChildren(Collection<ComponentChild.BuilderImpl> children) {
            this.children = ComponentChildListCopier.copyFromBuilder(children);
        }

        @Override
        public final Builder children(Collection<ComponentChild> children) {
            this.children = ComponentChildListCopier.copy(children);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder children(ComponentChild... children) {
            children(Arrays.asList(children));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder children(Consumer<ComponentChild.Builder>... children) {
            children(Stream.of(children).map(c -> ComponentChild.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Map<String, ComponentDataConfiguration.Builder> getCollectionProperties() {
            Map<String, ComponentDataConfiguration.Builder> result = ComponentCollectionPropertiesCopier
                    .copyToBuilder(this.collectionProperties);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setCollectionProperties(Map<String, ComponentDataConfiguration.BuilderImpl> collectionProperties) {
            this.collectionProperties = ComponentCollectionPropertiesCopier.copyFromBuilder(collectionProperties);
        }

        @Override
        public final Builder collectionProperties(Map<String, ComponentDataConfiguration> collectionProperties) {
            this.collectionProperties = ComponentCollectionPropertiesCopier.copy(collectionProperties);
            return this;
        }

        public final String getComponentType() {
            return componentType;
        }

        public final void setComponentType(String componentType) {
            this.componentType = componentType;
        }

        @Override
        public final Builder componentType(String componentType) {
            this.componentType = componentType;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getEnvironmentName() {
            return environmentName;
        }

        public final void setEnvironmentName(String environmentName) {
            this.environmentName = environmentName;
        }

        @Override
        public final Builder environmentName(String environmentName) {
            this.environmentName = environmentName;
            return this;
        }

        public final Map<String, ComponentEvent.Builder> getEvents() {
            Map<String, ComponentEvent.Builder> result = ComponentEventsCopier.copyToBuilder(this.events);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setEvents(Map<String, ComponentEvent.BuilderImpl> events) {
            this.events = ComponentEventsCopier.copyFromBuilder(events);
        }

        @Override
        public final Builder events(Map<String, ComponentEvent> events) {
            this.events = ComponentEventsCopier.copy(events);
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, ? extends Map<String, String>> getOverrides() {
            if (overrides instanceof SdkAutoConstructMap) {
                return null;
            }
            return overrides;
        }

        public final void setOverrides(Map<String, ? extends Map<String, String>> overrides) {
            this.overrides = ComponentOverridesCopier.copy(overrides);
        }

        @Override
        public final Builder overrides(Map<String, ? extends Map<String, String>> overrides) {
            this.overrides = ComponentOverridesCopier.copy(overrides);
            return this;
        }

        public final Map<String, ComponentProperty.Builder> getProperties() {
            Map<String, ComponentProperty.Builder> result = ComponentPropertiesCopier.copyToBuilder(this.properties);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setProperties(Map<String, ComponentProperty.BuilderImpl> properties) {
            this.properties = ComponentPropertiesCopier.copyFromBuilder(properties);
        }

        @Override
        public final Builder properties(Map<String, ComponentProperty> properties) {
            this.properties = ComponentPropertiesCopier.copy(properties);
            return this;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final String getSourceId() {
            return sourceId;
        }

        public final void setSourceId(String sourceId) {
            this.sourceId = sourceId;
        }

        @Override
        public final Builder sourceId(String sourceId) {
            this.sourceId = sourceId;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final List<ComponentVariant.Builder> getVariants() {
            List<ComponentVariant.Builder> result = ComponentVariantsCopier.copyToBuilder(this.variants);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariants(Collection<ComponentVariant.BuilderImpl> variants) {
            this.variants = ComponentVariantsCopier.copyFromBuilder(variants);
        }

        @Override
        public final Builder variants(Collection<ComponentVariant> variants) {
            this.variants = ComponentVariantsCopier.copy(variants);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variants(ComponentVariant... variants) {
            variants(Arrays.asList(variants));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variants(Consumer<ComponentVariant.Builder>... variants) {
            variants(Stream.of(variants).map(c -> ComponentVariant.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Component build() {
            return new Component(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
