/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplifyuibuilder;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplifyuibuilder.internal.AmplifyUiBuilderServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.amplifyuibuilder.model.AmplifyUiBuilderException;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateFormRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateFormResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.CreateThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteFormRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteFormResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.DeleteThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExchangeCodeForTokenRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExchangeCodeForTokenResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportComponentsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportComponentsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportFormsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportFormsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportThemesRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ExportThemesResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetCodegenJobRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetCodegenJobResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetFormRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetFormResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetMetadataRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetMetadataResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.GetThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.InternalServerException;
import software.amazon.awssdk.services.amplifyuibuilder.model.InvalidParameterException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListCodegenJobsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListCodegenJobsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListComponentsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListFormsRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListFormsResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.ListThemesResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.PutMetadataFlagRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.PutMetadataFlagResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.RefreshTokenRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.RefreshTokenResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ResourceConflictException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ResourceNotFoundException;
import software.amazon.awssdk.services.amplifyuibuilder.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.amplifyuibuilder.model.StartCodegenJobRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.StartCodegenJobResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.ThrottlingException;
import software.amazon.awssdk.services.amplifyuibuilder.model.UnauthorizedException;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateComponentRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateComponentResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateFormRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateFormResponse;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateThemeRequest;
import software.amazon.awssdk.services.amplifyuibuilder.model.UpdateThemeResponse;
import software.amazon.awssdk.services.amplifyuibuilder.transform.CreateComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.CreateFormRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.CreateThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.DeleteComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.DeleteFormRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.DeleteThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExchangeCodeForTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExportComponentsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExportFormsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ExportThemesRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetCodegenJobRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetFormRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetMetadataRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.GetThemeRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListCodegenJobsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListFormsRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.ListThemesRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.PutMetadataFlagRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.RefreshTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.StartCodegenJobRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.UpdateComponentRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.UpdateFormRequestMarshaller;
import software.amazon.awssdk.services.amplifyuibuilder.transform.UpdateThemeRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmplifyUiBuilderAsyncClient}.
 *
 * @see AmplifyUiBuilderAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyUiBuilderAsyncClient implements AmplifyUiBuilderAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmplifyUiBuilderAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAmplifyUiBuilderAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new component for an Amplify app.
     * </p>
     *
     * @param createComponentRequest
     * @return A Java Future containing the result of the CreateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.CreateComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/CreateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComponentResponse> createComponent(CreateComponentRequest createComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createComponentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComponentRequest, CreateComponentResponse>()
                            .withOperationName("CreateComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createComponentRequest));
            CompletableFuture<CreateComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new form for an Amplify app.
     * </p>
     *
     * @param createFormRequest
     * @return A Java Future containing the result of the CreateForm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.CreateForm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/CreateForm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFormResponse> createForm(CreateFormRequest createFormRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFormRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFormRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateForm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFormResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFormResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFormRequest, CreateFormResponse>().withOperationName("CreateForm")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateFormRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createFormRequest));
            CompletableFuture<CreateFormResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a theme to apply to the components in an Amplify app.
     * </p>
     *
     * @param createThemeRequest
     * @return A Java Future containing the result of the CreateTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You exceeded your service quota. Service quotas, also referred to as
     *         limits, are the maximum number of service resources or operations for your Amazon Web Services account.</li>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.CreateTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/CreateTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateThemeResponse> createTheme(CreateThemeRequest createThemeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createThemeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateThemeRequest, CreateThemeResponse>()
                            .withOperationName("CreateTheme").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createThemeRequest));
            CompletableFuture<CreateThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a component from an Amplify app.
     * </p>
     *
     * @param deleteComponentRequest
     * @return A Java Future containing the result of the DeleteComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.DeleteComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/DeleteComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComponentResponse> deleteComponent(DeleteComponentRequest deleteComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteComponentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComponentRequest, DeleteComponentResponse>()
                            .withOperationName("DeleteComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteComponentRequest));
            CompletableFuture<DeleteComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a form from an Amplify app.
     * </p>
     *
     * @param deleteFormRequest
     * @return A Java Future containing the result of the DeleteForm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.DeleteForm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/DeleteForm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFormResponse> deleteForm(DeleteFormRequest deleteFormRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFormRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFormRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteForm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFormResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFormResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFormRequest, DeleteFormResponse>().withOperationName("DeleteForm")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteFormRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteFormRequest));
            CompletableFuture<DeleteFormResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a theme from an Amplify app.
     * </p>
     *
     * @param deleteThemeRequest
     * @return A Java Future containing the result of the DeleteTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.DeleteTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/DeleteTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteThemeResponse> deleteTheme(DeleteThemeRequest deleteThemeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteThemeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteThemeRequest, DeleteThemeResponse>()
                            .withOperationName("DeleteTheme").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteThemeRequest));
            CompletableFuture<DeleteThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This is for internal use.
     * </p>
     * </note>
     * <p>
     * Amplify uses this action to exchange an access code for a token.
     * </p>
     *
     * @param exchangeCodeForTokenRequest
     * @return A Java Future containing the result of the ExchangeCodeForToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExchangeCodeForToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExchangeCodeForToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExchangeCodeForTokenResponse> exchangeCodeForToken(
            ExchangeCodeForTokenRequest exchangeCodeForTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exchangeCodeForTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exchangeCodeForTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExchangeCodeForToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExchangeCodeForTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExchangeCodeForTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExchangeCodeForTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExchangeCodeForTokenRequest, ExchangeCodeForTokenResponse>()
                            .withOperationName("ExchangeCodeForToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExchangeCodeForTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exchangeCodeForTokenRequest));
            CompletableFuture<ExchangeCodeForTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports component configurations to code that is ready to integrate into an Amplify app.
     * </p>
     *
     * @param exportComponentsRequest
     * @return A Java Future containing the result of the ExportComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExportComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExportComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportComponentsResponse> exportComponents(ExportComponentsRequest exportComponentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportComponentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportComponentsRequest, ExportComponentsResponse>()
                            .withOperationName("ExportComponents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportComponentsRequest));
            CompletableFuture<ExportComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports form configurations to code that is ready to integrate into an Amplify app.
     * </p>
     *
     * @param exportFormsRequest
     * @return A Java Future containing the result of the ExportForms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExportForms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExportForms" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportFormsResponse> exportForms(ExportFormsRequest exportFormsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportFormsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportFormsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportForms");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportFormsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportFormsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportFormsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportFormsRequest, ExportFormsResponse>()
                            .withOperationName("ExportForms").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportFormsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportFormsRequest));
            CompletableFuture<ExportFormsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports theme configurations to code that is ready to integrate into an Amplify app.
     * </p>
     *
     * @param exportThemesRequest
     * @return A Java Future containing the result of the ExportThemes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ExportThemes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ExportThemes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportThemesResponse> exportThemes(ExportThemesRequest exportThemesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportThemesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportThemesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportThemes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportThemesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ExportThemesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportThemesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportThemesRequest, ExportThemesResponse>()
                            .withOperationName("ExportThemes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportThemesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportThemesRequest));
            CompletableFuture<ExportThemesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing code generation job.
     * </p>
     *
     * @param getCodegenJobRequest
     * @return A Java Future containing the result of the GetCodegenJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetCodegenJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetCodegenJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCodegenJobResponse> getCodegenJob(GetCodegenJobRequest getCodegenJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCodegenJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCodegenJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCodegenJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCodegenJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCodegenJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCodegenJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCodegenJobRequest, GetCodegenJobResponse>()
                            .withOperationName("GetCodegenJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCodegenJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCodegenJobRequest));
            CompletableFuture<GetCodegenJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing component for an Amplify app.
     * </p>
     *
     * @param getComponentRequest
     * @return A Java Future containing the result of the GetComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetComponent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentResponse> getComponent(GetComponentRequest getComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getComponentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                            .withOperationName("GetComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getComponentRequest));
            CompletableFuture<GetComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing form for an Amplify app.
     * </p>
     *
     * @param getFormRequest
     * @return A Java Future containing the result of the GetForm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetForm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetForm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFormResponse> getForm(GetFormRequest getFormRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFormRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFormRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetForm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFormResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFormResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFormRequest, GetFormResponse>().withOperationName("GetForm")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetFormRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getFormRequest));
            CompletableFuture<GetFormResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns existing metadata for an Amplify app.
     * </p>
     *
     * @param getMetadataRequest
     * @return A Java Future containing the result of the GetMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You don't have permission to perform this operation.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetMetadata" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMetadataResponse> getMetadata(GetMetadataRequest getMetadataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetadataRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetadata");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMetadataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMetadataRequest, GetMetadataResponse>()
                            .withOperationName("GetMetadata").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMetadataRequest));
            CompletableFuture<GetMetadataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an existing theme for an Amplify app.
     * </p>
     *
     * @param getThemeRequest
     * @return A Java Future containing the result of the GetTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.GetTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/GetTheme" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetThemeResponse> getTheme(GetThemeRequest getThemeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getThemeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThemeRequest, GetThemeResponse>().withOperationName("GetTheme")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetThemeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getThemeRequest));
            CompletableFuture<GetThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of code generation jobs for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listCodegenJobsRequest
     * @return A Java Future containing the result of the ListCodegenJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListCodegenJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListCodegenJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCodegenJobsResponse> listCodegenJobs(ListCodegenJobsRequest listCodegenJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCodegenJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCodegenJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCodegenJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCodegenJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCodegenJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCodegenJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCodegenJobsRequest, ListCodegenJobsResponse>()
                            .withOperationName("ListCodegenJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCodegenJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCodegenJobsRequest));
            CompletableFuture<ListCodegenJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of components for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listComponentsRequest
     * @return A Java Future containing the result of the ListComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentsResponse> listComponents(ListComponentsRequest listComponentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listComponentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                            .withOperationName("ListComponents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listComponentsRequest));
            CompletableFuture<ListComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of forms for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listFormsRequest
     * @return A Java Future containing the result of the ListForms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListForms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListForms" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFormsResponse> listForms(ListFormsRequest listFormsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFormsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFormsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListForms");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFormsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFormsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFormsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFormsRequest, ListFormsResponse>().withOperationName("ListForms")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListFormsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listFormsRequest));
            CompletableFuture<ListFormsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of themes for a specified Amplify app and backend environment.
     * </p>
     *
     * @param listThemesRequest
     * @return A Java Future containing the result of the ListThemes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.ListThemes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/ListThemes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListThemesResponse> listThemes(ListThemesRequest listThemesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listThemesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listThemesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListThemes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListThemesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListThemesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListThemesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListThemesRequest, ListThemesResponse>().withOperationName("ListThemes")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListThemesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listThemesRequest));
            CompletableFuture<ListThemesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stores the metadata information about a feature on a form.
     * </p>
     *
     * @param putMetadataFlagRequest
     * @return A Java Future containing the result of the PutMetadataFlag operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedException You don't have permission to perform this operation.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.PutMetadataFlag
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/PutMetadataFlag"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutMetadataFlagResponse> putMetadataFlag(PutMetadataFlagRequest putMetadataFlagRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetadataFlagRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetadataFlagRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetadataFlag");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutMetadataFlagResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutMetadataFlagResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutMetadataFlagResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutMetadataFlagRequest, PutMetadataFlagResponse>()
                            .withOperationName("PutMetadataFlag").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutMetadataFlagRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putMetadataFlagRequest));
            CompletableFuture<PutMetadataFlagResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This is for internal use.
     * </p>
     * </note>
     * <p>
     * Amplify uses this action to refresh a previously issued access token that might have expired.
     * </p>
     *
     * @param refreshTokenRequest
     * @return A Java Future containing the result of the RefreshToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.RefreshToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/RefreshToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RefreshTokenResponse> refreshToken(RefreshTokenRequest refreshTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(refreshTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, refreshTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RefreshToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RefreshTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RefreshTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RefreshTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RefreshTokenRequest, RefreshTokenResponse>()
                            .withOperationName("RefreshToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RefreshTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(refreshTokenRequest));
            CompletableFuture<RefreshTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a code generation job for a specified Amplify app and backend environment.
     * </p>
     *
     * @param startCodegenJobRequest
     * @return A Java Future containing the result of the StartCodegenJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.StartCodegenJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/StartCodegenJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCodegenJobResponse> startCodegenJob(StartCodegenJobRequest startCodegenJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCodegenJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCodegenJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCodegenJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCodegenJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCodegenJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartCodegenJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCodegenJobRequest, StartCodegenJobResponse>()
                            .withOperationName("StartCodegenJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCodegenJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCodegenJobRequest));
            CompletableFuture<StartCodegenJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing component.
     * </p>
     *
     * @param updateComponentRequest
     * @return A Java Future containing the result of the UpdateComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.UpdateComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/UpdateComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComponentResponse> updateComponent(UpdateComponentRequest updateComponentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateComponentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComponentRequest, UpdateComponentResponse>()
                            .withOperationName("UpdateComponent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateComponentRequest));
            CompletableFuture<UpdateComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing form.
     * </p>
     *
     * @param updateFormRequest
     * @return A Java Future containing the result of the UpdateForm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.UpdateForm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/UpdateForm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFormResponse> updateForm(UpdateFormRequest updateFormRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFormRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFormRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateForm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFormResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFormResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFormResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFormRequest, UpdateFormResponse>().withOperationName("UpdateForm")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateFormRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateFormRequest));
            CompletableFuture<UpdateFormResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing theme.
     * </p>
     *
     * @param updateThemeRequest
     * @return A Java Future containing the result of the UpdateTheme operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceConflictException The resource specified in the request conflicts with an existing resource.</li>
     *         <li>InternalServerException An internal error has occurred. Please retry your request.</li>
     *         <li>InvalidParameterException An invalid or out-of-range value was supplied for the input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyUiBuilderException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyUiBuilderAsyncClient.UpdateTheme
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifyuibuilder-2021-08-11/UpdateTheme" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateThemeResponse> updateTheme(UpdateThemeRequest updateThemeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateThemeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateThemeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyUIBuilder");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTheme");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateThemeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateThemeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateThemeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateThemeRequest, UpdateThemeResponse>()
                            .withOperationName("UpdateTheme").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateThemeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateThemeRequest));
            CompletableFuture<UpdateThemeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AmplifyUiBuilderServiceClientConfiguration serviceClientConfiguration() {
        return new AmplifyUiBuilderServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyUiBuilderException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AmplifyUiBuilderServiceClientConfigurationBuilder serviceConfigBuilder = new AmplifyUiBuilderServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
