/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.apigateway.transform.ApiStageMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * API stage name of the associated API stage in a usage plan.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApiStage implements StructuredPojo, ToCopyableBuilder<ApiStage.Builder, ApiStage> {
    private final String apiId;

    private final String stage;

    private final Map<String, ThrottleSettings> throttle;

    private ApiStage(BuilderImpl builder) {
        this.apiId = builder.apiId;
        this.stage = builder.stage;
        this.throttle = builder.throttle;
    }

    /**
     * <p>
     * API Id of the associated API stage in a usage plan.
     * </p>
     * 
     * @return API Id of the associated API stage in a usage plan.
     */
    public String apiId() {
        return apiId;
    }

    /**
     * <p>
     * API stage name of the associated API stage in a usage plan.
     * </p>
     * 
     * @return API stage name of the associated API stage in a usage plan.
     */
    public String stage() {
        return stage;
    }

    /**
     * <p>
     * Map containing method level throttling information for API stage in a usage plan.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Map containing method level throttling information for API stage in a usage plan.
     */
    public Map<String, ThrottleSettings> throttle() {
        return throttle;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(apiId());
        hashCode = 31 * hashCode + Objects.hashCode(stage());
        hashCode = 31 * hashCode + Objects.hashCode(throttle());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApiStage)) {
            return false;
        }
        ApiStage other = (ApiStage) obj;
        return Objects.equals(apiId(), other.apiId()) && Objects.equals(stage(), other.stage())
                && Objects.equals(throttle(), other.throttle());
    }

    @Override
    public String toString() {
        return ToString.builder("ApiStage").add("ApiId", apiId()).add("Stage", stage()).add("Throttle", throttle()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "apiId":
            return Optional.ofNullable(clazz.cast(apiId()));
        case "stage":
            return Optional.ofNullable(clazz.cast(stage()));
        case "throttle":
            return Optional.ofNullable(clazz.cast(throttle()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ApiStageMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ApiStage> {
        /**
         * <p>
         * API Id of the associated API stage in a usage plan.
         * </p>
         * 
         * @param apiId
         *        API Id of the associated API stage in a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiId(String apiId);

        /**
         * <p>
         * API stage name of the associated API stage in a usage plan.
         * </p>
         * 
         * @param stage
         *        API stage name of the associated API stage in a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stage(String stage);

        /**
         * <p>
         * Map containing method level throttling information for API stage in a usage plan.
         * </p>
         * 
         * @param throttle
         *        Map containing method level throttling information for API stage in a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throttle(Map<String, ThrottleSettings> throttle);
    }

    static final class BuilderImpl implements Builder {
        private String apiId;

        private String stage;

        private Map<String, ThrottleSettings> throttle = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ApiStage model) {
            apiId(model.apiId);
            stage(model.stage);
            throttle(model.throttle);
        }

        public final String getApiId() {
            return apiId;
        }

        @Override
        public final Builder apiId(String apiId) {
            this.apiId = apiId;
            return this;
        }

        public final void setApiId(String apiId) {
            this.apiId = apiId;
        }

        public final String getStage() {
            return stage;
        }

        @Override
        public final Builder stage(String stage) {
            this.stage = stage;
            return this;
        }

        public final void setStage(String stage) {
            this.stage = stage;
        }

        public final Map<String, ThrottleSettings.Builder> getThrottle() {
            return throttle != null ? CollectionUtils.mapValues(throttle, ThrottleSettings::toBuilder) : null;
        }

        @Override
        public final Builder throttle(Map<String, ThrottleSettings> throttle) {
            this.throttle = MapOfApiStageThrottleSettingsCopier.copy(throttle);
            return this;
        }

        public final void setThrottle(Map<String, ThrottleSettings.BuilderImpl> throttle) {
            this.throttle = MapOfApiStageThrottleSettingsCopier.copyFromBuilder(throttle);
        }

        @Override
        public ApiStage build() {
            return new ApiStage(this);
        }
    }
}
