/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.apigateway.transform.ClientCertificateMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a client certificate used to configure client-side SSL authentication while sending requests to the
 * integration endpoint.
 * </p>
 * <div class="remarks">Client certificates are used to authenticate an API by the backend server. To authenticate an
 * API client (or user), use IAM roles and policies, a custom <a>Authorizer</a> or an Amazon Cognito user pool.</div>
 * <div class="seeAlso"> <a href=
 * "http://docs.aws.amazon.com/apigateway/latest/developerguide/getting-started-client-side-ssl-authentication.html">Use
 * Client-Side Certificate</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClientCertificate implements StructuredPojo, ToCopyableBuilder<ClientCertificate.Builder, ClientCertificate> {
    private final String clientCertificateId;

    private final String description;

    private final String pemEncodedCertificate;

    private final Instant createdDate;

    private final Instant expirationDate;

    private ClientCertificate(BuilderImpl builder) {
        this.clientCertificateId = builder.clientCertificateId;
        this.description = builder.description;
        this.pemEncodedCertificate = builder.pemEncodedCertificate;
        this.createdDate = builder.createdDate;
        this.expirationDate = builder.expirationDate;
    }

    /**
     * <p>
     * The identifier of the client certificate.
     * </p>
     * 
     * @return The identifier of the client certificate.
     */
    public String clientCertificateId() {
        return clientCertificateId;
    }

    /**
     * <p>
     * The description of the client certificate.
     * </p>
     * 
     * @return The description of the client certificate.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The PEM-encoded public key of the client certificate, which can be used to configure certificate authentication
     * in the integration endpoint .
     * </p>
     * 
     * @return The PEM-encoded public key of the client certificate, which can be used to configure certificate
     *         authentication in the integration endpoint .
     */
    public String pemEncodedCertificate() {
        return pemEncodedCertificate;
    }

    /**
     * <p>
     * The timestamp when the client certificate was created.
     * </p>
     * 
     * @return The timestamp when the client certificate was created.
     */
    public Instant createdDate() {
        return createdDate;
    }

    /**
     * <p>
     * The timestamp when the client certificate will expire.
     * </p>
     * 
     * @return The timestamp when the client certificate will expire.
     */
    public Instant expirationDate() {
        return expirationDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clientCertificateId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(pemEncodedCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(createdDate());
        hashCode = 31 * hashCode + Objects.hashCode(expirationDate());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClientCertificate)) {
            return false;
        }
        ClientCertificate other = (ClientCertificate) obj;
        return Objects.equals(clientCertificateId(), other.clientCertificateId())
                && Objects.equals(description(), other.description())
                && Objects.equals(pemEncodedCertificate(), other.pemEncodedCertificate())
                && Objects.equals(createdDate(), other.createdDate()) && Objects.equals(expirationDate(), other.expirationDate());
    }

    @Override
    public String toString() {
        return ToString.builder("ClientCertificate").add("ClientCertificateId", clientCertificateId())
                .add("Description", description()).add("PemEncodedCertificate", pemEncodedCertificate())
                .add("CreatedDate", createdDate()).add("ExpirationDate", expirationDate()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "clientCertificateId":
            return Optional.ofNullable(clazz.cast(clientCertificateId()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "pemEncodedCertificate":
            return Optional.ofNullable(clazz.cast(pemEncodedCertificate()));
        case "createdDate":
            return Optional.ofNullable(clazz.cast(createdDate()));
        case "expirationDate":
            return Optional.ofNullable(clazz.cast(expirationDate()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ClientCertificateMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ClientCertificate> {
        /**
         * <p>
         * The identifier of the client certificate.
         * </p>
         * 
         * @param clientCertificateId
         *        The identifier of the client certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientCertificateId(String clientCertificateId);

        /**
         * <p>
         * The description of the client certificate.
         * </p>
         * 
         * @param description
         *        The description of the client certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The PEM-encoded public key of the client certificate, which can be used to configure certificate
         * authentication in the integration endpoint .
         * </p>
         * 
         * @param pemEncodedCertificate
         *        The PEM-encoded public key of the client certificate, which can be used to configure certificate
         *        authentication in the integration endpoint .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pemEncodedCertificate(String pemEncodedCertificate);

        /**
         * <p>
         * The timestamp when the client certificate was created.
         * </p>
         * 
         * @param createdDate
         *        The timestamp when the client certificate was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdDate(Instant createdDate);

        /**
         * <p>
         * The timestamp when the client certificate will expire.
         * </p>
         * 
         * @param expirationDate
         *        The timestamp when the client certificate will expire.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expirationDate(Instant expirationDate);
    }

    static final class BuilderImpl implements Builder {
        private String clientCertificateId;

        private String description;

        private String pemEncodedCertificate;

        private Instant createdDate;

        private Instant expirationDate;

        private BuilderImpl() {
        }

        private BuilderImpl(ClientCertificate model) {
            clientCertificateId(model.clientCertificateId);
            description(model.description);
            pemEncodedCertificate(model.pemEncodedCertificate);
            createdDate(model.createdDate);
            expirationDate(model.expirationDate);
        }

        public final String getClientCertificateId() {
            return clientCertificateId;
        }

        @Override
        public final Builder clientCertificateId(String clientCertificateId) {
            this.clientCertificateId = clientCertificateId;
            return this;
        }

        public final void setClientCertificateId(String clientCertificateId) {
            this.clientCertificateId = clientCertificateId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getPemEncodedCertificate() {
            return pemEncodedCertificate;
        }

        @Override
        public final Builder pemEncodedCertificate(String pemEncodedCertificate) {
            this.pemEncodedCertificate = pemEncodedCertificate;
            return this;
        }

        public final void setPemEncodedCertificate(String pemEncodedCertificate) {
            this.pemEncodedCertificate = pemEncodedCertificate;
        }

        public final Instant getCreatedDate() {
            return createdDate;
        }

        @Override
        public final Builder createdDate(Instant createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public final void setCreatedDate(Instant createdDate) {
            this.createdDate = createdDate;
        }

        public final Instant getExpirationDate() {
            return expirationDate;
        }

        @Override
        public final Builder expirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
            return this;
        }

        public final void setExpirationDate(Instant expirationDate) {
            this.expirationDate = expirationDate;
        }

        @Override
        public ClientCertificate build() {
            return new ClientCertificate(this);
        }
    }
}
