/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.apigateway.transform.EndpointConfigurationMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The endpoint configuration to indicate the types of endpoints an API (<a>RestApi</a>) or its custom domain name
 * (<a>DomainName</a>) has.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EndpointConfiguration implements StructuredPojo,
        ToCopyableBuilder<EndpointConfiguration.Builder, EndpointConfiguration> {
    private final List<String> types;

    private EndpointConfiguration(BuilderImpl builder) {
        this.types = builder.types;
    }

    /**
     * <p>
     * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
     * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API and
     * its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type is
     * <code>PRIVATE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
     *         edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional
     *         API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the
     *         endpoint type is <code>PRIVATE</code>.
     */
    public List<EndpointType> types() {
        return TypeConverter.convert(types, EndpointType::fromValue);
    }

    /**
     * <p>
     * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
     * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API and
     * its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type is
     * <code>PRIVATE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
     *         edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional
     *         API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the
     *         endpoint type is <code>PRIVATE</code>.
     */
    public List<String> typesAsStrings() {
        return types;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EndpointConfiguration)) {
            return false;
        }
        EndpointConfiguration other = (EndpointConfiguration) obj;
        return Objects.equals(typesAsStrings(), other.typesAsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("EndpointConfiguration").add("Types", typesAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "types":
            return Optional.ofNullable(clazz.cast(typesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        EndpointConfigurationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, EndpointConfiguration> {
        /**
         * <p>
         * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
         * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API
         * and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type
         * is <code>PRIVATE</code>.
         * </p>
         * 
         * @param types
         *        A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For
         *        an edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a
         *        regional API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private
         *        API, the endpoint type is <code>PRIVATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typesWithStrings(Collection<String> types);

        /**
         * <p>
         * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
         * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API
         * and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type
         * is <code>PRIVATE</code>.
         * </p>
         * 
         * @param types
         *        A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For
         *        an edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a
         *        regional API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private
         *        API, the endpoint type is <code>PRIVATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typesWithStrings(String... types);

        /**
         * <p>
         * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
         * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API
         * and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type
         * is <code>PRIVATE</code>.
         * </p>
         * 
         * @param types
         *        A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For
         *        an edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a
         *        regional API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private
         *        API, the endpoint type is <code>PRIVATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder types(Collection<EndpointType> types);

        /**
         * <p>
         * A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For an
         * edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a regional API
         * and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private API, the endpoint type
         * is <code>PRIVATE</code>.
         * </p>
         * 
         * @param types
         *        A list of endpoint types of an API (<a>RestApi</a>) or its custom domain name (<a>DomainName</a>). For
         *        an edge-optimized API and its custom domain name, the endpoint type is <code>"EDGE"</code>. For a
         *        regional API and its custom domain name, the endpoint type is <code>REGIONAL</code>. For a private
         *        API, the endpoint type is <code>PRIVATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder types(EndpointType... types);
    }

    static final class BuilderImpl implements Builder {
        private List<String> types = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EndpointConfiguration model) {
            typesWithStrings(model.types);
        }

        public final Collection<String> getTypes() {
            return types;
        }

        @Override
        public final Builder typesWithStrings(Collection<String> types) {
            this.types = ListOfEndpointTypeCopier.copy(types);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder typesWithStrings(String... types) {
            typesWithStrings(Arrays.asList(types));
            return this;
        }

        @Override
        public final Builder types(Collection<EndpointType> types) {
            this.types = ListOfEndpointTypeCopier.copyEnumToString(types);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder types(EndpointType... types) {
            types(Arrays.asList(types));
            return this;
        }

        public final void setTypes(Collection<String> types) {
            this.types = ListOfEndpointTypeCopier.copy(types);
        }

        @Override
        public EndpointConfiguration build() {
            return new EndpointConfiguration(this);
        }
    }
}
