/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.nio.ByteBuffer;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A POST request to import an API to API Gateway using an input of an API definition file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportRestApiRequest extends ApiGatewayRequest implements
        ToCopyableBuilder<ImportRestApiRequest.Builder, ImportRestApiRequest> {
    private final Boolean failOnWarnings;

    private final Map<String, String> parameters;

    private final SdkBytes body;

    private ImportRestApiRequest(BuilderImpl builder) {
        super(builder);
        this.failOnWarnings = builder.failOnWarnings;
        this.parameters = builder.parameters;
        this.body = builder.body;
    }

    /**
     * <p>
     * A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (<code>false</code>
     * ) when a warning is encountered. The default value is <code>false</code>.
     * </p>
     * 
     * @return A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
     *         <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
     */
    public Boolean failOnWarnings() {
        return failOnWarnings;
    }

    /**
     * <p>
     * A key-value map of context-specific query string parameters specifying the behavior of different API importing
     * operations. The following shows operation-specific parameters and their supported values.
     * </p>
     * <p>
     * To exclude <a>DocumentationParts</a> from the import, set <code>parameters</code> as
     * <code>ignore=documentation</code>.
     * </p>
     * <p>
     * To configure the endpoint type, set <code>parameters</code> as <code>endpointConfigurationTypes=EDGE</code>,
     * <code>endpointConfigurationTypes=REGIONAL</code>, or <code>endpointConfigurationTypes=PRIVATE</code>. The default
     * endpoint type is <code>EDGE</code>.
     * </p>
     * <p>
     * To handle imported <code>basePath</code>, set <code>parameters</code> as <code>basePath=ignore</code>,
     * <code>basePath=prepend</code> or <code>basePath=split</code>.
     * </p>
     * <p>
     * For example, the AWS CLI command to exclude documentation from the imported API is:
     * </p>
     * 
     * <pre>
     * <code>aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'</code>
     * </pre>
     * <p>
     * The AWS CLI command to set the regional endpoint on the imported API is:
     * </p>
     * 
     * <pre>
     * <code>aws apigateway import-rest-api --parameters endpointConfigurationTypes=REGIONAL --body 'file:///path/to/imported-api-body.json'</code>
     * </pre>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A key-value map of context-specific query string parameters specifying the behavior of different API
     *         importing operations. The following shows operation-specific parameters and their supported values.</p>
     *         <p>
     *         To exclude <a>DocumentationParts</a> from the import, set <code>parameters</code> as
     *         <code>ignore=documentation</code>.
     *         </p>
     *         <p>
     *         To configure the endpoint type, set <code>parameters</code> as
     *         <code>endpointConfigurationTypes=EDGE</code>, <code>endpointConfigurationTypes=REGIONAL</code>, or
     *         <code>endpointConfigurationTypes=PRIVATE</code>. The default endpoint type is <code>EDGE</code>.
     *         </p>
     *         <p>
     *         To handle imported <code>basePath</code>, set <code>parameters</code> as <code>basePath=ignore</code>,
     *         <code>basePath=prepend</code> or <code>basePath=split</code>.
     *         </p>
     *         <p>
     *         For example, the AWS CLI command to exclude documentation from the imported API is:
     *         </p>
     * 
     *         <pre>
     * <code>aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'</code>
     * </pre>
     *         <p>
     *         The AWS CLI command to set the regional endpoint on the imported API is:
     *         </p>
     * 
     * <pre><code>aws apigateway import-rest-api --parameters endpointConfigurationTypes=REGIONAL --body 'file:///path/to/imported-api-body.json'</code>
     */
    public Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * [Required] The POST request body containing external API definitions. Currently, only Swagger definition JSON
     * files are supported. The maximum size of the API definition file is 2MB.
     * </p>
     * 
     * @return [Required] The POST request body containing external API definitions. Currently, only Swagger definition
     *         JSON files are supported. The maximum size of the API definition file is 2MB.
     */
    public SdkBytes body() {
        return body;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(failOnWarnings());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(body());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportRestApiRequest)) {
            return false;
        }
        ImportRestApiRequest other = (ImportRestApiRequest) obj;
        return Objects.equals(failOnWarnings(), other.failOnWarnings()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(body(), other.body());
    }

    @Override
    public String toString() {
        return ToString.builder("ImportRestApiRequest").add("FailOnWarnings", failOnWarnings()).add("Parameters", parameters())
                .add("Body", body()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "failOnWarnings":
            return Optional.ofNullable(clazz.cast(failOnWarnings()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "body":
            return Optional.ofNullable(clazz.cast(body()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends ApiGatewayRequest.Builder, CopyableBuilder<Builder, ImportRestApiRequest> {
        /**
         * <p>
         * A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
         * <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
         * </p>
         * 
         * @param failOnWarnings
         *        A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
         *        <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failOnWarnings(Boolean failOnWarnings);

        /**
         * <p>
         * A key-value map of context-specific query string parameters specifying the behavior of different API
         * importing operations. The following shows operation-specific parameters and their supported values.
         * </p>
         * <p>
         * To exclude <a>DocumentationParts</a> from the import, set <code>parameters</code> as
         * <code>ignore=documentation</code>.
         * </p>
         * <p>
         * To configure the endpoint type, set <code>parameters</code> as <code>endpointConfigurationTypes=EDGE</code>,
         * <code>endpointConfigurationTypes=REGIONAL</code>, or <code>endpointConfigurationTypes=PRIVATE</code>. The
         * default endpoint type is <code>EDGE</code>.
         * </p>
         * <p>
         * To handle imported <code>basePath</code>, set <code>parameters</code> as <code>basePath=ignore</code>,
         * <code>basePath=prepend</code> or <code>basePath=split</code>.
         * </p>
         * <p>
         * For example, the AWS CLI command to exclude documentation from the imported API is:
         * </p>
         * 
         * <pre>
         * <code>aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'</code>
         * </pre>
         * <p>
         * The AWS CLI command to set the regional endpoint on the imported API is:
         * </p>
         * 
         * <pre>
         * <code>aws apigateway import-rest-api --parameters endpointConfigurationTypes=REGIONAL --body 'file:///path/to/imported-api-body.json'</code>
         * </pre>
         * 
         * @param parameters
         *        A key-value map of context-specific query string parameters specifying the behavior of different API
         *        importing operations. The following shows operation-specific parameters and their supported
         *        values.</p>
         *        <p>
         *        To exclude <a>DocumentationParts</a> from the import, set <code>parameters</code> as
         *        <code>ignore=documentation</code>.
         *        </p>
         *        <p>
         *        To configure the endpoint type, set <code>parameters</code> as
         *        <code>endpointConfigurationTypes=EDGE</code>, <code>endpointConfigurationTypes=REGIONAL</code>, or
         *        <code>endpointConfigurationTypes=PRIVATE</code>. The default endpoint type is <code>EDGE</code>.
         *        </p>
         *        <p>
         *        To handle imported <code>basePath</code>, set <code>parameters</code> as <code>basePath=ignore</code>,
         *        <code>basePath=prepend</code> or <code>basePath=split</code>.
         *        </p>
         *        <p>
         *        For example, the AWS CLI command to exclude documentation from the imported API is:
         *        </p>
         * 
         *        <pre>
         * <code>aws apigateway import-rest-api --parameters ignore=documentation --body 'file:///path/to/imported-api-body.json'</code>
         * </pre>
         *        <p>
         *        The AWS CLI command to set the regional endpoint on the imported API is:
         *        </p>
         * 
         *        <pre>
         * <code>aws apigateway import-rest-api --parameters endpointConfigurationTypes=REGIONAL --body 'file:///path/to/imported-api-body.json'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * [Required] The POST request body containing external API definitions. Currently, only Swagger definition JSON
         * files are supported. The maximum size of the API definition file is 2MB.
         * </p>
         * 
         * @param body
         *        [Required] The POST request body containing external API definitions. Currently, only Swagger
         *        definition JSON files are supported. The maximum size of the API definition file is 2MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(SdkBytes body);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayRequest.BuilderImpl implements Builder {
        private Boolean failOnWarnings;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private SdkBytes body;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportRestApiRequest model) {
            super(model);
            failOnWarnings(model.failOnWarnings);
            parameters(model.parameters);
            body(model.body);
        }

        public final Boolean getFailOnWarnings() {
            return failOnWarnings;
        }

        @Override
        public final Builder failOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
            return this;
        }

        public final void setFailOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
        }

        public final Map<String, String> getParameters() {
            return parameters;
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = MapOfStringToStringCopier.copy(parameters);
            return this;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = MapOfStringToStringCopier.copy(parameters);
        }

        public final ByteBuffer getBody() {
            return body == null ? null : body.asByteBuffer();
        }

        @Override
        public final Builder body(SdkBytes body) {
            this.body = StandardMemberCopier.copy(body);
            return this;
        }

        public final void setBody(ByteBuffer body) {
            body(body == null ? null : SdkBytes.fromByteBuffer(body));
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ImportRestApiRequest build() {
            return new ImportRestApiRequest(this);
        }
    }
}
