/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.apigateway.transform.IntegrationMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an HTTP, HTTP_PROXY, AWS, AWS_PROXY, or Mock integration.
 * </p>
 * <div class="remarks">In the API Gateway console, the built-in Lambda integration is an AWS integration.</div> <div
 * class="seeAlso"> <a
 * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an API</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Integration implements StructuredPojo, ToCopyableBuilder<Integration.Builder, Integration> {
    private final String type;

    private final String httpMethod;

    private final String uri;

    private final String connectionType;

    private final String connectionId;

    private final String credentials;

    private final Map<String, String> requestParameters;

    private final Map<String, String> requestTemplates;

    private final String passthroughBehavior;

    private final String contentHandling;

    private final Integer timeoutInMillis;

    private final String cacheNamespace;

    private final List<String> cacheKeyParameters;

    private final Map<String, IntegrationResponse> integrationResponses;

    private Integration(BuilderImpl builder) {
        this.type = builder.type;
        this.httpMethod = builder.httpMethod;
        this.uri = builder.uri;
        this.connectionType = builder.connectionType;
        this.connectionId = builder.connectionId;
        this.credentials = builder.credentials;
        this.requestParameters = builder.requestParameters;
        this.requestTemplates = builder.requestTemplates;
        this.passthroughBehavior = builder.passthroughBehavior;
        this.contentHandling = builder.contentHandling;
        this.timeoutInMillis = builder.timeoutInMillis;
        this.cacheNamespace = builder.cacheNamespace;
        this.cacheKeyParameters = builder.cacheKeyParameters;
        this.integrationResponses = builder.integrationResponses;
    }

    /**
     * <p>
     * Specifies an API method integration type. The valid value is one of the following:
     * </p>
     * <ul>
     * <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the Lambda
     * function-invoking action. With the Lambda function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS integration.</li>
     * <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking action with
     * the client request passed through as-is. This integration is also referred to as the Lambda proxy integration.</li>
     * <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private HTTP
     * endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
     * <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a private
     * HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred to as the HTTP
     * proxy integration.</li>
     * <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint without
     * invoking any backend.</li>
     * </ul>
     * <p>
     * For the HTTP and HTTP proxy integrations, each integration can specify a protocol (<code>http/https</code>), port
     * and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP proxy
     * integration with a <code>connectionType</code> of <code>VPC_LINK</code> is referred to as a private integration
     * and uses a <a>VpcLink</a> to connect API Gateway to a network load balancer of a VPC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IntegrationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies an API method integration type. The valid value is one of the following:</p>
     *         <ul>
     *         <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the
     *         Lambda function-invoking action. With the Lambda function-invoking action, this is referred to as the
     *         Lambda custom integration. With any other AWS service action, this is known as AWS integration.</li>
     *         <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking
     *         action with the client request passed through as-is. This integration is also referred to as the Lambda
     *         proxy integration.</li>
     *         <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private
     *         HTTP endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
     *         <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
     *         private HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred
     *         to as the HTTP proxy integration.</li>
     *         <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint
     *         without invoking any backend.</li>
     *         </ul>
     *         <p>
     *         For the HTTP and HTTP proxy integrations, each integration can specify a protocol (
     *         <code>http/https</code>), port and path. Standard 80 and 443 ports are supported as well as custom ports
     *         above 1024. An HTTP or HTTP proxy integration with a <code>connectionType</code> of <code>VPC_LINK</code>
     *         is referred to as a private integration and uses a <a>VpcLink</a> to connect API Gateway to a network
     *         load balancer of a VPC.
     * @see IntegrationType
     */
    public IntegrationType type() {
        return IntegrationType.fromValue(type);
    }

    /**
     * <p>
     * Specifies an API method integration type. The valid value is one of the following:
     * </p>
     * <ul>
     * <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the Lambda
     * function-invoking action. With the Lambda function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS integration.</li>
     * <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking action with
     * the client request passed through as-is. This integration is also referred to as the Lambda proxy integration.</li>
     * <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private HTTP
     * endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
     * <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a private
     * HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred to as the HTTP
     * proxy integration.</li>
     * <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint without
     * invoking any backend.</li>
     * </ul>
     * <p>
     * For the HTTP and HTTP proxy integrations, each integration can specify a protocol (<code>http/https</code>), port
     * and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP proxy
     * integration with a <code>connectionType</code> of <code>VPC_LINK</code> is referred to as a private integration
     * and uses a <a>VpcLink</a> to connect API Gateway to a network load balancer of a VPC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IntegrationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies an API method integration type. The valid value is one of the following:</p>
     *         <ul>
     *         <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the
     *         Lambda function-invoking action. With the Lambda function-invoking action, this is referred to as the
     *         Lambda custom integration. With any other AWS service action, this is known as AWS integration.</li>
     *         <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking
     *         action with the client request passed through as-is. This integration is also referred to as the Lambda
     *         proxy integration.</li>
     *         <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private
     *         HTTP endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
     *         <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
     *         private HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred
     *         to as the HTTP proxy integration.</li>
     *         <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint
     *         without invoking any backend.</li>
     *         </ul>
     *         <p>
     *         For the HTTP and HTTP proxy integrations, each integration can specify a protocol (
     *         <code>http/https</code>), port and path. Standard 80 and 443 ports are supported as well as custom ports
     *         above 1024. An HTTP or HTTP proxy integration with a <code>connectionType</code> of <code>VPC_LINK</code>
     *         is referred to as a private integration and uses a <a>VpcLink</a> to connect API Gateway to a network
     *         load balancer of a VPC.
     * @see IntegrationType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Specifies the integration's HTTP method type.
     * </p>
     * 
     * @return Specifies the integration's HTTP method type.
     */
    public String httpMethod() {
        return httpMethod;
    }

    /**
     * <p>
     * Specifies Uniform Resource Identifier (URI) of the integration endpoint.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>HTTP</code> or <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded HTTP(S)
     * URL according to the <a target="_blank" href="https://en.wikipedia.org/wiki/Uniform_Resource_Identifier">RFC-3986
     * specification</a>, for either standard integration, where <code>connectionType</code> is not
     * <code>VPC_LINK</code>, or private integration, where <code>connectionType</code> is <code>VPC_LINK</code>. For a
     * private HTTP integration, the URI is not used for routing.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
     * <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}</code>. Here,
     * <code>{Region}</code> is the API Gateway region (e.g., <code>us-east-1</code>); <code>{service}</code> is the
     * name of the integrated AWS service (e.g., <code>s3</code>); and <code>{subdomain}</code> is a designated
     * subdomain supported by certain AWS service for fast host-name lookup. <code>action</code> can be used for an AWS
     * service action-based API, using an <code>Action={name}&amp;{p1}={v1}&amp;p2={v2}...</code> query string. The
     * ensuing <code>{service_api}</code> refers to a supported action <code>{name}</code> plus any required input
     * parameters. Alternatively, <code>path</code> can be used for an AWS service path-based API. The ensuing
     * <code>service_api</code> refers to the path to an AWS service resource, including the region of the integrated
     * AWS service, if applicable. For example, for integration with the S3 API of
     * <code><a href="http://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectGET.html">GetObject</a></code>, the
     * <code>uri</code> can be either
     * <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
     * <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies Uniform Resource Identifier (URI) of the integration endpoint.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>HTTP</code> or <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded
     *         HTTP(S) URL according to the <a target="_blank"
     *         href="https://en.wikipedia.org/wiki/Uniform_Resource_Identifier">RFC-3986 specification</a>, for either
     *         standard integration, where <code>connectionType</code> is not <code>VPC_LINK</code>, or private
     *         integration, where <code>connectionType</code> is <code>VPC_LINK</code>. For a private HTTP integration,
     *         the URI is not used for routing.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
     *         <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}</code>. Here,
     *         <code>{Region}</code> is the API Gateway region (e.g., <code>us-east-1</code>); <code>{service}</code> is
     *         the name of the integrated AWS service (e.g., <code>s3</code>); and <code>{subdomain}</code> is a
     *         designated subdomain supported by certain AWS service for fast host-name lookup. <code>action</code> can
     *         be used for an AWS service action-based API, using an
     *         <code>Action={name}&amp;{p1}={v1}&amp;p2={v2}...</code> query string. The ensuing
     *         <code>{service_api}</code> refers to a supported action <code>{name}</code> plus any required input
     *         parameters. Alternatively, <code>path</code> can be used for an AWS service path-based API. The ensuing
     *         <code>service_api</code> refers to the path to an AWS service resource, including the region of the
     *         integrated AWS service, if applicable. For example, for integration with the S3 API of
     *         <code><a href="http://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectGET.html">GetObject</a></code>,
     *         the <code>uri</code> can be either
     *         <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
     *         <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>
     *         </p>
     *         </li>
     */
    public String uri() {
        return uri;
    }

    /**
     * <p>
     * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
     * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
     * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code>
     *         for connections through the public routable internet or <code>VPC_LINK</code> for private connections
     *         between API Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * @see ConnectionType
     */
    public ConnectionType connectionType() {
        return ConnectionType.fromValue(connectionType);
    }

    /**
     * <p>
     * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
     * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
     * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code>
     *         for connections through the public routable internet or <code>VPC_LINK</code> for private connections
     *         between API Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * @see ConnectionType
     */
    public String connectionTypeAsString() {
        return connectionType;
    }

    /**
     * <p>
     * The (<a href="http://docs.aws.amazon.com/apigateway/api-reference/resource/vpc-link/#id"><code>id</code></a>) of
     * the <a>VpcLink</a> used for the integration when <code>connectionType=VPC_LINK</code> and undefined, otherwise.
     * </p>
     * 
     * @return The (<a href="http://docs.aws.amazon.com/apigateway/api-reference/resource/vpc-link/#id"><code>id</code>
     *         </a>) of the <a>VpcLink</a> used for the integration when <code>connectionType=VPC_LINK</code> and
     *         undefined, otherwise.
     */
    public String connectionId() {
        return connectionId;
    }

    /**
     * <p>
     * Specifies the credentials required for the integration, if any. For AWS integrations, three options are
     * available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To
     * require that the caller's identity be passed through from the request, specify the string
     * <code>arn:aws:iam::\*:user/\*</code>. To use resource-based permissions on supported AWS services, specify null.
     * </p>
     * 
     * @return Specifies the credentials required for the integration, if any. For AWS integrations, three options are
     *         available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name (ARN).
     *         To require that the caller's identity be passed through from the request, specify the string
     *         <code>arn:aws:iam::\*:user/\*</code>. To use resource-based permissions on supported AWS services,
     *         specify null.
     */
    public String credentials() {
        return credentials;
    }

    /**
     * <p>
     * A key-value map specifying request parameters that are passed from the method request to the back end. The key is
     * an integration request parameter name and the associated value is a method request parameter value or static
     * value that must be enclosed within single quotes and pre-encoded as required by the back end. The method request
     * parameter value must match the pattern of <code>method.request.{location}.{name}</code>, where
     * <code>location</code> is <code>querystring</code>, <code>path</code>, or <code>header</code> and
     * <code>name</code> must be a valid and unique method request parameter name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A key-value map specifying request parameters that are passed from the method request to the back end.
     *         The key is an integration request parameter name and the associated value is a method request parameter
     *         value or static value that must be enclosed within single quotes and pre-encoded as required by the back
     *         end. The method request parameter value must match the pattern of
     *         <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>,
     *         <code>path</code>, or <code>header</code> and <code>name</code> must be a valid and unique method request
     *         parameter name.
     */
    public Map<String, String> requestParameters() {
        return requestParameters;
    }

    /**
     * <p>
     * Represents a map of Velocity templates that are applied on the request payload based on the value of the
     * Content-Type header sent by the client. The content type value is the key in this map, and the template (as a
     * String) is the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Represents a map of Velocity templates that are applied on the request payload based on the value of the
     *         Content-Type header sent by the client. The content type value is the key in this map, and the template
     *         (as a String) is the value.
     */
    public Map<String, String> requestTemplates() {
        return requestTemplates;
    }

    /**
     * <div>
     * <p>
     * Specifies how the method request body of an unmapped content type will be passed through the integration request
     * to the back end without transformation. A content type is unmapped if no mapping template is defined in the
     * integration or the content type does not match any of the mapped content types, as specified in
     * <code>requestTemplates</code>. The valid value is one of the following:
     * </p>
     * <ul>
     * <li> <code>WHEN_NO_MATCH</code>: passes the method request body through the integration request to the back end
     * without transformation when the method request content type does not match any content type associated with the
     * mapping templates defined in the integration request.</li>
     * <li> <code>WHEN_NO_TEMPLATES</code>: passes the method request body through the integration request to the back
     * end without transformation when no mapping template is defined in the integration request. If a template is
     * defined when this option is selected, the method request of an unmapped content-type will be rejected with an
     * HTTP <code>415 Unsupported Media Type</code> response.</li>
     * <li> <code>NEVER</code>: rejects the method request with an HTTP <code>415 Unsupported Media Type</code> response
     * when either the method request content type does not match any content type associated with the mapping templates
     * defined in the integration request or no mapping template is defined in the integration request.</li>
     * </ul>
     * </div>
     * 
     * @return <p>
     *         Specifies how the method request body of an unmapped content type will be passed through the integration
     *         request to the back end without transformation. A content type is unmapped if no mapping template is
     *         defined in the integration or the content type does not match any of the mapped content types, as
     *         specified in <code>requestTemplates</code>. The valid value is one of the following:
     *         </p>
     *         <ul>
     *         <li> <code>WHEN_NO_MATCH</code>: passes the method request body through the integration request to the
     *         back end without transformation when the method request content type does not match any content type
     *         associated with the mapping templates defined in the integration request.</li>
     *         <li> <code>WHEN_NO_TEMPLATES</code>: passes the method request body through the integration request to the
     *         back end without transformation when no mapping template is defined in the integration request. If a
     *         template is defined when this option is selected, the method request of an unmapped content-type will be
     *         rejected with an HTTP <code>415 Unsupported Media Type</code> response.</li>
     *         <li> <code>NEVER</code>: rejects the method request with an HTTP <code>415 Unsupported Media Type</code>
     *         response when either the method request content type does not match any content type associated with the
     *         mapping templates defined in the integration request or no mapping template is defined in the integration
     *         request.</li>
     *         </ul>
     */
    public String passthroughBehavior() {
        return passthroughBehavior;
    }

    /**
     * <p>
     * Specifies how to handle request payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the corresponding
     * binary blob.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If this property is not defined, the request payload will be passed through from the method request to
     * integration request without modification, provided that the <code>passthroughBehaviors</code> is configured to
     * support payload pass-through.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle request payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the
     *         corresponding binary blob.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If this property is not defined, the request payload will be passed through from the method request to
     *         integration request without modification, provided that the <code>passthroughBehaviors</code> is
     *         configured to support payload pass-through.
     * @see ContentHandlingStrategy
     */
    public ContentHandlingStrategy contentHandling() {
        return ContentHandlingStrategy.fromValue(contentHandling);
    }

    /**
     * <p>
     * Specifies how to handle request payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the corresponding
     * binary blob.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If this property is not defined, the request payload will be passed through from the method request to
     * integration request without modification, provided that the <code>passthroughBehaviors</code> is configured to
     * support payload pass-through.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle request payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the
     *         corresponding binary blob.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If this property is not defined, the request payload will be passed through from the method request to
     *         integration request without modification, provided that the <code>passthroughBehaviors</code> is
     *         configured to support payload pass-through.
     * @see ContentHandlingStrategy
     */
    public String contentHandlingAsString() {
        return contentHandling;
    }

    /**
     * <p>
     * Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
     * </p>
     * 
     * @return Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29
     *         seconds.
     */
    public Integer timeoutInMillis() {
        return timeoutInMillis;
    }

    /**
     * <p>
     * Specifies the integration's cache namespace.
     * </p>
     * 
     * @return Specifies the integration's cache namespace.
     */
    public String cacheNamespace() {
        return cacheNamespace;
    }

    /**
     * <p>
     * Specifies the integration's cache key parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the integration's cache key parameters.
     */
    public List<String> cacheKeyParameters() {
        return cacheKeyParameters;
    }

    /**
     * <p>
     * Specifies the integration's responses.
     * </p>
     * <div class="remarks">
     * <p/>
     * <h4>Example: Get integration responses of a method</h4>
     * <h5>Request</h5>
     * <p/>
     * 
     * <pre>
     * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160607T191449Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160607/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash} </code>
     * </pre>
     * 
     * <h5>Response</h5>
     * <p>
     * The successful response returns <code>200 OK</code> status and a payload as follows:
     * </p>
     * 
     * <pre>
     * <code>{ "_links": { "curies": { "href": "http://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true }, "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.Content-Type": "'application/xml'" }, "responseTemplates": { "application/json": "$util.urlDecode(\"%3CkinesisStreams%3E#foreach($stream in $input.path('$.StreamNames'))%3Cstream%3E%3Cname%3E$stream%3C/name%3E%3C/stream%3E#end%3C/kinesisStreams%3E\")\n" }, "statusCode": "200" }</code>
     * </pre>
     * <p/>
     * </div> <div class="seeAlso"> <a
     * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an API</a>
     * </div>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the integration's responses.</p> <div class="remarks">
     *         <p/>
     *         <h4>Example: Get integration responses of a method</h4>
     *         <h5>Request</h5>
     *         <p/>
     * 
     *         <pre>
     * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160607T191449Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160607/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash} </code>
     * </pre>
     * 
     *         <h5>Response</h5>
     *         <p>
     *         The successful response returns <code>200 OK</code> status and a payload as follows:
     *         </p>
     * 
     *         <pre>
     * <code>{ "_links": { "curies": { "href": "http://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true }, "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.Content-Type": "'application/xml'" }, "responseTemplates": { "application/json": "$util.urlDecode(\"%3CkinesisStreams%3E#foreach($stream in $input.path('$.StreamNames'))%3Cstream%3E%3Cname%3E$stream%3C/name%3E%3C/stream%3E#end%3C/kinesisStreams%3E\")\n" }, "statusCode": "200" }</code>
     * </pre>
     *         <p/>
     *         </div> <div class="seeAlso"> <a
     *         href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an
     *         API</a>
     */
    public Map<String, IntegrationResponse> integrationResponses() {
        return integrationResponses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(httpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        hashCode = 31 * hashCode + Objects.hashCode(connectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(connectionId());
        hashCode = 31 * hashCode + Objects.hashCode(credentials());
        hashCode = 31 * hashCode + Objects.hashCode(requestParameters());
        hashCode = 31 * hashCode + Objects.hashCode(requestTemplates());
        hashCode = 31 * hashCode + Objects.hashCode(passthroughBehavior());
        hashCode = 31 * hashCode + Objects.hashCode(contentHandlingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMillis());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNamespace());
        hashCode = 31 * hashCode + Objects.hashCode(cacheKeyParameters());
        hashCode = 31 * hashCode + Objects.hashCode(integrationResponses());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Integration)) {
            return false;
        }
        Integration other = (Integration) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(httpMethod(), other.httpMethod())
                && Objects.equals(uri(), other.uri()) && Objects.equals(connectionTypeAsString(), other.connectionTypeAsString())
                && Objects.equals(connectionId(), other.connectionId()) && Objects.equals(credentials(), other.credentials())
                && Objects.equals(requestParameters(), other.requestParameters())
                && Objects.equals(requestTemplates(), other.requestTemplates())
                && Objects.equals(passthroughBehavior(), other.passthroughBehavior())
                && Objects.equals(contentHandlingAsString(), other.contentHandlingAsString())
                && Objects.equals(timeoutInMillis(), other.timeoutInMillis())
                && Objects.equals(cacheNamespace(), other.cacheNamespace())
                && Objects.equals(cacheKeyParameters(), other.cacheKeyParameters())
                && Objects.equals(integrationResponses(), other.integrationResponses());
    }

    @Override
    public String toString() {
        return ToString.builder("Integration").add("Type", typeAsString()).add("HttpMethod", httpMethod()).add("Uri", uri())
                .add("ConnectionType", connectionTypeAsString()).add("ConnectionId", connectionId())
                .add("Credentials", credentials()).add("RequestParameters", requestParameters())
                .add("RequestTemplates", requestTemplates()).add("PassthroughBehavior", passthroughBehavior())
                .add("ContentHandling", contentHandlingAsString()).add("TimeoutInMillis", timeoutInMillis())
                .add("CacheNamespace", cacheNamespace()).add("CacheKeyParameters", cacheKeyParameters())
                .add("IntegrationResponses", integrationResponses()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "httpMethod":
            return Optional.ofNullable(clazz.cast(httpMethod()));
        case "uri":
            return Optional.ofNullable(clazz.cast(uri()));
        case "connectionType":
            return Optional.ofNullable(clazz.cast(connectionTypeAsString()));
        case "connectionId":
            return Optional.ofNullable(clazz.cast(connectionId()));
        case "credentials":
            return Optional.ofNullable(clazz.cast(credentials()));
        case "requestParameters":
            return Optional.ofNullable(clazz.cast(requestParameters()));
        case "requestTemplates":
            return Optional.ofNullable(clazz.cast(requestTemplates()));
        case "passthroughBehavior":
            return Optional.ofNullable(clazz.cast(passthroughBehavior()));
        case "contentHandling":
            return Optional.ofNullable(clazz.cast(contentHandlingAsString()));
        case "timeoutInMillis":
            return Optional.ofNullable(clazz.cast(timeoutInMillis()));
        case "cacheNamespace":
            return Optional.ofNullable(clazz.cast(cacheNamespace()));
        case "cacheKeyParameters":
            return Optional.ofNullable(clazz.cast(cacheKeyParameters()));
        case "integrationResponses":
            return Optional.ofNullable(clazz.cast(integrationResponses()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        IntegrationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Integration> {
        /**
         * <p>
         * Specifies an API method integration type. The valid value is one of the following:
         * </p>
         * <ul>
         * <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the Lambda
         * function-invoking action. With the Lambda function-invoking action, this is referred to as the Lambda custom
         * integration. With any other AWS service action, this is known as AWS integration.</li>
         * <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking action
         * with the client request passed through as-is. This integration is also referred to as the Lambda proxy
         * integration.</li>
         * <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private HTTP
         * endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
         * <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
         * private HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred to as
         * the HTTP proxy integration.</li>
         * <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint
         * without invoking any backend.</li>
         * </ul>
         * <p>
         * For the HTTP and HTTP proxy integrations, each integration can specify a protocol (<code>http/https</code>),
         * port and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP
         * proxy integration with a <code>connectionType</code> of <code>VPC_LINK</code> is referred to as a private
         * integration and uses a <a>VpcLink</a> to connect API Gateway to a network load balancer of a VPC.
         * </p>
         * 
         * @param type
         *        Specifies an API method integration type. The valid value is one of the following:</p>
         *        <ul>
         *        <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the
         *        Lambda function-invoking action. With the Lambda function-invoking action, this is referred to as the
         *        Lambda custom integration. With any other AWS service action, this is known as AWS integration.</li>
         *        <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking
         *        action with the client request passed through as-is. This integration is also referred to as the
         *        Lambda proxy integration.</li>
         *        <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a
         *        private HTTP endpoint within a VPC. This integration is also referred to as the HTTP custom
         *        integration.</li>
         *        <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
         *        private HTTP endpoint within a VPC, with the client request passed through as-is. This is also
         *        referred to as the HTTP proxy integration.</li>
         *        <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back"
         *        endpoint without invoking any backend.</li>
         *        </ul>
         *        <p>
         *        For the HTTP and HTTP proxy integrations, each integration can specify a protocol (
         *        <code>http/https</code>), port and path. Standard 80 and 443 ports are supported as well as custom
         *        ports above 1024. An HTTP or HTTP proxy integration with a <code>connectionType</code> of
         *        <code>VPC_LINK</code> is referred to as a private integration and uses a <a>VpcLink</a> to connect API
         *        Gateway to a network load balancer of a VPC.
         * @see IntegrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies an API method integration type. The valid value is one of the following:
         * </p>
         * <ul>
         * <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the Lambda
         * function-invoking action. With the Lambda function-invoking action, this is referred to as the Lambda custom
         * integration. With any other AWS service action, this is known as AWS integration.</li>
         * <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking action
         * with the client request passed through as-is. This integration is also referred to as the Lambda proxy
         * integration.</li>
         * <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a private HTTP
         * endpoint within a VPC. This integration is also referred to as the HTTP custom integration.</li>
         * <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
         * private HTTP endpoint within a VPC, with the client request passed through as-is. This is also referred to as
         * the HTTP proxy integration.</li>
         * <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back" endpoint
         * without invoking any backend.</li>
         * </ul>
         * <p>
         * For the HTTP and HTTP proxy integrations, each integration can specify a protocol (<code>http/https</code>),
         * port and path. Standard 80 and 443 ports are supported as well as custom ports above 1024. An HTTP or HTTP
         * proxy integration with a <code>connectionType</code> of <code>VPC_LINK</code> is referred to as a private
         * integration and uses a <a>VpcLink</a> to connect API Gateway to a network load balancer of a VPC.
         * </p>
         * 
         * @param type
         *        Specifies an API method integration type. The valid value is one of the following:</p>
         *        <ul>
         *        <li><code>AWS</code>: for integrating the API method request with an AWS service action, including the
         *        Lambda function-invoking action. With the Lambda function-invoking action, this is referred to as the
         *        Lambda custom integration. With any other AWS service action, this is known as AWS integration.</li>
         *        <li><code>AWS_PROXY</code>: for integrating the API method request with the Lambda function-invoking
         *        action with the client request passed through as-is. This integration is also referred to as the
         *        Lambda proxy integration.</li>
         *        <li><code>HTTP</code>: for integrating the API method request with an HTTP endpoint, including a
         *        private HTTP endpoint within a VPC. This integration is also referred to as the HTTP custom
         *        integration.</li>
         *        <li><code>HTTP_PROXY</code>: for integrating the API method request with an HTTP endpoint, including a
         *        private HTTP endpoint within a VPC, with the client request passed through as-is. This is also
         *        referred to as the HTTP proxy integration.</li>
         *        <li><code>MOCK</code>: for integrating the API method request with API Gateway as a "loop-back"
         *        endpoint without invoking any backend.</li>
         *        </ul>
         *        <p>
         *        For the HTTP and HTTP proxy integrations, each integration can specify a protocol (
         *        <code>http/https</code>), port and path. Standard 80 and 443 ports are supported as well as custom
         *        ports above 1024. An HTTP or HTTP proxy integration with a <code>connectionType</code> of
         *        <code>VPC_LINK</code> is referred to as a private integration and uses a <a>VpcLink</a> to connect API
         *        Gateway to a network load balancer of a VPC.
         * @see IntegrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationType
         */
        Builder type(IntegrationType type);

        /**
         * <p>
         * Specifies the integration's HTTP method type.
         * </p>
         * 
         * @param httpMethod
         *        Specifies the integration's HTTP method type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpMethod(String httpMethod);

        /**
         * <p>
         * Specifies Uniform Resource Identifier (URI) of the integration endpoint.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>HTTP</code> or <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded
         * HTTP(S) URL according to the <a target="_blank"
         * href="https://en.wikipedia.org/wiki/Uniform_Resource_Identifier">RFC-3986 specification</a>, for either
         * standard integration, where <code>connectionType</code> is not <code>VPC_LINK</code>, or private integration,
         * where <code>connectionType</code> is <code>VPC_LINK</code>. For a private HTTP integration, the URI is not
         * used for routing.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
         * <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}</code>. Here,
         * <code>{Region}</code> is the API Gateway region (e.g., <code>us-east-1</code>); <code>{service}</code> is the
         * name of the integrated AWS service (e.g., <code>s3</code>); and <code>{subdomain}</code> is a designated
         * subdomain supported by certain AWS service for fast host-name lookup. <code>action</code> can be used for an
         * AWS service action-based API, using an <code>Action={name}&amp;{p1}={v1}&amp;p2={v2}...</code> query string.
         * The ensuing <code>{service_api}</code> refers to a supported action <code>{name}</code> plus any required
         * input parameters. Alternatively, <code>path</code> can be used for an AWS service path-based API. The ensuing
         * <code>service_api</code> refers to the path to an AWS service resource, including the region of the
         * integrated AWS service, if applicable. For example, for integration with the S3 API of
         * <code><a href="http://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectGET.html">GetObject</a></code>, the
         * <code>uri</code> can be either
         * <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
         * <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param uri
         *        Specifies Uniform Resource Identifier (URI) of the integration endpoint.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>HTTP</code> or <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded
         *        HTTP(S) URL according to the <a target="_blank"
         *        href="https://en.wikipedia.org/wiki/Uniform_Resource_Identifier">RFC-3986 specification</a>, for
         *        either standard integration, where <code>connectionType</code> is not <code>VPC_LINK</code>, or
         *        private integration, where <code>connectionType</code> is <code>VPC_LINK</code>. For a private HTTP
         *        integration, the URI is not used for routing.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
         *        <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api}</code>. Here,
         *        <code>{Region}</code> is the API Gateway region (e.g., <code>us-east-1</code>); <code>{service}</code>
         *        is the name of the integrated AWS service (e.g., <code>s3</code>); and <code>{subdomain}</code> is a
         *        designated subdomain supported by certain AWS service for fast host-name lookup. <code>action</code>
         *        can be used for an AWS service action-based API, using an
         *        <code>Action={name}&amp;{p1}={v1}&amp;p2={v2}...</code> query string. The ensuing
         *        <code>{service_api}</code> refers to a supported action <code>{name}</code> plus any required input
         *        parameters. Alternatively, <code>path</code> can be used for an AWS service path-based API. The
         *        ensuing <code>service_api</code> refers to the path to an AWS service resource, including the region
         *        of the integrated AWS service, if applicable. For example, for integration with the S3 API of
         *        <code><a href="http://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectGET.html">GetObject</a></code>
         *        , the <code>uri</code> can be either
         *        <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
         *        <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);

        /**
         * <p>
         * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
         * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
         * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
         * </p>
         * 
         * @param connectionType
         *        The type of the network connection to the integration endpoint. The valid value is
         *        <code>INTERNET</code> for connections through the public routable internet or <code>VPC_LINK</code>
         *        for private connections between API Gateway and a network load balancer in a VPC. The default value is
         *        <code>INTERNET</code>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
         * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
         * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
         * </p>
         * 
         * @param connectionType
         *        The type of the network connection to the integration endpoint. The valid value is
         *        <code>INTERNET</code> for connections through the public routable internet or <code>VPC_LINK</code>
         *        for private connections between API Gateway and a network load balancer in a VPC. The default value is
         *        <code>INTERNET</code>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(ConnectionType connectionType);

        /**
         * <p>
         * The (<a href="http://docs.aws.amazon.com/apigateway/api-reference/resource/vpc-link/#id"><code>id</code></a>)
         * of the <a>VpcLink</a> used for the integration when <code>connectionType=VPC_LINK</code> and undefined,
         * otherwise.
         * </p>
         * 
         * @param connectionId
         *        The (<a href="http://docs.aws.amazon.com/apigateway/api-reference/resource/vpc-link/#id">
         *        <code>id</code></a>) of the <a>VpcLink</a> used for the integration when
         *        <code>connectionType=VPC_LINK</code> and undefined, otherwise.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionId(String connectionId);

        /**
         * <p>
         * Specifies the credentials required for the integration, if any. For AWS integrations, three options are
         * available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To
         * require that the caller's identity be passed through from the request, specify the string
         * <code>arn:aws:iam::\*:user/\*</code>. To use resource-based permissions on supported AWS services, specify
         * null.
         * </p>
         * 
         * @param credentials
         *        Specifies the credentials required for the integration, if any. For AWS integrations, three options
         *        are available. To specify an IAM Role for API Gateway to assume, use the role's Amazon Resource Name
         *        (ARN). To require that the caller's identity be passed through from the request, specify the string
         *        <code>arn:aws:iam::\*:user/\*</code>. To use resource-based permissions on supported AWS services,
         *        specify null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentials(String credentials);

        /**
         * <p>
         * A key-value map specifying request parameters that are passed from the method request to the back end. The
         * key is an integration request parameter name and the associated value is a method request parameter value or
         * static value that must be enclosed within single quotes and pre-encoded as required by the back end. The
         * method request parameter value must match the pattern of <code>method.request.{location}.{name}</code>, where
         * <code>location</code> is <code>querystring</code>, <code>path</code>, or <code>header</code> and
         * <code>name</code> must be a valid and unique method request parameter name.
         * </p>
         * 
         * @param requestParameters
         *        A key-value map specifying request parameters that are passed from the method request to the back end.
         *        The key is an integration request parameter name and the associated value is a method request
         *        parameter value or static value that must be enclosed within single quotes and pre-encoded as required
         *        by the back end. The method request parameter value must match the pattern of
         *        <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>
         *        , <code>path</code>, or <code>header</code> and <code>name</code> must be a valid and unique method
         *        request parameter name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestParameters(Map<String, String> requestParameters);

        /**
         * <p>
         * Represents a map of Velocity templates that are applied on the request payload based on the value of the
         * Content-Type header sent by the client. The content type value is the key in this map, and the template (as a
         * String) is the value.
         * </p>
         * 
         * @param requestTemplates
         *        Represents a map of Velocity templates that are applied on the request payload based on the value of
         *        the Content-Type header sent by the client. The content type value is the key in this map, and the
         *        template (as a String) is the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestTemplates(Map<String, String> requestTemplates);

        /**
         * <div>
         * <p>
         * Specifies how the method request body of an unmapped content type will be passed through the integration
         * request to the back end without transformation. A content type is unmapped if no mapping template is defined
         * in the integration or the content type does not match any of the mapped content types, as specified in
         * <code>requestTemplates</code>. The valid value is one of the following:
         * </p>
         * <ul>
         * <li> <code>WHEN_NO_MATCH</code>: passes the method request body through the integration request to the back
         * end without transformation when the method request content type does not match any content type associated
         * with the mapping templates defined in the integration request.</li>
         * <li> <code>WHEN_NO_TEMPLATES</code>: passes the method request body through the integration request to the
         * back end without transformation when no mapping template is defined in the integration request. If a template
         * is defined when this option is selected, the method request of an unmapped content-type will be rejected with
         * an HTTP <code>415 Unsupported Media Type</code> response.</li>
         * <li> <code>NEVER</code>: rejects the method request with an HTTP <code>415 Unsupported Media Type</code>
         * response when either the method request content type does not match any content type associated with the
         * mapping templates defined in the integration request or no mapping template is defined in the integration
         * request.</li>
         * </ul>
         * </div>
         * 
         * @param passthroughBehavior
         *        <p>
         *        Specifies how the method request body of an unmapped content type will be passed through the
         *        integration request to the back end without transformation. A content type is unmapped if no mapping
         *        template is defined in the integration or the content type does not match any of the mapped content
         *        types, as specified in <code>requestTemplates</code>. The valid value is one of the following:
         *        </p>
         *        <ul>
         *        <li> <code>WHEN_NO_MATCH</code>: passes the method request body through the integration request to the
         *        back end without transformation when the method request content type does not match any content type
         *        associated with the mapping templates defined in the integration request.</li>
         *        <li> <code>WHEN_NO_TEMPLATES</code>: passes the method request body through the integration request to
         *        the back end without transformation when no mapping template is defined in the integration request. If
         *        a template is defined when this option is selected, the method request of an unmapped content-type
         *        will be rejected with an HTTP <code>415 Unsupported Media Type</code> response.</li>
         *        <li> <code>NEVER</code>: rejects the method request with an HTTP
         *        <code>415 Unsupported Media Type</code> response when either the method request content type does not
         *        match any content type associated with the mapping templates defined in the integration request or no
         *        mapping template is defined in the integration request.</li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder passthroughBehavior(String passthroughBehavior);

        /**
         * <p>
         * Specifies how to handle request payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the corresponding
         * binary blob.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If this property is not defined, the request payload will be passed through from the method request to
         * integration request without modification, provided that the <code>passthroughBehaviors</code> is configured
         * to support payload pass-through.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle request payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the
         *        corresponding binary blob.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded
         *        string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If this property is not defined, the request payload will be passed through from the method request to
         *        integration request without modification, provided that the <code>passthroughBehaviors</code> is
         *        configured to support payload pass-through.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(String contentHandling);

        /**
         * <p>
         * Specifies how to handle request payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the corresponding
         * binary blob.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If this property is not defined, the request payload will be passed through from the method request to
         * integration request without modification, provided that the <code>passthroughBehaviors</code> is configured
         * to support payload pass-through.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle request payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_BINARY</code>: Converts a request payload from a Base64-encoded string to the
         *        corresponding binary blob.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_TEXT</code>: Converts a request payload from a binary blob to a Base64-encoded
         *        string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If this property is not defined, the request payload will be passed through from the method request to
         *        integration request without modification, provided that the <code>passthroughBehaviors</code> is
         *        configured to support payload pass-through.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(ContentHandlingStrategy contentHandling);

        /**
         * <p>
         * Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
         * </p>
         * 
         * @param timeoutInMillis
         *        Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29
         *        seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMillis(Integer timeoutInMillis);

        /**
         * <p>
         * Specifies the integration's cache namespace.
         * </p>
         * 
         * @param cacheNamespace
         *        Specifies the integration's cache namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNamespace(String cacheNamespace);

        /**
         * <p>
         * Specifies the integration's cache key parameters.
         * </p>
         * 
         * @param cacheKeyParameters
         *        Specifies the integration's cache key parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheKeyParameters(Collection<String> cacheKeyParameters);

        /**
         * <p>
         * Specifies the integration's cache key parameters.
         * </p>
         * 
         * @param cacheKeyParameters
         *        Specifies the integration's cache key parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheKeyParameters(String... cacheKeyParameters);

        /**
         * <p>
         * Specifies the integration's responses.
         * </p>
         * <div class="remarks">
         * <p/>
         * <h4>Example: Get integration responses of a method</h4>
         * <h5>Request</h5>
         * <p/>
         * 
         * <pre>
         * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160607T191449Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160607/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash} </code>
         * </pre>
         * 
         * <h5>Response</h5>
         * <p>
         * The successful response returns <code>200 OK</code> status and a payload as follows:
         * </p>
         * 
         * <pre>
         * <code>{ "_links": { "curies": { "href": "http://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true }, "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.Content-Type": "'application/xml'" }, "responseTemplates": { "application/json": "$util.urlDecode(\"%3CkinesisStreams%3E#foreach($stream in $input.path('$.StreamNames'))%3Cstream%3E%3Cname%3E$stream%3C/name%3E%3C/stream%3E#end%3C/kinesisStreams%3E\")\n" }, "statusCode": "200" }</code>
         * </pre>
         * <p/>
         * </div> <div class="seeAlso"> <a
         * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an API</a>
         * </div>
         * 
         * @param integrationResponses
         *        Specifies the integration's responses.
         *        </p>
         *        <div class="remarks">
         *        <p/>
         *        <h4>Example: Get integration responses of a method</h4>
         *        <h5>Request</h5>
         *        <p/>
         * 
         *        <pre>
         * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160607T191449Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160607/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash} </code>
         * </pre>
         * 
         *        <h5>Response</h5>
         *        <p>
         *        The successful response returns <code>200 OK</code> status and a payload as follows:
         *        </p>
         * 
         *        <pre>
         * <code>{ "_links": { "curies": { "href": "http://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true }, "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.Content-Type": "'application/xml'" }, "responseTemplates": { "application/json": "$util.urlDecode(\"%3CkinesisStreams%3E#foreach($stream in $input.path('$.StreamNames'))%3Cstream%3E%3Cname%3E$stream%3C/name%3E%3C/stream%3E#end%3C/kinesisStreams%3E\")\n" }, "statusCode": "200" }</code>
         * </pre>
         *        <p/>
         *        </div> <div class="seeAlso"> <a
         *        href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an
         *        API</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integrationResponses(Map<String, IntegrationResponse> integrationResponses);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String httpMethod;

        private String uri;

        private String connectionType;

        private String connectionId;

        private String credentials;

        private Map<String, String> requestParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> requestTemplates = DefaultSdkAutoConstructMap.getInstance();

        private String passthroughBehavior;

        private String contentHandling;

        private Integer timeoutInMillis;

        private String cacheNamespace;

        private List<String> cacheKeyParameters = DefaultSdkAutoConstructList.getInstance();

        private Map<String, IntegrationResponse> integrationResponses = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Integration model) {
            type(model.type);
            httpMethod(model.httpMethod);
            uri(model.uri);
            connectionType(model.connectionType);
            connectionId(model.connectionId);
            credentials(model.credentials);
            requestParameters(model.requestParameters);
            requestTemplates(model.requestTemplates);
            passthroughBehavior(model.passthroughBehavior);
            contentHandling(model.contentHandling);
            timeoutInMillis(model.timeoutInMillis);
            cacheNamespace(model.cacheNamespace);
            cacheKeyParameters(model.cacheKeyParameters);
            integrationResponses(model.integrationResponses);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(IntegrationType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getHttpMethod() {
            return httpMethod;
        }

        @Override
        public final Builder httpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public final void setHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
        }

        public final String getUri() {
            return uri;
        }

        @Override
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        public final String getConnectionType() {
            return connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        @Override
        public final Builder connectionType(ConnectionType connectionType) {
            this.connectionType(connectionType.toString());
            return this;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        public final String getConnectionId() {
            return connectionId;
        }

        @Override
        public final Builder connectionId(String connectionId) {
            this.connectionId = connectionId;
            return this;
        }

        public final void setConnectionId(String connectionId) {
            this.connectionId = connectionId;
        }

        public final String getCredentials() {
            return credentials;
        }

        @Override
        public final Builder credentials(String credentials) {
            this.credentials = credentials;
            return this;
        }

        public final void setCredentials(String credentials) {
            this.credentials = credentials;
        }

        public final Map<String, String> getRequestParameters() {
            return requestParameters;
        }

        @Override
        public final Builder requestParameters(Map<String, String> requestParameters) {
            this.requestParameters = MapOfStringToStringCopier.copy(requestParameters);
            return this;
        }

        public final void setRequestParameters(Map<String, String> requestParameters) {
            this.requestParameters = MapOfStringToStringCopier.copy(requestParameters);
        }

        public final Map<String, String> getRequestTemplates() {
            return requestTemplates;
        }

        @Override
        public final Builder requestTemplates(Map<String, String> requestTemplates) {
            this.requestTemplates = MapOfStringToStringCopier.copy(requestTemplates);
            return this;
        }

        public final void setRequestTemplates(Map<String, String> requestTemplates) {
            this.requestTemplates = MapOfStringToStringCopier.copy(requestTemplates);
        }

        public final String getPassthroughBehavior() {
            return passthroughBehavior;
        }

        @Override
        public final Builder passthroughBehavior(String passthroughBehavior) {
            this.passthroughBehavior = passthroughBehavior;
            return this;
        }

        public final void setPassthroughBehavior(String passthroughBehavior) {
            this.passthroughBehavior = passthroughBehavior;
        }

        public final String getContentHandling() {
            return contentHandling;
        }

        @Override
        public final Builder contentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
            return this;
        }

        @Override
        public final Builder contentHandling(ContentHandlingStrategy contentHandling) {
            this.contentHandling(contentHandling.toString());
            return this;
        }

        public final void setContentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
        }

        public final Integer getTimeoutInMillis() {
            return timeoutInMillis;
        }

        @Override
        public final Builder timeoutInMillis(Integer timeoutInMillis) {
            this.timeoutInMillis = timeoutInMillis;
            return this;
        }

        public final void setTimeoutInMillis(Integer timeoutInMillis) {
            this.timeoutInMillis = timeoutInMillis;
        }

        public final String getCacheNamespace() {
            return cacheNamespace;
        }

        @Override
        public final Builder cacheNamespace(String cacheNamespace) {
            this.cacheNamespace = cacheNamespace;
            return this;
        }

        public final void setCacheNamespace(String cacheNamespace) {
            this.cacheNamespace = cacheNamespace;
        }

        public final Collection<String> getCacheKeyParameters() {
            return cacheKeyParameters;
        }

        @Override
        public final Builder cacheKeyParameters(Collection<String> cacheKeyParameters) {
            this.cacheKeyParameters = ListOfStringCopier.copy(cacheKeyParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheKeyParameters(String... cacheKeyParameters) {
            cacheKeyParameters(Arrays.asList(cacheKeyParameters));
            return this;
        }

        public final void setCacheKeyParameters(Collection<String> cacheKeyParameters) {
            this.cacheKeyParameters = ListOfStringCopier.copy(cacheKeyParameters);
        }

        public final Map<String, IntegrationResponse.Builder> getIntegrationResponses() {
            return integrationResponses != null ? CollectionUtils.mapValues(integrationResponses, IntegrationResponse::toBuilder)
                    : null;
        }

        @Override
        public final Builder integrationResponses(Map<String, IntegrationResponse> integrationResponses) {
            this.integrationResponses = MapOfIntegrationResponseCopier.copy(integrationResponses);
            return this;
        }

        public final void setIntegrationResponses(Map<String, IntegrationResponse.BuilderImpl> integrationResponses) {
            this.integrationResponses = MapOfIntegrationResponseCopier.copyFromBuilder(integrationResponses);
        }

        @Override
        public Integration build() {
            return new Integration(this);
        }
    }
}
