/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A gateway response of a given response type and status code, with optional response parameters and mapping templates.
 * </p>
 * <div class="remarks"> For more information about valid gateway response types, see <a
 * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html">Gateway
 * Response Types Supported by API Gateway</a> <div class="example"> <h4>Example: Get a Gateway Response of a given
 * response type</h4> <h5>Request</h5>
 * <p>
 * This example shows how to get a gateway response of the <code>MISSING_AUTHENTICATION_TOKEN</code> type.
 * </p>
 * 
 * <pre>
 * <code>GET /restapis/o81lxisefl/gatewayresponses/MISSING_AUTHENTICATION_TOKEN HTTP/1.1 Host: beta-apigateway.us-east-1.amazonaws.com Content-Type: application/json X-Amz-Date: 20170503T202516Z Authorization: AWS4-HMAC-SHA256 Credential={access-key-id}/20170503/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature=1b52460e3159c1a26cff29093855d50ea141c1c5b937528fecaf60f51129697a Cache-Control: no-cache Postman-Token: 3b2a1ce9-c848-2e26-2e2f-9c2caefbed45 </code>
 * </pre>
 * <p>
 * The response type is specified as a URL path.
 * </p>
 * <h5>Response</h5>
 * <p>
 * The successful operation returns the <code>200 OK</code> status code and a payload similar to the following:
 * </p>
 * 
 * <pre>
 * <code>{ "_links": { "curies": { "href": "http://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-gatewayresponse-{rel}.html", "name": "gatewayresponse", "templated": true }, "self": { "href": "/restapis/o81lxisefl/gatewayresponses/MISSING_AUTHENTICATION_TOKEN" }, "gatewayresponse:delete": { "href": "/restapis/o81lxisefl/gatewayresponses/MISSING_AUTHENTICATION_TOKEN" }, "gatewayresponse:put": { "href": "/restapis/o81lxisefl/gatewayresponses/{response_type}", "templated": true }, "gatewayresponse:update": { "href": "/restapis/o81lxisefl/gatewayresponses/MISSING_AUTHENTICATION_TOKEN" } }, "defaultResponse": false, "responseParameters": { "gatewayresponse.header.x-request-path": "method.request.path.petId", "gatewayresponse.header.Access-Control-Allow-Origin": "&apos;a.b.c&apos;", "gatewayresponse.header.x-request-query": "method.request.querystring.q", "gatewayresponse.header.x-request-header": "method.request.header.Accept" }, "responseTemplates": { "application/json": "{\n \"message\": $context.error.messageString,\n \"type\": \"$context.error.responseType\",\n \"stage\": \"$context.stage\",\n \"resourcePath\": \"$context.resourcePath\",\n \"stageVariables.a\": \"$stageVariables.a\",\n \"statusCode\": \"&apos;404&apos;\"\n}" }, "responseType": "MISSING_AUTHENTICATION_TOKEN", "statusCode": "404" }</code>
 * </pre>
 * <p>
 * </p>
 * </div> </div> <div class="seeAlso"> <a
 * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html">Customize Gateway
 * Responses</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateGatewayResponseResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<UpdateGatewayResponseResponse.Builder, UpdateGatewayResponseResponse> {
    private final String responseType;

    private final String statusCode;

    private final Map<String, String> responseParameters;

    private final Map<String, String> responseTemplates;

    private final Boolean defaultResponse;

    private UpdateGatewayResponseResponse(BuilderImpl builder) {
        super(builder);
        this.responseType = builder.responseType;
        this.statusCode = builder.statusCode;
        this.responseParameters = builder.responseParameters;
        this.responseTemplates = builder.responseTemplates;
        this.defaultResponse = builder.defaultResponse;
    }

    /**
     * <p>
     * The response type of the associated <a>GatewayResponse</a>. Valid values are
     * <ul>
     * <li>ACCESS_DENIED</li>
     * <li>API_CONFIGURATION_ERROR</li>
     * <li>AUTHORIZER_FAILURE</li>
     * <li>AUTHORIZER_CONFIGURATION_ERROR</li>
     * <li>BAD_REQUEST_PARAMETERS</li>
     * <li>BAD_REQUEST_BODY</li>
     * <li>DEFAULT_4XX</li>
     * <li>DEFAULT_5XX</li>
     * <li>EXPIRED_TOKEN</li>
     * <li>INVALID_SIGNATURE</li>
     * <li>INTEGRATION_FAILURE</li>
     * <li>INTEGRATION_TIMEOUT</li>
     * <li>INVALID_API_KEY</li>
     * <li>MISSING_AUTHENTICATION_TOKEN</li>
     * <li>QUOTA_EXCEEDED</li>
     * <li>REQUEST_TOO_LARGE</li>
     * <li>RESOURCE_NOT_FOUND</li>
     * <li>THROTTLED</li>
     * <li>UNAUTHORIZED</li>
     * <li>UNSUPPORTED_MEDIA_TYPE</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseType} will
     * return {@link GatewayResponseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #responseTypeAsString}.
     * </p>
     * 
     * @return The response type of the associated <a>GatewayResponse</a>. Valid values are
     *         <ul>
     *         <li>ACCESS_DENIED</li>
     *         <li>API_CONFIGURATION_ERROR</li>
     *         <li>AUTHORIZER_FAILURE</li>
     *         <li>AUTHORIZER_CONFIGURATION_ERROR</li>
     *         <li>BAD_REQUEST_PARAMETERS</li>
     *         <li>BAD_REQUEST_BODY</li>
     *         <li>DEFAULT_4XX</li>
     *         <li>DEFAULT_5XX</li>
     *         <li>EXPIRED_TOKEN</li>
     *         <li>INVALID_SIGNATURE</li>
     *         <li>INTEGRATION_FAILURE</li>
     *         <li>INTEGRATION_TIMEOUT</li>
     *         <li>INVALID_API_KEY</li>
     *         <li>MISSING_AUTHENTICATION_TOKEN</li>
     *         <li>QUOTA_EXCEEDED</li>
     *         <li>REQUEST_TOO_LARGE</li>
     *         <li>RESOURCE_NOT_FOUND</li>
     *         <li>THROTTLED</li>
     *         <li>UNAUTHORIZED</li>
     *         <li>UNSUPPORTED_MEDIA_TYPE</li>
     *         </ul>
     * @see GatewayResponseType
     */
    public GatewayResponseType responseType() {
        return GatewayResponseType.fromValue(responseType);
    }

    /**
     * <p>
     * The response type of the associated <a>GatewayResponse</a>. Valid values are
     * <ul>
     * <li>ACCESS_DENIED</li>
     * <li>API_CONFIGURATION_ERROR</li>
     * <li>AUTHORIZER_FAILURE</li>
     * <li>AUTHORIZER_CONFIGURATION_ERROR</li>
     * <li>BAD_REQUEST_PARAMETERS</li>
     * <li>BAD_REQUEST_BODY</li>
     * <li>DEFAULT_4XX</li>
     * <li>DEFAULT_5XX</li>
     * <li>EXPIRED_TOKEN</li>
     * <li>INVALID_SIGNATURE</li>
     * <li>INTEGRATION_FAILURE</li>
     * <li>INTEGRATION_TIMEOUT</li>
     * <li>INVALID_API_KEY</li>
     * <li>MISSING_AUTHENTICATION_TOKEN</li>
     * <li>QUOTA_EXCEEDED</li>
     * <li>REQUEST_TOO_LARGE</li>
     * <li>RESOURCE_NOT_FOUND</li>
     * <li>THROTTLED</li>
     * <li>UNAUTHORIZED</li>
     * <li>UNSUPPORTED_MEDIA_TYPE</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseType} will
     * return {@link GatewayResponseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #responseTypeAsString}.
     * </p>
     * 
     * @return The response type of the associated <a>GatewayResponse</a>. Valid values are
     *         <ul>
     *         <li>ACCESS_DENIED</li>
     *         <li>API_CONFIGURATION_ERROR</li>
     *         <li>AUTHORIZER_FAILURE</li>
     *         <li>AUTHORIZER_CONFIGURATION_ERROR</li>
     *         <li>BAD_REQUEST_PARAMETERS</li>
     *         <li>BAD_REQUEST_BODY</li>
     *         <li>DEFAULT_4XX</li>
     *         <li>DEFAULT_5XX</li>
     *         <li>EXPIRED_TOKEN</li>
     *         <li>INVALID_SIGNATURE</li>
     *         <li>INTEGRATION_FAILURE</li>
     *         <li>INTEGRATION_TIMEOUT</li>
     *         <li>INVALID_API_KEY</li>
     *         <li>MISSING_AUTHENTICATION_TOKEN</li>
     *         <li>QUOTA_EXCEEDED</li>
     *         <li>REQUEST_TOO_LARGE</li>
     *         <li>RESOURCE_NOT_FOUND</li>
     *         <li>THROTTLED</li>
     *         <li>UNAUTHORIZED</li>
     *         <li>UNSUPPORTED_MEDIA_TYPE</li>
     *         </ul>
     * @see GatewayResponseType
     */
    public String responseTypeAsString() {
        return responseType;
    }

    /**
     * <p>
     * The HTTP status code for this <a>GatewayResponse</a>.
     * </p>
     * 
     * @return The HTTP status code for this <a>GatewayResponse</a>.
     */
    public String statusCode() {
        return statusCode;
    }

    /**
     * <p>
     * Response parameters (paths, query strings and headers) of the <a>GatewayResponse</a> as a string-to-string map of
     * key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Response parameters (paths, query strings and headers) of the <a>GatewayResponse</a> as a
     *         string-to-string map of key-value pairs.
     */
    public Map<String, String> responseParameters() {
        return responseParameters;
    }

    /**
     * <p>
     * Response templates of the <a>GatewayResponse</a> as a string-to-string map of key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Response templates of the <a>GatewayResponse</a> as a string-to-string map of key-value pairs.
     */
    public Map<String, String> responseTemplates() {
        return responseTemplates;
    }

    /**
     * <p>
     * A Boolean flag to indicate whether this <a>GatewayResponse</a> is the default gateway response (<code>true</code>
     * ) or not (<code>false</code>). A default gateway response is one generated by API Gateway without any
     * customization by an API developer.
     * </p>
     * 
     * @return A Boolean flag to indicate whether this <a>GatewayResponse</a> is the default gateway response (
     *         <code>true</code>) or not (<code>false</code>). A default gateway response is one generated by API
     *         Gateway without any customization by an API developer.
     */
    public Boolean defaultResponse() {
        return defaultResponse;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(responseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(responseParameters());
        hashCode = 31 * hashCode + Objects.hashCode(responseTemplates());
        hashCode = 31 * hashCode + Objects.hashCode(defaultResponse());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateGatewayResponseResponse)) {
            return false;
        }
        UpdateGatewayResponseResponse other = (UpdateGatewayResponseResponse) obj;
        return Objects.equals(responseTypeAsString(), other.responseTypeAsString())
                && Objects.equals(statusCode(), other.statusCode())
                && Objects.equals(responseParameters(), other.responseParameters())
                && Objects.equals(responseTemplates(), other.responseTemplates())
                && Objects.equals(defaultResponse(), other.defaultResponse());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateGatewayResponseResponse").add("ResponseType", responseTypeAsString())
                .add("StatusCode", statusCode()).add("ResponseParameters", responseParameters())
                .add("ResponseTemplates", responseTemplates()).add("DefaultResponse", defaultResponse()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "responseType":
            return Optional.ofNullable(clazz.cast(responseTypeAsString()));
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "responseParameters":
            return Optional.ofNullable(clazz.cast(responseParameters()));
        case "responseTemplates":
            return Optional.ofNullable(clazz.cast(responseTemplates()));
        case "defaultResponse":
            return Optional.ofNullable(clazz.cast(defaultResponse()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends ApiGatewayResponse.Builder, CopyableBuilder<Builder, UpdateGatewayResponseResponse> {
        /**
         * <p>
         * The response type of the associated <a>GatewayResponse</a>. Valid values are
         * <ul>
         * <li>ACCESS_DENIED</li>
         * <li>API_CONFIGURATION_ERROR</li>
         * <li>AUTHORIZER_FAILURE</li>
         * <li>AUTHORIZER_CONFIGURATION_ERROR</li>
         * <li>BAD_REQUEST_PARAMETERS</li>
         * <li>BAD_REQUEST_BODY</li>
         * <li>DEFAULT_4XX</li>
         * <li>DEFAULT_5XX</li>
         * <li>EXPIRED_TOKEN</li>
         * <li>INVALID_SIGNATURE</li>
         * <li>INTEGRATION_FAILURE</li>
         * <li>INTEGRATION_TIMEOUT</li>
         * <li>INVALID_API_KEY</li>
         * <li>MISSING_AUTHENTICATION_TOKEN</li>
         * <li>QUOTA_EXCEEDED</li>
         * <li>REQUEST_TOO_LARGE</li>
         * <li>RESOURCE_NOT_FOUND</li>
         * <li>THROTTLED</li>
         * <li>UNAUTHORIZED</li>
         * <li>UNSUPPORTED_MEDIA_TYPE</li>
         * </ul>
         * </p>
         * 
         * @param responseType
         *        The response type of the associated <a>GatewayResponse</a>. Valid values are
         *        <ul>
         *        <li>ACCESS_DENIED</li>
         *        <li>API_CONFIGURATION_ERROR</li>
         *        <li>AUTHORIZER_FAILURE</li>
         *        <li>AUTHORIZER_CONFIGURATION_ERROR</li>
         *        <li>BAD_REQUEST_PARAMETERS</li>
         *        <li>BAD_REQUEST_BODY</li>
         *        <li>DEFAULT_4XX</li>
         *        <li>DEFAULT_5XX</li>
         *        <li>EXPIRED_TOKEN</li>
         *        <li>INVALID_SIGNATURE</li>
         *        <li>INTEGRATION_FAILURE</li>
         *        <li>INTEGRATION_TIMEOUT</li>
         *        <li>INVALID_API_KEY</li>
         *        <li>MISSING_AUTHENTICATION_TOKEN</li>
         *        <li>QUOTA_EXCEEDED</li>
         *        <li>REQUEST_TOO_LARGE</li>
         *        <li>RESOURCE_NOT_FOUND</li>
         *        <li>THROTTLED</li>
         *        <li>UNAUTHORIZED</li>
         *        <li>UNSUPPORTED_MEDIA_TYPE</li>
         *        </ul>
         * @see GatewayResponseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayResponseType
         */
        Builder responseType(String responseType);

        /**
         * <p>
         * The response type of the associated <a>GatewayResponse</a>. Valid values are
         * <ul>
         * <li>ACCESS_DENIED</li>
         * <li>API_CONFIGURATION_ERROR</li>
         * <li>AUTHORIZER_FAILURE</li>
         * <li>AUTHORIZER_CONFIGURATION_ERROR</li>
         * <li>BAD_REQUEST_PARAMETERS</li>
         * <li>BAD_REQUEST_BODY</li>
         * <li>DEFAULT_4XX</li>
         * <li>DEFAULT_5XX</li>
         * <li>EXPIRED_TOKEN</li>
         * <li>INVALID_SIGNATURE</li>
         * <li>INTEGRATION_FAILURE</li>
         * <li>INTEGRATION_TIMEOUT</li>
         * <li>INVALID_API_KEY</li>
         * <li>MISSING_AUTHENTICATION_TOKEN</li>
         * <li>QUOTA_EXCEEDED</li>
         * <li>REQUEST_TOO_LARGE</li>
         * <li>RESOURCE_NOT_FOUND</li>
         * <li>THROTTLED</li>
         * <li>UNAUTHORIZED</li>
         * <li>UNSUPPORTED_MEDIA_TYPE</li>
         * </ul>
         * </p>
         * 
         * @param responseType
         *        The response type of the associated <a>GatewayResponse</a>. Valid values are
         *        <ul>
         *        <li>ACCESS_DENIED</li>
         *        <li>API_CONFIGURATION_ERROR</li>
         *        <li>AUTHORIZER_FAILURE</li>
         *        <li>AUTHORIZER_CONFIGURATION_ERROR</li>
         *        <li>BAD_REQUEST_PARAMETERS</li>
         *        <li>BAD_REQUEST_BODY</li>
         *        <li>DEFAULT_4XX</li>
         *        <li>DEFAULT_5XX</li>
         *        <li>EXPIRED_TOKEN</li>
         *        <li>INVALID_SIGNATURE</li>
         *        <li>INTEGRATION_FAILURE</li>
         *        <li>INTEGRATION_TIMEOUT</li>
         *        <li>INVALID_API_KEY</li>
         *        <li>MISSING_AUTHENTICATION_TOKEN</li>
         *        <li>QUOTA_EXCEEDED</li>
         *        <li>REQUEST_TOO_LARGE</li>
         *        <li>RESOURCE_NOT_FOUND</li>
         *        <li>THROTTLED</li>
         *        <li>UNAUTHORIZED</li>
         *        <li>UNSUPPORTED_MEDIA_TYPE</li>
         *        </ul>
         * @see GatewayResponseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayResponseType
         */
        Builder responseType(GatewayResponseType responseType);

        /**
         * <p>
         * The HTTP status code for this <a>GatewayResponse</a>.
         * </p>
         * 
         * @param statusCode
         *        The HTTP status code for this <a>GatewayResponse</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * Response parameters (paths, query strings and headers) of the <a>GatewayResponse</a> as a string-to-string
         * map of key-value pairs.
         * </p>
         * 
         * @param responseParameters
         *        Response parameters (paths, query strings and headers) of the <a>GatewayResponse</a> as a
         *        string-to-string map of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseParameters(Map<String, String> responseParameters);

        /**
         * <p>
         * Response templates of the <a>GatewayResponse</a> as a string-to-string map of key-value pairs.
         * </p>
         * 
         * @param responseTemplates
         *        Response templates of the <a>GatewayResponse</a> as a string-to-string map of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseTemplates(Map<String, String> responseTemplates);

        /**
         * <p>
         * A Boolean flag to indicate whether this <a>GatewayResponse</a> is the default gateway response (
         * <code>true</code>) or not (<code>false</code>). A default gateway response is one generated by API Gateway
         * without any customization by an API developer.
         * </p>
         * 
         * @param defaultResponse
         *        A Boolean flag to indicate whether this <a>GatewayResponse</a> is the default gateway response (
         *        <code>true</code>) or not (<code>false</code>). A default gateway response is one generated by API
         *        Gateway without any customization by an API developer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultResponse(Boolean defaultResponse);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String responseType;

        private String statusCode;

        private Map<String, String> responseParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> responseTemplates = DefaultSdkAutoConstructMap.getInstance();

        private Boolean defaultResponse;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateGatewayResponseResponse model) {
            super(model);
            responseType(model.responseType);
            statusCode(model.statusCode);
            responseParameters(model.responseParameters);
            responseTemplates(model.responseTemplates);
            defaultResponse(model.defaultResponse);
        }

        public final String getResponseType() {
            return responseType;
        }

        @Override
        public final Builder responseType(String responseType) {
            this.responseType = responseType;
            return this;
        }

        @Override
        public final Builder responseType(GatewayResponseType responseType) {
            this.responseType(responseType.toString());
            return this;
        }

        public final void setResponseType(String responseType) {
            this.responseType = responseType;
        }

        public final String getStatusCode() {
            return statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        public final Map<String, String> getResponseParameters() {
            return responseParameters;
        }

        @Override
        public final Builder responseParameters(Map<String, String> responseParameters) {
            this.responseParameters = MapOfStringToStringCopier.copy(responseParameters);
            return this;
        }

        public final void setResponseParameters(Map<String, String> responseParameters) {
            this.responseParameters = MapOfStringToStringCopier.copy(responseParameters);
        }

        public final Map<String, String> getResponseTemplates() {
            return responseTemplates;
        }

        @Override
        public final Builder responseTemplates(Map<String, String> responseTemplates) {
            this.responseTemplates = MapOfStringToStringCopier.copy(responseTemplates);
            return this;
        }

        public final void setResponseTemplates(Map<String, String> responseTemplates) {
            this.responseTemplates = MapOfStringToStringCopier.copy(responseTemplates);
        }

        public final Boolean getDefaultResponse() {
            return defaultResponse;
        }

        @Override
        public final Builder defaultResponse(Boolean defaultResponse) {
            this.defaultResponse = defaultResponse;
            return this;
        }

        public final void setDefaultResponse(Boolean defaultResponse) {
            this.defaultResponse = defaultResponse;
        }

        @Override
        public UpdateGatewayResponseResponse build() {
            return new UpdateGatewayResponseResponse(this);
        }
    }
}
