/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A API Gateway VPC link for a <a>RestApi</a> to access resources in an Amazon Virtual Private Cloud (VPC).
 * </p>
 * <div class="remarks">
 * <p>
 * <p>
 * To enable access to a resource in an Amazon Virtual Private Cloud through Amazon API Gateway, you, as an API
 * developer, create a <a>VpcLink</a> resource targeted for one or more network load balancers of the VPC and then
 * integrate an API method with a private integration that uses the <a>VpcLink</a>. The private integration has an
 * integration type of <code>HTTP</code> or <code>HTTP_PROXY</code> and has a connection type of <code>VPC_LINK</code>.
 * The integration uses the <code>connectionId</code> property to identify the <a>VpcLink</a> used.
 * </p>
 * </p> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateVpcLinkResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<UpdateVpcLinkResponse.Builder, UpdateVpcLinkResponse> {
    private final String id;

    private final String name;

    private final String description;

    private final List<String> targetArns;

    private final String status;

    private final String statusMessage;

    private UpdateVpcLinkResponse(BuilderImpl builder) {
        super(builder);
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.targetArns = builder.targetArns;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
    }

    /**
     * <p>
     * The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this <a>VpcLink</a>.
     * </p>
     * 
     * @return The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this
     *         <a>VpcLink</a>.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name used to label and identify the VPC link.
     * </p>
     * 
     * @return The name used to label and identify the VPC link.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The description of the VPC link.
     * </p>
     * 
     * @return The description of the VPC link.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers must be owned
     * by the same AWS account of the API owner.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers must
     *         be owned by the same AWS account of the API owner.
     */
    public List<String> targetArns() {
        return targetArns;
    }

    /**
     * <p>
     * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is <code>PENDING</code>
     * and will fail if the status is <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcLinkStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     *         <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
     *         <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
     * @see VpcLinkStatus
     */
    public VpcLinkStatus status() {
        return VpcLinkStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is <code>PENDING</code>
     * and will fail if the status is <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcLinkStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     *         <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
     *         <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
     * @see VpcLinkStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A description about the VPC link status.
     * </p>
     * 
     * @return A description about the VPC link status.
     */
    public String statusMessage() {
        return statusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(targetArns());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateVpcLinkResponse)) {
            return false;
        }
        UpdateVpcLinkResponse other = (UpdateVpcLinkResponse) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && Objects.equals(targetArns(), other.targetArns())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage());
    }

    @Override
    public String toString() {
        return ToString.builder("UpdateVpcLinkResponse").add("Id", id()).add("Name", name()).add("Description", description())
                .add("TargetArns", targetArns()).add("Status", statusAsString()).add("StatusMessage", statusMessage()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "targetArns":
            return Optional.ofNullable(clazz.cast(targetArns()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends ApiGatewayResponse.Builder, CopyableBuilder<Builder, UpdateVpcLinkResponse> {
        /**
         * <p>
         * The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this <a>VpcLink</a>.
         * </p>
         * 
         * @param id
         *        The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this
         *        <a>VpcLink</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name used to label and identify the VPC link.
         * </p>
         * 
         * @param name
         *        The name used to label and identify the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the VPC link.
         * </p>
         * 
         * @param description
         *        The description of the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers must be
         * owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers
         *        must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(Collection<String> targetArns);

        /**
         * <p>
         * The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers must be
         * owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers
         *        must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(String... targetArns);

        /**
         * <p>
         * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         * <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         *        <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         *        <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * @see VpcLinkStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcLinkStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         * <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         *        <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         *        <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * @see VpcLinkStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcLinkStatus
         */
        Builder status(VpcLinkStatus status);

        /**
         * <p>
         * A description about the VPC link status.
         * </p>
         * 
         * @param statusMessage
         *        A description about the VPC link status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String id;

        private String name;

        private String description;

        private List<String> targetArns = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String statusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateVpcLinkResponse model) {
            super(model);
            id(model.id);
            name(model.name);
            description(model.description);
            targetArns(model.targetArns);
            status(model.status);
            statusMessage(model.statusMessage);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getTargetArns() {
            return targetArns;
        }

        @Override
        public final Builder targetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetArns(String... targetArns) {
            targetArns(Arrays.asList(targetArns));
            return this;
        }

        public final void setTargetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(VpcLinkStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public UpdateVpcLinkResponse build() {
            return new UpdateVpcLinkResponse(this);
        }
    }
}
