/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.PayloadTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The POST request to import API keys from an external source, such as a CSV-formatted file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportApiKeysRequest extends ApiGatewayRequest implements
        ToCopyableBuilder<ImportApiKeysRequest.Builder, ImportApiKeysRequest> {
    private static final SdkField<SdkBytes> BODY_FIELD = SdkField
            .<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(ImportApiKeysRequest::body))
            .setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("body").build(),
                    PayloadTrait.create()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ImportApiKeysRequest::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("format").build()).build();

    private static final SdkField<Boolean> FAIL_ON_WARNINGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ImportApiKeysRequest::failOnWarnings)).setter(setter(Builder::failOnWarnings))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("failonwarnings").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BODY_FIELD, FORMAT_FIELD,
            FAIL_ON_WARNINGS_FIELD));

    private final SdkBytes body;

    private final String format;

    private final Boolean failOnWarnings;

    private ImportApiKeysRequest(BuilderImpl builder) {
        super(builder);
        this.body = builder.body;
        this.format = builder.format;
        this.failOnWarnings = builder.failOnWarnings;
    }

    /**
     * <p>
     * The payload of the POST request to import API keys. For the payload format, see <a
     * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/api-key-file-format.html">API Key File
     * Format</a>.
     * </p>
     * 
     * @return The payload of the POST request to import API keys. For the payload format, see <a
     *         href="http://docs.aws.amazon.com/apigateway/latest/developerguide/api-key-file-format.html">API Key File
     *         Format</a>.
     */
    public SdkBytes body() {
        return body;
    }

    /**
     * <p>
     * A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code> format
     * is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link ApiKeysFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code>
     *         format is supported.
     * @see ApiKeysFormat
     */
    public ApiKeysFormat format() {
        return ApiKeysFormat.fromValue(format);
    }

    /**
     * <p>
     * A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code> format
     * is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link ApiKeysFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code>
     *         format is supported.
     * @see ApiKeysFormat
     */
    public String formatAsString() {
        return format;
    }

    /**
     * <p>
     * A query parameter to indicate whether to rollback <a>ApiKey</a> importation (<code>true</code>) or not (
     * <code>false</code>) when error is encountered.
     * </p>
     * 
     * @return A query parameter to indicate whether to rollback <a>ApiKey</a> importation (<code>true</code>) or not (
     *         <code>false</code>) when error is encountered.
     */
    public Boolean failOnWarnings() {
        return failOnWarnings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(body());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failOnWarnings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportApiKeysRequest)) {
            return false;
        }
        ImportApiKeysRequest other = (ImportApiKeysRequest) obj;
        return Objects.equals(body(), other.body()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(failOnWarnings(), other.failOnWarnings());
    }

    @Override
    public String toString() {
        return ToString.builder("ImportApiKeysRequest").add("Body", body()).add("Format", formatAsString())
                .add("FailOnWarnings", failOnWarnings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "body":
            return Optional.ofNullable(clazz.cast(body()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "failOnWarnings":
            return Optional.ofNullable(clazz.cast(failOnWarnings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImportApiKeysRequest, T> g) {
        return obj -> g.apply((ImportApiKeysRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayRequest.Builder, SdkPojo, CopyableBuilder<Builder, ImportApiKeysRequest> {
        /**
         * <p>
         * The payload of the POST request to import API keys. For the payload format, see <a
         * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/api-key-file-format.html">API Key File
         * Format</a>.
         * </p>
         * 
         * @param body
         *        The payload of the POST request to import API keys. For the payload format, see <a
         *        href="http://docs.aws.amazon.com/apigateway/latest/developerguide/api-key-file-format.html">API Key
         *        File Format</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(SdkBytes body);

        /**
         * <p>
         * A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code>
         * format is supported.
         * </p>
         * 
         * @param format
         *        A query parameter to specify the input format to imported API keys. Currently, only the
         *        <code>csv</code> format is supported.
         * @see ApiKeysFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApiKeysFormat
         */
        Builder format(String format);

        /**
         * <p>
         * A query parameter to specify the input format to imported API keys. Currently, only the <code>csv</code>
         * format is supported.
         * </p>
         * 
         * @param format
         *        A query parameter to specify the input format to imported API keys. Currently, only the
         *        <code>csv</code> format is supported.
         * @see ApiKeysFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ApiKeysFormat
         */
        Builder format(ApiKeysFormat format);

        /**
         * <p>
         * A query parameter to indicate whether to rollback <a>ApiKey</a> importation (<code>true</code>) or not (
         * <code>false</code>) when error is encountered.
         * </p>
         * 
         * @param failOnWarnings
         *        A query parameter to indicate whether to rollback <a>ApiKey</a> importation (<code>true</code>) or not
         *        (<code>false</code>) when error is encountered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failOnWarnings(Boolean failOnWarnings);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayRequest.BuilderImpl implements Builder {
        private SdkBytes body;

        private String format;

        private Boolean failOnWarnings;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportApiKeysRequest model) {
            super(model);
            body(model.body);
            format(model.format);
            failOnWarnings(model.failOnWarnings);
        }

        public final ByteBuffer getBody() {
            return body == null ? null : body.asByteBuffer();
        }

        @Override
        public final Builder body(SdkBytes body) {
            this.body = StandardMemberCopier.copy(body);
            return this;
        }

        public final void setBody(ByteBuffer body) {
            body(body == null ? null : SdkBytes.fromByteBuffer(body));
        }

        public final String getFormat() {
            return format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(ApiKeysFormat format) {
            this.format(format.toString());
            return this;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        public final Boolean getFailOnWarnings() {
            return failOnWarnings;
        }

        @Override
        public final Builder failOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
            return this;
        }

        public final void setFailOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ImportApiKeysRequest build() {
            return new ImportApiKeysRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
