/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an integration response. The status code must map to an existing <a>MethodResponse</a>, and parameters and
 * templates can be used to transform the back-end response.
 * </p>
 * <div class="seeAlso"> <a
 * href="http://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an API</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutIntegrationResponseResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<PutIntegrationResponseResponse.Builder, PutIntegrationResponseResponse> {
    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutIntegrationResponseResponse::statusCode)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<String> SELECTION_PATTERN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutIntegrationResponseResponse::selectionPattern)).setter(setter(Builder::selectionPattern))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("selectionPattern").build()).build();

    private static final SdkField<Map<String, String>> RESPONSE_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(PutIntegrationResponseResponse::responseParameters))
            .setter(setter(Builder::responseParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> RESPONSE_TEMPLATES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(PutIntegrationResponseResponse::responseTemplates))
            .setter(setter(Builder::responseTemplates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseTemplates").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CONTENT_HANDLING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutIntegrationResponseResponse::contentHandlingAsString)).setter(setter(Builder::contentHandling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contentHandling").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            SELECTION_PATTERN_FIELD, RESPONSE_PARAMETERS_FIELD, RESPONSE_TEMPLATES_FIELD, CONTENT_HANDLING_FIELD));

    private final String statusCode;

    private final String selectionPattern;

    private final Map<String, String> responseParameters;

    private final Map<String, String> responseTemplates;

    private final String contentHandling;

    private PutIntegrationResponseResponse(BuilderImpl builder) {
        super(builder);
        this.statusCode = builder.statusCode;
        this.selectionPattern = builder.selectionPattern;
        this.responseParameters = builder.responseParameters;
        this.responseTemplates = builder.responseTemplates;
        this.contentHandling = builder.contentHandling;
    }

    /**
     * <p>
     * Specifies the status code that is used to map the integration response to an existing <a>MethodResponse</a>.
     * </p>
     * 
     * @return Specifies the status code that is used to map the integration response to an existing
     *         <a>MethodResponse</a>.
     */
    public String statusCode() {
        return statusCode;
    }

    /**
     * <p>
     * Specifies the regular expression (regex) pattern used to choose an integration response based on the response
     * from the back end. For example, if the success response returns nothing and the error response returns some
     * string, you could use the <code>.+</code> regex to match error response. However, make sure that the error
     * response does not contain any newline (<code>\n</code>) character in such cases. If the back end is an AWS Lambda
     * function, the AWS Lambda function error header is matched. For all other HTTP and AWS back ends, the HTTP status
     * code is matched.
     * </p>
     * 
     * @return Specifies the regular expression (regex) pattern used to choose an integration response based on the
     *         response from the back end. For example, if the success response returns nothing and the error response
     *         returns some string, you could use the <code>.+</code> regex to match error response. However, make sure
     *         that the error response does not contain any newline (<code>\n</code>) character in such cases. If the
     *         back end is an AWS Lambda function, the AWS Lambda function error header is matched. For all other HTTP
     *         and AWS back ends, the HTTP status code is matched.
     */
    public String selectionPattern() {
        return selectionPattern;
    }

    /**
     * <p>
     * A key-value map specifying response parameters that are passed to the method response from the back end. The key
     * is a method response header parameter name and the mapped value is an integration response header value, a static
     * value enclosed within a pair of single quotes, or a JSON expression from the integration response body. The
     * mapping key must match the pattern of <code>method.response.header.{name}</code>, where <code>name</code> is a
     * valid and unique header name. The mapped non-static value must match the pattern of
     * <code>integration.response.header.{name}</code> or <code>integration.response.body.{JSON-expression}</code>,
     * where <code>name</code> is a valid and unique response header name and <code>JSON-expression</code> is a valid
     * JSON expression without the <code>$</code> prefix.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A key-value map specifying response parameters that are passed to the method response from the back end.
     *         The key is a method response header parameter name and the mapped value is an integration response header
     *         value, a static value enclosed within a pair of single quotes, or a JSON expression from the integration
     *         response body. The mapping key must match the pattern of <code>method.response.header.{name}</code>,
     *         where <code>name</code> is a valid and unique header name. The mapped non-static value must match the
     *         pattern of <code>integration.response.header.{name}</code> or
     *         <code>integration.response.body.{JSON-expression}</code>, where <code>name</code> is a valid and unique
     *         response header name and <code>JSON-expression</code> is a valid JSON expression without the
     *         <code>$</code> prefix.
     */
    public Map<String, String> responseParameters() {
        return responseParameters;
    }

    /**
     * <p>
     * Specifies the templates used to transform the integration response body. Response templates are represented as a
     * key/value map, with a content-type as the key and a template as the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the templates used to transform the integration response body. Response templates are
     *         represented as a key/value map, with a content-type as the key and a template as the value.
     */
    public Map<String, String> responseTemplates() {
        return responseTemplates;
    }

    /**
     * <p>
     * Specifies how to handle response payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the corresponding
     * binary blob.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If this property is not defined, the response payload will be passed through from the integration response to the
     * method response without modification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle response payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the
     *         corresponding binary blob.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If this property is not defined, the response payload will be passed through from the integration
     *         response to the method response without modification.
     * @see ContentHandlingStrategy
     */
    public ContentHandlingStrategy contentHandling() {
        return ContentHandlingStrategy.fromValue(contentHandling);
    }

    /**
     * <p>
     * Specifies how to handle response payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the corresponding
     * binary blob.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If this property is not defined, the response payload will be passed through from the integration response to the
     * method response without modification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle response payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the
     *         corresponding binary blob.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If this property is not defined, the response payload will be passed through from the integration
     *         response to the method response without modification.
     * @see ContentHandlingStrategy
     */
    public String contentHandlingAsString() {
        return contentHandling;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(selectionPattern());
        hashCode = 31 * hashCode + Objects.hashCode(responseParameters());
        hashCode = 31 * hashCode + Objects.hashCode(responseTemplates());
        hashCode = 31 * hashCode + Objects.hashCode(contentHandlingAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutIntegrationResponseResponse)) {
            return false;
        }
        PutIntegrationResponseResponse other = (PutIntegrationResponseResponse) obj;
        return Objects.equals(statusCode(), other.statusCode()) && Objects.equals(selectionPattern(), other.selectionPattern())
                && Objects.equals(responseParameters(), other.responseParameters())
                && Objects.equals(responseTemplates(), other.responseTemplates())
                && Objects.equals(contentHandlingAsString(), other.contentHandlingAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("PutIntegrationResponseResponse").add("StatusCode", statusCode())
                .add("SelectionPattern", selectionPattern()).add("ResponseParameters", responseParameters())
                .add("ResponseTemplates", responseTemplates()).add("ContentHandling", contentHandlingAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "selectionPattern":
            return Optional.ofNullable(clazz.cast(selectionPattern()));
        case "responseParameters":
            return Optional.ofNullable(clazz.cast(responseParameters()));
        case "responseTemplates":
            return Optional.ofNullable(clazz.cast(responseTemplates()));
        case "contentHandling":
            return Optional.ofNullable(clazz.cast(contentHandlingAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutIntegrationResponseResponse, T> g) {
        return obj -> g.apply((PutIntegrationResponseResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo,
            CopyableBuilder<Builder, PutIntegrationResponseResponse> {
        /**
         * <p>
         * Specifies the status code that is used to map the integration response to an existing <a>MethodResponse</a>.
         * </p>
         * 
         * @param statusCode
         *        Specifies the status code that is used to map the integration response to an existing
         *        <a>MethodResponse</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * Specifies the regular expression (regex) pattern used to choose an integration response based on the response
         * from the back end. For example, if the success response returns nothing and the error response returns some
         * string, you could use the <code>.+</code> regex to match error response. However, make sure that the error
         * response does not contain any newline (<code>\n</code>) character in such cases. If the back end is an AWS
         * Lambda function, the AWS Lambda function error header is matched. For all other HTTP and AWS back ends, the
         * HTTP status code is matched.
         * </p>
         * 
         * @param selectionPattern
         *        Specifies the regular expression (regex) pattern used to choose an integration response based on the
         *        response from the back end. For example, if the success response returns nothing and the error
         *        response returns some string, you could use the <code>.+</code> regex to match error response.
         *        However, make sure that the error response does not contain any newline (<code>\n</code>) character in
         *        such cases. If the back end is an AWS Lambda function, the AWS Lambda function error header is
         *        matched. For all other HTTP and AWS back ends, the HTTP status code is matched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectionPattern(String selectionPattern);

        /**
         * <p>
         * A key-value map specifying response parameters that are passed to the method response from the back end. The
         * key is a method response header parameter name and the mapped value is an integration response header value,
         * a static value enclosed within a pair of single quotes, or a JSON expression from the integration response
         * body. The mapping key must match the pattern of <code>method.response.header.{name}</code>, where
         * <code>name</code> is a valid and unique header name. The mapped non-static value must match the pattern of
         * <code>integration.response.header.{name}</code> or <code>integration.response.body.{JSON-expression}</code>,
         * where <code>name</code> is a valid and unique response header name and <code>JSON-expression</code> is a
         * valid JSON expression without the <code>$</code> prefix.
         * </p>
         * 
         * @param responseParameters
         *        A key-value map specifying response parameters that are passed to the method response from the back
         *        end. The key is a method response header parameter name and the mapped value is an integration
         *        response header value, a static value enclosed within a pair of single quotes, or a JSON expression
         *        from the integration response body. The mapping key must match the pattern of
         *        <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name.
         *        The mapped non-static value must match the pattern of <code>integration.response.header.{name}</code>
         *        or <code>integration.response.body.{JSON-expression}</code>, where <code>name</code> is a valid and
         *        unique response header name and <code>JSON-expression</code> is a valid JSON expression without the
         *        <code>$</code> prefix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseParameters(Map<String, String> responseParameters);

        /**
         * <p>
         * Specifies the templates used to transform the integration response body. Response templates are represented
         * as a key/value map, with a content-type as the key and a template as the value.
         * </p>
         * 
         * @param responseTemplates
         *        Specifies the templates used to transform the integration response body. Response templates are
         *        represented as a key/value map, with a content-type as the key and a template as the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseTemplates(Map<String, String> responseTemplates);

        /**
         * <p>
         * Specifies how to handle response payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the corresponding
         * binary blob.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If this property is not defined, the response payload will be passed through from the integration response to
         * the method response without modification.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle response payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the
         *        corresponding binary blob.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded
         *        string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If this property is not defined, the response payload will be passed through from the integration
         *        response to the method response without modification.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(String contentHandling);

        /**
         * <p>
         * Specifies how to handle response payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the corresponding
         * binary blob.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded string.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If this property is not defined, the response payload will be passed through from the integration response to
         * the method response without modification.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle response payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_BINARY</code>: Converts a response payload from a Base64-encoded string to the
         *        corresponding binary blob.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONVERT_TO_TEXT</code>: Converts a response payload from a binary blob to a Base64-encoded
         *        string.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If this property is not defined, the response payload will be passed through from the integration
         *        response to the method response without modification.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(ContentHandlingStrategy contentHandling);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String statusCode;

        private String selectionPattern;

        private Map<String, String> responseParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> responseTemplates = DefaultSdkAutoConstructMap.getInstance();

        private String contentHandling;

        private BuilderImpl() {
        }

        private BuilderImpl(PutIntegrationResponseResponse model) {
            super(model);
            statusCode(model.statusCode);
            selectionPattern(model.selectionPattern);
            responseParameters(model.responseParameters);
            responseTemplates(model.responseTemplates);
            contentHandling(model.contentHandling);
        }

        public final String getStatusCode() {
            return statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        public final String getSelectionPattern() {
            return selectionPattern;
        }

        @Override
        public final Builder selectionPattern(String selectionPattern) {
            this.selectionPattern = selectionPattern;
            return this;
        }

        public final void setSelectionPattern(String selectionPattern) {
            this.selectionPattern = selectionPattern;
        }

        public final Map<String, String> getResponseParameters() {
            return responseParameters;
        }

        @Override
        public final Builder responseParameters(Map<String, String> responseParameters) {
            this.responseParameters = MapOfStringToStringCopier.copy(responseParameters);
            return this;
        }

        public final void setResponseParameters(Map<String, String> responseParameters) {
            this.responseParameters = MapOfStringToStringCopier.copy(responseParameters);
        }

        public final Map<String, String> getResponseTemplates() {
            return responseTemplates;
        }

        @Override
        public final Builder responseTemplates(Map<String, String> responseTemplates) {
            this.responseTemplates = MapOfStringToStringCopier.copy(responseTemplates);
            return this;
        }

        public final void setResponseTemplates(Map<String, String> responseTemplates) {
            this.responseTemplates = MapOfStringToStringCopier.copy(responseTemplates);
        }

        public final String getContentHandling() {
            return contentHandling;
        }

        @Override
        public final Builder contentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
            return this;
        }

        @Override
        public final Builder contentHandling(ContentHandlingStrategy contentHandling) {
            this.contentHandling(contentHandling.toString());
            return this;
        }

        public final void setContentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
        }

        @Override
        public PutIntegrationResponseResponse build() {
            return new PutIntegrationResponseResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
