/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input configuration for a canary deployment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentCanarySettings implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentCanarySettings.Builder, DeploymentCanarySettings> {
    private static final SdkField<Double> PERCENT_TRAFFIC_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(DeploymentCanarySettings::percentTraffic)).setter(setter(Builder::percentTraffic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("percentTraffic").build()).build();

    private static final SdkField<Map<String, String>> STAGE_VARIABLE_OVERRIDES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(DeploymentCanarySettings::stageVariableOverrides))
            .setter(setter(Builder::stageVariableOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stageVariableOverrides").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> USE_STAGE_CACHE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(DeploymentCanarySettings::useStageCache)).setter(setter(Builder::useStageCache))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("useStageCache").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PERCENT_TRAFFIC_FIELD,
            STAGE_VARIABLE_OVERRIDES_FIELD, USE_STAGE_CACHE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Double percentTraffic;

    private final Map<String, String> stageVariableOverrides;

    private final Boolean useStageCache;

    private DeploymentCanarySettings(BuilderImpl builder) {
        this.percentTraffic = builder.percentTraffic;
        this.stageVariableOverrides = builder.stageVariableOverrides;
        this.useStageCache = builder.useStageCache;
    }

    /**
     * <p>
     * The percentage (0.0-100.0) of traffic routed to the canary deployment.
     * </p>
     * 
     * @return The percentage (0.0-100.0) of traffic routed to the canary deployment.
     */
    public Double percentTraffic() {
        return percentTraffic;
    }

    /**
     * Returns true if the StageVariableOverrides property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasStageVariableOverrides() {
        return stageVariableOverrides != null && !(stageVariableOverrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A stage variable overrides used for the canary release deployment. They can override existing stage variables or
     * add new stage variables for the canary release deployment. These stage variables are represented as a
     * string-to-string map between stage variable names and their values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStageVariableOverrides()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A stage variable overrides used for the canary release deployment. They can override existing stage
     *         variables or add new stage variables for the canary release deployment. These stage variables are
     *         represented as a string-to-string map between stage variable names and their values.
     */
    public Map<String, String> stageVariableOverrides() {
        return stageVariableOverrides;
    }

    /**
     * <p>
     * A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.
     * </p>
     * 
     * @return A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.
     */
    public Boolean useStageCache() {
        return useStageCache;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(percentTraffic());
        hashCode = 31 * hashCode + Objects.hashCode(stageVariableOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(useStageCache());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentCanarySettings)) {
            return false;
        }
        DeploymentCanarySettings other = (DeploymentCanarySettings) obj;
        return Objects.equals(percentTraffic(), other.percentTraffic())
                && Objects.equals(stageVariableOverrides(), other.stageVariableOverrides())
                && Objects.equals(useStageCache(), other.useStageCache());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DeploymentCanarySettings").add("PercentTraffic", percentTraffic())
                .add("StageVariableOverrides", stageVariableOverrides()).add("UseStageCache", useStageCache()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "percentTraffic":
            return Optional.ofNullable(clazz.cast(percentTraffic()));
        case "stageVariableOverrides":
            return Optional.ofNullable(clazz.cast(stageVariableOverrides()));
        case "useStageCache":
            return Optional.ofNullable(clazz.cast(useStageCache()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentCanarySettings, T> g) {
        return obj -> g.apply((DeploymentCanarySettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentCanarySettings> {
        /**
         * <p>
         * The percentage (0.0-100.0) of traffic routed to the canary deployment.
         * </p>
         * 
         * @param percentTraffic
         *        The percentage (0.0-100.0) of traffic routed to the canary deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentTraffic(Double percentTraffic);

        /**
         * <p>
         * A stage variable overrides used for the canary release deployment. They can override existing stage variables
         * or add new stage variables for the canary release deployment. These stage variables are represented as a
         * string-to-string map between stage variable names and their values.
         * </p>
         * 
         * @param stageVariableOverrides
         *        A stage variable overrides used for the canary release deployment. They can override existing stage
         *        variables or add new stage variables for the canary release deployment. These stage variables are
         *        represented as a string-to-string map between stage variable names and their values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stageVariableOverrides(Map<String, String> stageVariableOverrides);

        /**
         * <p>
         * A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.
         * </p>
         * 
         * @param useStageCache
         *        A Boolean flag to indicate whether the canary release deployment uses the stage cache or not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useStageCache(Boolean useStageCache);
    }

    static final class BuilderImpl implements Builder {
        private Double percentTraffic;

        private Map<String, String> stageVariableOverrides = DefaultSdkAutoConstructMap.getInstance();

        private Boolean useStageCache;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentCanarySettings model) {
            percentTraffic(model.percentTraffic);
            stageVariableOverrides(model.stageVariableOverrides);
            useStageCache(model.useStageCache);
        }

        public final Double getPercentTraffic() {
            return percentTraffic;
        }

        @Override
        public final Builder percentTraffic(Double percentTraffic) {
            this.percentTraffic = percentTraffic;
            return this;
        }

        public final void setPercentTraffic(Double percentTraffic) {
            this.percentTraffic = percentTraffic;
        }

        public final Map<String, String> getStageVariableOverrides() {
            return stageVariableOverrides;
        }

        @Override
        public final Builder stageVariableOverrides(Map<String, String> stageVariableOverrides) {
            this.stageVariableOverrides = MapOfStringToStringCopier.copy(stageVariableOverrides);
            return this;
        }

        public final void setStageVariableOverrides(Map<String, String> stageVariableOverrides) {
            this.stageVariableOverrides = MapOfStringToStringCopier.copy(stageVariableOverrides);
        }

        public final Boolean getUseStageCache() {
            return useStageCache;
        }

        @Override
        public final Builder useStageCache(Boolean useStageCache) {
            this.useStageCache = useStageCache;
            return this;
        }

        public final void setUseStageCache(Boolean useStageCache) {
            this.useStageCache = useStageCache;
        }

        @Override
        public DeploymentCanarySettings build() {
            return new DeploymentCanarySettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
