/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an authorization layer for methods. If enabled on a method, API Gateway will activate the authorizer when
 * a client calls the method.
 * </p>
 * <div class="seeAlso"> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html">Use Lambda
 * Function as Authorizer</a> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html">Use
 * Cognito User Pool as Authorizer</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetAuthorizerResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<GetAuthorizerResponse.Builder, GetAuthorizerResponse> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<List<String>> PROVIDER_AR_NS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GetAuthorizerResponse::providerARNs))
            .setter(setter(Builder::providerARNs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("providerARNs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AUTH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::authType)).setter(setter(Builder::authType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authType").build()).build();

    private static final SdkField<String> AUTHORIZER_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::authorizerUri)).setter(setter(Builder::authorizerUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerUri").build()).build();

    private static final SdkField<String> AUTHORIZER_CREDENTIALS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::authorizerCredentials)).setter(setter(Builder::authorizerCredentials))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerCredentials").build())
            .build();

    private static final SdkField<String> IDENTITY_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::identitySource)).setter(setter(Builder::identitySource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identitySource").build()).build();

    private static final SdkField<String> IDENTITY_VALIDATION_EXPRESSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(GetAuthorizerResponse::identityValidationExpression))
            .setter(setter(Builder::identityValidationExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identityValidationExpression")
                    .build()).build();

    private static final SdkField<Integer> AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(GetAuthorizerResponse::authorizerResultTtlInSeconds))
            .setter(setter(Builder::authorizerResultTtlInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerResultTtlInSeconds")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            TYPE_FIELD, PROVIDER_AR_NS_FIELD, AUTH_TYPE_FIELD, AUTHORIZER_URI_FIELD, AUTHORIZER_CREDENTIALS_FIELD,
            IDENTITY_SOURCE_FIELD, IDENTITY_VALIDATION_EXPRESSION_FIELD, AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD));

    private final String id;

    private final String name;

    private final String type;

    private final List<String> providerARNs;

    private final String authType;

    private final String authorizerUri;

    private final String authorizerCredentials;

    private final String identitySource;

    private final String identityValidationExpression;

    private final Integer authorizerResultTtlInSeconds;

    private GetAuthorizerResponse(BuilderImpl builder) {
        super(builder);
        this.id = builder.id;
        this.name = builder.name;
        this.type = builder.type;
        this.providerARNs = builder.providerARNs;
        this.authType = builder.authType;
        this.authorizerUri = builder.authorizerUri;
        this.authorizerCredentials = builder.authorizerCredentials;
        this.identitySource = builder.identitySource;
        this.identityValidationExpression = builder.identityValidationExpression;
        this.authorizerResultTtlInSeconds = builder.authorizerResultTtlInSeconds;
    }

    /**
     * <p>
     * The identifier for the authorizer resource.
     * </p>
     * 
     * @return The identifier for the authorizer resource.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * [Required] The name of the authorizer.
     * </p>
     * 
     * @return [Required] The name of the authorizer.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single authorization token
     * submitted in a custom header, <code>REQUEST</code> for a Lambda function using incoming request parameters, and
     * <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single
     *         authorization token submitted in a custom header, <code>REQUEST</code> for a Lambda function using
     *         incoming request parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
     * @see AuthorizerType
     */
    public AuthorizerType type() {
        return AuthorizerType.fromValue(type);
    }

    /**
     * <p>
     * The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single authorization token
     * submitted in a custom header, <code>REQUEST</code> for a Lambda function using incoming request parameters, and
     * <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single
     *         authorization token submitted in a custom header, <code>REQUEST</code> for a Lambda function using
     *         incoming request parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
     * @see AuthorizerType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * Returns true if the ProviderARNs property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasProviderARNs() {
        return providerARNs != null && !(providerARNs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each element is
     * of this format: <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>. For a
     * <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProviderARNs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each
     *         element is of this format: <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>
     *         . For a <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
     */
    public List<String> providerARNs() {
        return providerARNs;
    }

    /**
     * <p>
     * Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
     * </p>
     * 
     * @return Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
     */
    public String authType() {
        return authType;
    }

    /**
     * <p>
     * Specifies the authorizer's Uniform Resource Identifier (URI). For <code>TOKEN</code> or <code>REQUEST</code>
     * authorizers, this must be a well-formed Lambda function URI, for example,
     * <code>arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations</code>
     * . In general, the URI has this form <code>arn:aws:apigateway:{region}:lambda:path/{service_api}</code>, where
     * <code>{region}</code> is the same as the region hosting the Lambda function, <code>path</code> indicates that the
     * remaining substring in the URI should be treated as the path to the resource, including the initial
     * <code>/</code>. For Lambda functions, this is usually of the form
     * <code>/2015-03-31/functions/[FunctionARN]/invocations</code>.
     * </p>
     * 
     * @return Specifies the authorizer's Uniform Resource Identifier (URI). For <code>TOKEN</code> or
     *         <code>REQUEST</code> authorizers, this must be a well-formed Lambda function URI, for example,
     *         <code>arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations</code>
     *         . In general, the URI has this form <code>arn:aws:apigateway:{region}:lambda:path/{service_api}</code>,
     *         where <code>{region}</code> is the same as the region hosting the Lambda function, <code>path</code>
     *         indicates that the remaining substring in the URI should be treated as the path to the resource,
     *         including the initial <code>/</code>. For Lambda functions, this is usually of the form
     *         <code>/2015-03-31/functions/[FunctionARN]/invocations</code>.
     */
    public String authorizerUri() {
        return authorizerUri;
    }

    /**
     * <p>
     * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
     * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on
     * the Lambda function, specify null.
     * </p>
     * 
     * @return Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an
     *         IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based
     *         permissions on the Lambda function, specify null.
     */
    public String authorizerCredentials() {
        return authorizerCredentials;
    }

    /**
     * <p>
     * The identity source for which authorization is requested.
     * <ul>
     * <li>For a <code>TOKEN</code> or <code>COGNITO_USER_POOLS</code> authorizer, this is required and specifies the
     * request header mapping expression for the custom header holding the authorization token submitted by the client.
     * For example, if the token header name is <code>Auth</code>, the header mapping expression is
     * <code>method.request.header.Auth</code>.</li>
     * <li>For the <code>REQUEST</code> authorizer, this is required when authorization caching is enabled. The value is
     * a comma-separated string of one or more mapping expressions of the specified request parameters. For example, if
     * an <code>Auth</code> header, a <code>Name</code> query string parameter are defined as identity sources, this
     * value is <code>method.request.header.Auth, method.request.querystring.Name</code>. These parameters will be used
     * to derive the authorization caching key and to perform runtime validation of the <code>REQUEST</code> authorizer
     * by verifying all of the identity-related request parameters are present, not null and non-empty. Only when this
     * is true does the authorizer invoke the authorizer Lambda function, otherwise, it returns a 401 Unauthorized
     * response without calling the Lambda function. The valid value is a string of comma-separated mapping expressions
     * of the specified request parameters. When the authorization caching is not enabled, this property is optional.</li>
     * </ul>
     * </p>
     * 
     * @return The identity source for which authorization is requested.
     *         <ul>
     *         <li>For a <code>TOKEN</code> or <code>COGNITO_USER_POOLS</code> authorizer, this is required and
     *         specifies the request header mapping expression for the custom header holding the authorization token
     *         submitted by the client. For example, if the token header name is <code>Auth</code>, the header mapping
     *         expression is <code>method.request.header.Auth</code>.</li>
     *         <li>For the <code>REQUEST</code> authorizer, this is required when authorization caching is enabled. The
     *         value is a comma-separated string of one or more mapping expressions of the specified request parameters.
     *         For example, if an <code>Auth</code> header, a <code>Name</code> query string parameter are defined as
     *         identity sources, this value is <code>method.request.header.Auth, method.request.querystring.Name</code>.
     *         These parameters will be used to derive the authorization caching key and to perform runtime validation
     *         of the <code>REQUEST</code> authorizer by verifying all of the identity-related request parameters are
     *         present, not null and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda
     *         function, otherwise, it returns a 401 Unauthorized response without calling the Lambda function. The
     *         valid value is a string of comma-separated mapping expressions of the specified request parameters. When
     *         the authorization caching is not enabled, this property is optional.</li>
     *         </ul>
     */
    public String identitySource() {
        return identitySource;
    }

    /**
     * <p>
     * A validation expression for the incoming identity token. For <code>TOKEN</code> authorizers, this value is a
     * regular expression. For <code>COGNITO_USER_POOLS</code> authorizers, API Gateway will match the <code>aud</code>
     * field of the incoming token from the client against the specified regular expression. It will invoke the
     * authorizer's Lambda function when there is a match. Otherwise, it will return a 401 Unauthorized response without
     * calling the Lambda function. The validation expression does not apply to the <code>REQUEST</code> authorizer.
     * </p>
     * 
     * @return A validation expression for the incoming identity token. For <code>TOKEN</code> authorizers, this value
     *         is a regular expression. For <code>COGNITO_USER_POOLS</code> authorizers, API Gateway will match the
     *         <code>aud</code> field of the incoming token from the client against the specified regular expression. It
     *         will invoke the authorizer's Lambda function when there is a match. Otherwise, it will return a 401
     *         Unauthorized response without calling the Lambda function. The validation expression does not apply to
     *         the <code>REQUEST</code> authorizer.
     */
    public String identityValidationExpression() {
        return identityValidationExpression;
    }

    /**
     * <p>
     * The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If it is
     * greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default value is 300.
     * The maximum value is 3600, or 1 hour.
     * </p>
     * 
     * @return The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If it
     *         is greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default
     *         value is 300. The maximum value is 3600, or 1 hour.
     */
    public Integer authorizerResultTtlInSeconds() {
        return authorizerResultTtlInSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(providerARNs());
        hashCode = 31 * hashCode + Objects.hashCode(authType());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerUri());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(identitySource());
        hashCode = 31 * hashCode + Objects.hashCode(identityValidationExpression());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerResultTtlInSeconds());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetAuthorizerResponse)) {
            return false;
        }
        GetAuthorizerResponse other = (GetAuthorizerResponse) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(providerARNs(), other.providerARNs())
                && Objects.equals(authType(), other.authType()) && Objects.equals(authorizerUri(), other.authorizerUri())
                && Objects.equals(authorizerCredentials(), other.authorizerCredentials())
                && Objects.equals(identitySource(), other.identitySource())
                && Objects.equals(identityValidationExpression(), other.identityValidationExpression())
                && Objects.equals(authorizerResultTtlInSeconds(), other.authorizerResultTtlInSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetAuthorizerResponse").add("Id", id()).add("Name", name()).add("Type", typeAsString())
                .add("ProviderARNs", providerARNs()).add("AuthType", authType()).add("AuthorizerUri", authorizerUri())
                .add("AuthorizerCredentials", authorizerCredentials()).add("IdentitySource", identitySource())
                .add("IdentityValidationExpression", identityValidationExpression())
                .add("AuthorizerResultTtlInSeconds", authorizerResultTtlInSeconds()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "providerARNs":
            return Optional.ofNullable(clazz.cast(providerARNs()));
        case "authType":
            return Optional.ofNullable(clazz.cast(authType()));
        case "authorizerUri":
            return Optional.ofNullable(clazz.cast(authorizerUri()));
        case "authorizerCredentials":
            return Optional.ofNullable(clazz.cast(authorizerCredentials()));
        case "identitySource":
            return Optional.ofNullable(clazz.cast(identitySource()));
        case "identityValidationExpression":
            return Optional.ofNullable(clazz.cast(identityValidationExpression()));
        case "authorizerResultTtlInSeconds":
            return Optional.ofNullable(clazz.cast(authorizerResultTtlInSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetAuthorizerResponse, T> g) {
        return obj -> g.apply((GetAuthorizerResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetAuthorizerResponse> {
        /**
         * <p>
         * The identifier for the authorizer resource.
         * </p>
         * 
         * @param id
         *        The identifier for the authorizer resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * [Required] The name of the authorizer.
         * </p>
         * 
         * @param name
         *        [Required] The name of the authorizer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single authorization
         * token submitted in a custom header, <code>REQUEST</code> for a Lambda function using incoming request
         * parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
         * </p>
         * 
         * @param type
         *        The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single
         *        authorization token submitted in a custom header, <code>REQUEST</code> for a Lambda function using
         *        incoming request parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user
         *        pool.
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder type(String type);

        /**
         * <p>
         * The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single authorization
         * token submitted in a custom header, <code>REQUEST</code> for a Lambda function using incoming request
         * parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user pool.
         * </p>
         * 
         * @param type
         *        The authorizer type. Valid values are <code>TOKEN</code> for a Lambda function using a single
         *        authorization token submitted in a custom header, <code>REQUEST</code> for a Lambda function using
         *        incoming request parameters, and <code>COGNITO_USER_POOLS</code> for using an Amazon Cognito user
         *        pool.
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder type(AuthorizerType type);

        /**
         * <p>
         * A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each element
         * is of this format: <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>. For a
         * <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
         * </p>
         * 
         * @param providerARNs
         *        A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each
         *        element is of this format:
         *        <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>. For a
         *        <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerARNs(Collection<String> providerARNs);

        /**
         * <p>
         * A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each element
         * is of this format: <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>. For a
         * <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
         * </p>
         * 
         * @param providerARNs
         *        A list of the Amazon Cognito user pool ARNs for the <code>COGNITO_USER_POOLS</code> authorizer. Each
         *        element is of this format:
         *        <code>arn:aws:cognito-idp:{region}:{account_id}:userpool/{user_pool_id}</code>. For a
         *        <code>TOKEN</code> or <code>REQUEST</code> authorizer, this is not defined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerARNs(String... providerARNs);

        /**
         * <p>
         * Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
         * </p>
         * 
         * @param authType
         *        Optional customer-defined field, used in OpenAPI imports and exports without functional impact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authType(String authType);

        /**
         * <p>
         * Specifies the authorizer's Uniform Resource Identifier (URI). For <code>TOKEN</code> or <code>REQUEST</code>
         * authorizers, this must be a well-formed Lambda function URI, for example,
         * <code>arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations</code>
         * . In general, the URI has this form <code>arn:aws:apigateway:{region}:lambda:path/{service_api}</code>, where
         * <code>{region}</code> is the same as the region hosting the Lambda function, <code>path</code> indicates that
         * the remaining substring in the URI should be treated as the path to the resource, including the initial
         * <code>/</code>. For Lambda functions, this is usually of the form
         * <code>/2015-03-31/functions/[FunctionARN]/invocations</code>.
         * </p>
         * 
         * @param authorizerUri
         *        Specifies the authorizer's Uniform Resource Identifier (URI). For <code>TOKEN</code> or
         *        <code>REQUEST</code> authorizers, this must be a well-formed Lambda function URI, for example,
         *        <code>arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:{account_id}:function:{lambda_function_name}/invocations</code>
         *        . In general, the URI has this form <code>arn:aws:apigateway:{region}:lambda:path/{service_api}</code>
         *        , where <code>{region}</code> is the same as the region hosting the Lambda function, <code>path</code>
         *        indicates that the remaining substring in the URI should be treated as the path to the resource,
         *        including the initial <code>/</code>. For Lambda functions, this is usually of the form
         *        <code>/2015-03-31/functions/[FunctionARN]/invocations</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerUri(String authorizerUri);

        /**
         * <p>
         * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
         * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions
         * on the Lambda function, specify null.
         * </p>
         * 
         * @param authorizerCredentials
         *        Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify
         *        an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use
         *        resource-based permissions on the Lambda function, specify null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerCredentials(String authorizerCredentials);

        /**
         * <p>
         * The identity source for which authorization is requested.
         * <ul>
         * <li>For a <code>TOKEN</code> or <code>COGNITO_USER_POOLS</code> authorizer, this is required and specifies
         * the request header mapping expression for the custom header holding the authorization token submitted by the
         * client. For example, if the token header name is <code>Auth</code>, the header mapping expression is
         * <code>method.request.header.Auth</code>.</li>
         * <li>For the <code>REQUEST</code> authorizer, this is required when authorization caching is enabled. The
         * value is a comma-separated string of one or more mapping expressions of the specified request parameters. For
         * example, if an <code>Auth</code> header, a <code>Name</code> query string parameter are defined as identity
         * sources, this value is <code>method.request.header.Auth, method.request.querystring.Name</code>. These
         * parameters will be used to derive the authorization caching key and to perform runtime validation of the
         * <code>REQUEST</code> authorizer by verifying all of the identity-related request parameters are present, not
         * null and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function,
         * otherwise, it returns a 401 Unauthorized response without calling the Lambda function. The valid value is a
         * string of comma-separated mapping expressions of the specified request parameters. When the authorization
         * caching is not enabled, this property is optional.</li>
         * </ul>
         * </p>
         * 
         * @param identitySource
         *        The identity source for which authorization is requested.
         *        <ul>
         *        <li>For a <code>TOKEN</code> or <code>COGNITO_USER_POOLS</code> authorizer, this is required and
         *        specifies the request header mapping expression for the custom header holding the authorization token
         *        submitted by the client. For example, if the token header name is <code>Auth</code>, the header
         *        mapping expression is <code>method.request.header.Auth</code>.</li>
         *        <li>For the <code>REQUEST</code> authorizer, this is required when authorization caching is enabled.
         *        The value is a comma-separated string of one or more mapping expressions of the specified request
         *        parameters. For example, if an <code>Auth</code> header, a <code>Name</code> query string parameter
         *        are defined as identity sources, this value is
         *        <code>method.request.header.Auth, method.request.querystring.Name</code>. These parameters will be
         *        used to derive the authorization caching key and to perform runtime validation of the
         *        <code>REQUEST</code> authorizer by verifying all of the identity-related request parameters are
         *        present, not null and non-empty. Only when this is true does the authorizer invoke the authorizer
         *        Lambda function, otherwise, it returns a 401 Unauthorized response without calling the Lambda
         *        function. The valid value is a string of comma-separated mapping expressions of the specified request
         *        parameters. When the authorization caching is not enabled, this property is optional.</li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identitySource(String identitySource);

        /**
         * <p>
         * A validation expression for the incoming identity token. For <code>TOKEN</code> authorizers, this value is a
         * regular expression. For <code>COGNITO_USER_POOLS</code> authorizers, API Gateway will match the
         * <code>aud</code> field of the incoming token from the client against the specified regular expression. It
         * will invoke the authorizer's Lambda function when there is a match. Otherwise, it will return a 401
         * Unauthorized response without calling the Lambda function. The validation expression does not apply to the
         * <code>REQUEST</code> authorizer.
         * </p>
         * 
         * @param identityValidationExpression
         *        A validation expression for the incoming identity token. For <code>TOKEN</code> authorizers, this
         *        value is a regular expression. For <code>COGNITO_USER_POOLS</code> authorizers, API Gateway will match
         *        the <code>aud</code> field of the incoming token from the client against the specified regular
         *        expression. It will invoke the authorizer's Lambda function when there is a match. Otherwise, it will
         *        return a 401 Unauthorized response without calling the Lambda function. The validation expression does
         *        not apply to the <code>REQUEST</code> authorizer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityValidationExpression(String identityValidationExpression);

        /**
         * <p>
         * The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If it is
         * greater than 0, API Gateway will cache authorizer responses. If this field is not set, the default value is
         * 300. The maximum value is 3600, or 1 hour.
         * </p>
         * 
         * @param authorizerResultTtlInSeconds
         *        The TTL in seconds of cached authorizer results. If it equals 0, authorization caching is disabled. If
         *        it is greater than 0, API Gateway will cache authorizer responses. If this field is not set, the
         *        default value is 300. The maximum value is 3600, or 1 hour.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String id;

        private String name;

        private String type;

        private List<String> providerARNs = DefaultSdkAutoConstructList.getInstance();

        private String authType;

        private String authorizerUri;

        private String authorizerCredentials;

        private String identitySource;

        private String identityValidationExpression;

        private Integer authorizerResultTtlInSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(GetAuthorizerResponse model) {
            super(model);
            id(model.id);
            name(model.name);
            type(model.type);
            providerARNs(model.providerARNs);
            authType(model.authType);
            authorizerUri(model.authorizerUri);
            authorizerCredentials(model.authorizerCredentials);
            identitySource(model.identitySource);
            identityValidationExpression(model.identityValidationExpression);
            authorizerResultTtlInSeconds(model.authorizerResultTtlInSeconds);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(AuthorizerType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Collection<String> getProviderARNs() {
            return providerARNs;
        }

        @Override
        public final Builder providerARNs(Collection<String> providerARNs) {
            this.providerARNs = ListOfARNsCopier.copy(providerARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder providerARNs(String... providerARNs) {
            providerARNs(Arrays.asList(providerARNs));
            return this;
        }

        public final void setProviderARNs(Collection<String> providerARNs) {
            this.providerARNs = ListOfARNsCopier.copy(providerARNs);
        }

        public final String getAuthType() {
            return authType;
        }

        @Override
        public final Builder authType(String authType) {
            this.authType = authType;
            return this;
        }

        public final void setAuthType(String authType) {
            this.authType = authType;
        }

        public final String getAuthorizerUri() {
            return authorizerUri;
        }

        @Override
        public final Builder authorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
            return this;
        }

        public final void setAuthorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
        }

        public final String getAuthorizerCredentials() {
            return authorizerCredentials;
        }

        @Override
        public final Builder authorizerCredentials(String authorizerCredentials) {
            this.authorizerCredentials = authorizerCredentials;
            return this;
        }

        public final void setAuthorizerCredentials(String authorizerCredentials) {
            this.authorizerCredentials = authorizerCredentials;
        }

        public final String getIdentitySource() {
            return identitySource;
        }

        @Override
        public final Builder identitySource(String identitySource) {
            this.identitySource = identitySource;
            return this;
        }

        public final void setIdentitySource(String identitySource) {
            this.identitySource = identitySource;
        }

        public final String getIdentityValidationExpression() {
            return identityValidationExpression;
        }

        @Override
        public final Builder identityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
            return this;
        }

        public final void setIdentityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
        }

        public final Integer getAuthorizerResultTtlInSeconds() {
            return authorizerResultTtlInSeconds;
        }

        @Override
        public final Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
            return this;
        }

        public final void setAuthorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
        }

        @Override
        public GetAuthorizerResponse build() {
            return new GetAuthorizerResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
