/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a method response of a given HTTP status code returned to the client. The method response is passed from
 * the back end through the associated integration response that can be transformed using a mapping template.
 * </p>
 * <div class="remarks">
 * <p/>
 * <h4>Example: A <b>MethodResponse</b> instance of an API</h4>
 * <h5>Request</h5>
 * <p>
 * The example request retrieves a <b>MethodResponse</b> of the 200 status code.
 * </p>
 * 
 * <pre>
 * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160603T222952Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160603/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
 * </pre>
 * 
 * <h5>Response</h5>
 * <p>
 * The successful response returns <code>200 OK</code> status and a payload as follows:
 * </p>
 * 
 * <pre>
 * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true }, "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.Content-Type": false }, "statusCode": "200" }</code>
 * </pre>
 * <p/>
 * </div> <div class="seeAlso"> <a>Method</a>, <a>IntegrationResponse</a>, <a>Integration</a> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-create-api.html">Creating an API</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutMethodResponseResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<PutMethodResponseResponse.Builder, PutMethodResponseResponse> {
    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutMethodResponseResponse::statusCode)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<Map<String, Boolean>> RESPONSE_PARAMETERS_FIELD = SdkField
            .<Map<String, Boolean>> builder(MarshallingType.MAP)
            .getter(getter(PutMethodResponseResponse::responseParameters))
            .setter(setter(Builder::responseParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> RESPONSE_MODELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(PutMethodResponseResponse::responseModels))
            .setter(setter(Builder::responseModels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseModels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            RESPONSE_PARAMETERS_FIELD, RESPONSE_MODELS_FIELD));

    private final String statusCodeValue;

    private final Map<String, Boolean> responseParameters;

    private final Map<String, String> responseModels;

    private PutMethodResponseResponse(BuilderImpl builder) {
        super(builder);
        this.statusCodeValue = builder.statusCodeValue;
        this.responseParameters = builder.responseParameters;
        this.responseModels = builder.responseModels;
    }

    /**
     * <p>
     * The method response's status code.
     * </p>
     * 
     * @return The method response's status code.
     */
    public String statusCode() {
        return statusCodeValue;
    }

    /**
     * Returns true if the ResponseParameters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasResponseParameters() {
        return responseParameters != null && !(responseParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key-value map specifying required or optional response parameters that API Gateway can send back to the caller.
     * A key defines a method response header and the value specifies whether the associated method response header is
     * required or not. The expression of the key must match the pattern <code>method.response.header.{name}</code>,
     * where <code>name</code> is a valid and unique header name. API Gateway passes certain integration response data
     * to the method response headers specified here according to the mapping you prescribe in the API's
     * <a>IntegrationResponse</a>. The integration response data that can be mapped include an integration response
     * header expressed in <code>integration.response.header.{name}</code>, a static value enclosed within a pair of
     * single quotes (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response payload in
     * the form of <code>integration.response.body.{JSON-expression}</code>, where <code>JSON-expression</code> is a
     * valid JSON expression without the <code>$</code> prefix.)
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResponseParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A key-value map specifying required or optional response parameters that API Gateway can send back to the
     *         caller. A key defines a method response header and the value specifies whether the associated method
     *         response header is required or not. The expression of the key must match the pattern
     *         <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name.
     *         API Gateway passes certain integration response data to the method response headers specified here
     *         according to the mapping you prescribe in the API's <a>IntegrationResponse</a>. The integration response
     *         data that can be mapped include an integration response header expressed in
     *         <code>integration.response.header.{name}</code>, a static value enclosed within a pair of single quotes
     *         (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response payload in the
     *         form of <code>integration.response.body.{JSON-expression}</code>, where <code>JSON-expression</code> is a
     *         valid JSON expression without the <code>$</code> prefix.)
     */
    public Map<String, Boolean> responseParameters() {
        return responseParameters;
    }

    /**
     * Returns true if the ResponseModels property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResponseModels() {
        return responseModels != null && !(responseModels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the <a>Model</a> resources used for the response's content-type. Response models are represented as a
     * key/value map, with a content-type as the key and a <a>Model</a> name as the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResponseModels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the <a>Model</a> resources used for the response's content-type. Response models are
     *         represented as a key/value map, with a content-type as the key and a <a>Model</a> name as the value.
     */
    public Map<String, String> responseModels() {
        return responseModels;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(responseParameters());
        hashCode = 31 * hashCode + Objects.hashCode(responseModels());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutMethodResponseResponse)) {
            return false;
        }
        PutMethodResponseResponse other = (PutMethodResponseResponse) obj;
        return Objects.equals(statusCode(), other.statusCode())
                && Objects.equals(responseParameters(), other.responseParameters())
                && Objects.equals(responseModels(), other.responseModels());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutMethodResponseResponse").add("StatusCode", statusCode())
                .add("ResponseParameters", responseParameters()).add("ResponseModels", responseModels()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "responseParameters":
            return Optional.ofNullable(clazz.cast(responseParameters()));
        case "responseModels":
            return Optional.ofNullable(clazz.cast(responseModels()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutMethodResponseResponse, T> g) {
        return obj -> g.apply((PutMethodResponseResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo, CopyableBuilder<Builder, PutMethodResponseResponse> {
        /**
         * <p>
         * The method response's status code.
         * </p>
         * 
         * @param statusCodeValue
         *        The method response's status code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * A key-value map specifying required or optional response parameters that API Gateway can send back to the
         * caller. A key defines a method response header and the value specifies whether the associated method response
         * header is required or not. The expression of the key must match the pattern
         * <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name. API
         * Gateway passes certain integration response data to the method response headers specified here according to
         * the mapping you prescribe in the API's <a>IntegrationResponse</a>. The integration response data that can be
         * mapped include an integration response header expressed in <code>integration.response.header.{name}</code>, a
         * static value enclosed within a pair of single quotes (e.g., <code>'application/json'</code>), or a JSON
         * expression from the back-end response payload in the form of
         * <code>integration.response.body.{JSON-expression}</code>, where <code>JSON-expression</code> is a valid JSON
         * expression without the <code>$</code> prefix.)
         * </p>
         * 
         * @param responseParameters
         *        A key-value map specifying required or optional response parameters that API Gateway can send back to
         *        the caller. A key defines a method response header and the value specifies whether the associated
         *        method response header is required or not. The expression of the key must match the pattern
         *        <code>method.response.header.{name}</code>, where <code>name</code> is a valid and unique header name.
         *        API Gateway passes certain integration response data to the method response headers specified here
         *        according to the mapping you prescribe in the API's <a>IntegrationResponse</a>. The integration
         *        response data that can be mapped include an integration response header expressed in
         *        <code>integration.response.header.{name}</code>, a static value enclosed within a pair of single
         *        quotes (e.g., <code>'application/json'</code>), or a JSON expression from the back-end response
         *        payload in the form of <code>integration.response.body.{JSON-expression}</code>, where
         *        <code>JSON-expression</code> is a valid JSON expression without the <code>$</code> prefix.)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseParameters(Map<String, Boolean> responseParameters);

        /**
         * <p>
         * Specifies the <a>Model</a> resources used for the response's content-type. Response models are represented as
         * a key/value map, with a content-type as the key and a <a>Model</a> name as the value.
         * </p>
         * 
         * @param responseModels
         *        Specifies the <a>Model</a> resources used for the response's content-type. Response models are
         *        represented as a key/value map, with a content-type as the key and a <a>Model</a> name as the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseModels(Map<String, String> responseModels);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String statusCodeValue;

        private Map<String, Boolean> responseParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> responseModels = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutMethodResponseResponse model) {
            super(model);
            statusCode(model.statusCodeValue);
            responseParameters(model.responseParameters);
            responseModels(model.responseModels);
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        public final Map<String, Boolean> getResponseParameters() {
            return responseParameters;
        }

        @Override
        public final Builder responseParameters(Map<String, Boolean> responseParameters) {
            this.responseParameters = MapOfStringToBooleanCopier.copy(responseParameters);
            return this;
        }

        public final void setResponseParameters(Map<String, Boolean> responseParameters) {
            this.responseParameters = MapOfStringToBooleanCopier.copy(responseParameters);
        }

        public final Map<String, String> getResponseModels() {
            return responseModels;
        }

        @Override
        public final Builder responseModels(Map<String, String> responseModels) {
            this.responseModels = MapOfStringToStringCopier.copy(responseModels);
            return this;
        }

        public final void setResponseModels(Map<String, String> responseModels) {
            this.responseModels = MapOfStringToStringCopier.copy(responseModels);
        }

        @Override
        public PutMethodResponseResponse build() {
            return new PutMethodResponseResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
