/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a VPC link, under the caller's account in a selected region, in an asynchronous operation that typically
 * takes 2-4 minutes to complete and become operational. The caller must have permissions to create and update VPC
 * Endpoint services.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateVpcLinkRequest extends ApiGatewayRequest implements
        ToCopyableBuilder<CreateVpcLinkRequest.Builder, CreateVpcLinkRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVpcLinkRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVpcLinkRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> TARGET_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateVpcLinkRequest::targetArns))
            .setter(setter(Builder::targetArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(CreateVpcLinkRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            TARGET_ARNS_FIELD, TAGS_FIELD));

    private final String name;

    private final String description;

    private final List<String> targetArns;

    private final Map<String, String> tags;

    private CreateVpcLinkRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.targetArns = builder.targetArns;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * [Required] The name used to label and identify the VPC link.
     * </p>
     * 
     * @return [Required] The name used to label and identify the VPC link.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The description of the VPC link.
     * </p>
     * 
     * @return The description of the VPC link.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the TargetArns property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTargetArns() {
        return targetArns != null && !(targetArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers
     * must be owned by the same AWS account of the API owner.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load
     *         balancers must be owned by the same AWS account of the API owner.
     */
    public List<String> targetArns() {
        return targetArns;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128 characters
     * and must not start with <code>aws:</code>. The tag value can be up to 256 characters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128
     *         characters and must not start with <code>aws:</code>. The tag value can be up to 256 characters.
     */
    public Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(targetArns());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateVpcLinkRequest)) {
            return false;
        }
        CreateVpcLinkRequest other = (CreateVpcLinkRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(targetArns(), other.targetArns()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateVpcLinkRequest").add("Name", name()).add("Description", description())
                .add("TargetArns", targetArns()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "targetArns":
            return Optional.ofNullable(clazz.cast(targetArns()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateVpcLinkRequest, T> g) {
        return obj -> g.apply((CreateVpcLinkRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateVpcLinkRequest> {
        /**
         * <p>
         * [Required] The name used to label and identify the VPC link.
         * </p>
         * 
         * @param name
         *        [Required] The name used to label and identify the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the VPC link.
         * </p>
         * 
         * @param description
         *        The description of the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers
         * must be owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load
         *        balancers must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(Collection<String> targetArns);

        /**
         * <p>
         * [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load balancers
         * must be owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        [Required] The ARNs of network load balancers of the VPC targeted by the VPC link. The network load
         *        balancers must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(String... targetArns);

        /**
         * <p>
         * The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128
         * characters and must not start with <code>aws:</code>. The tag value can be up to 256 characters.
         * </p>
         * 
         * @param tags
         *        The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The tag key can be up to 128
         *        characters and must not start with <code>aws:</code>. The tag value can be up to 256 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private List<String> targetArns = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateVpcLinkRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            targetArns(model.targetArns);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getTargetArns() {
            return targetArns;
        }

        @Override
        public final Builder targetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetArns(String... targetArns) {
            targetArns(Arrays.asList(targetArns));
            return this;
        }

        public final void setTargetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateVpcLinkRequest build() {
            return new CreateVpcLinkRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
