/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents an AWS account that is associated with API Gateway.
 * </p>
 * <div class="remarks">
 * <p>
 * To view the account info, call <code>GET</code> on this resource.
 * </p>
 * <h4>Error Codes</h4>
 * <p>
 * The following exception may be thrown when the request fails.
 * </p>
 * <ul>
 * <li>UnauthorizedException</li>
 * <li>NotFoundException</li>
 * <li>TooManyRequestsException</li>
 * </ul>
 * <p>
 * For detailed error code information, including the corresponding HTTP Status Codes, see <a
 * href="https://docs.aws.amazon.com/apigateway/api-reference/handling-errors/#api-error-codes">API Gateway Error
 * Codes</a>
 * </p>
 * <h4>Example: Get the information about an account.</h4> <h5>Request</h5>
 * 
 * <pre>
 * <code>GET /account HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160531T184618Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash} </code>
 * </pre>
 * 
 * <h5>Response</h5>
 * <p>
 * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
 * </p>
 * 
 * <pre>
 * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/account-apigateway-{rel}.html", "name": "account", "templated": true }, "self": { "href": "/account" }, "account:update": { "href": "/account" } }, "cloudwatchRoleArn": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "throttleSettings": { "rateLimit": 500, "burstLimit": 1000 } } </code>
 * </pre>
 * <p>
 * In addition to making the REST API call directly, you can use the AWS CLI and an AWS SDK to access this resource.
 * </p>
 * </div> <div class="seeAlso"> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-limits.html">API Gateway Limits</a> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/welcome.html">Developer Guide</a>, <a
 * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-account.html">AWS CLI</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetAccountResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<GetAccountResponse.Builder, GetAccountResponse> {
    private static final SdkField<String> CLOUDWATCH_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("cloudwatchRoleArn").getter(getter(GetAccountResponse::cloudwatchRoleArn))
            .setter(setter(Builder::cloudwatchRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cloudwatchRoleArn").build()).build();

    private static final SdkField<ThrottleSettings> THROTTLE_SETTINGS_FIELD = SdkField
            .<ThrottleSettings> builder(MarshallingType.SDK_POJO).memberName("throttleSettings")
            .getter(getter(GetAccountResponse::throttleSettings)).setter(setter(Builder::throttleSettings))
            .constructor(ThrottleSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("throttleSettings").build()).build();

    private static final SdkField<List<String>> FEATURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("features")
            .getter(getter(GetAccountResponse::features))
            .setter(setter(Builder::features))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("features").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> API_KEY_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("apiKeyVersion").getter(getter(GetAccountResponse::apiKeyVersion)).setter(setter(Builder::apiKeyVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("apiKeyVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLOUDWATCH_ROLE_ARN_FIELD,
            THROTTLE_SETTINGS_FIELD, FEATURES_FIELD, API_KEY_VERSION_FIELD));

    private final String cloudwatchRoleArn;

    private final ThrottleSettings throttleSettings;

    private final List<String> features;

    private final String apiKeyVersion;

    private GetAccountResponse(BuilderImpl builder) {
        super(builder);
        this.cloudwatchRoleArn = builder.cloudwatchRoleArn;
        this.throttleSettings = builder.throttleSettings;
        this.features = builder.features;
        this.apiKeyVersion = builder.apiKeyVersion;
    }

    /**
     * <p>
     * The ARN of an Amazon CloudWatch role for the current <a>Account</a>.
     * </p>
     * 
     * @return The ARN of an Amazon CloudWatch role for the current <a>Account</a>.
     */
    public final String cloudwatchRoleArn() {
        return cloudwatchRoleArn;
    }

    /**
     * <p>
     * Specifies the API request limits configured for the current <a>Account</a>.
     * </p>
     * 
     * @return Specifies the API request limits configured for the current <a>Account</a>.
     */
    public final ThrottleSettings throttleSettings() {
        return throttleSettings;
    }

    /**
     * Returns true if the Features property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasFeatures() {
        return features != null && !(features instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of features supported for the account. When usage plans are enabled, the features list will include an
     * entry of <code>"UsagePlans"</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFeatures()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of features supported for the account. When usage plans are enabled, the features list will
     *         include an entry of <code>"UsagePlans"</code>.
     */
    public final List<String> features() {
        return features;
    }

    /**
     * <p>
     * The version of the API keys used for the account.
     * </p>
     * 
     * @return The version of the API keys used for the account.
     */
    public final String apiKeyVersion() {
        return apiKeyVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(cloudwatchRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(throttleSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeatures() ? features() : null);
        hashCode = 31 * hashCode + Objects.hashCode(apiKeyVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetAccountResponse)) {
            return false;
        }
        GetAccountResponse other = (GetAccountResponse) obj;
        return Objects.equals(cloudwatchRoleArn(), other.cloudwatchRoleArn())
                && Objects.equals(throttleSettings(), other.throttleSettings()) && hasFeatures() == other.hasFeatures()
                && Objects.equals(features(), other.features()) && Objects.equals(apiKeyVersion(), other.apiKeyVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetAccountResponse").add("CloudwatchRoleArn", cloudwatchRoleArn())
                .add("ThrottleSettings", throttleSettings()).add("Features", hasFeatures() ? features() : null)
                .add("ApiKeyVersion", apiKeyVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cloudwatchRoleArn":
            return Optional.ofNullable(clazz.cast(cloudwatchRoleArn()));
        case "throttleSettings":
            return Optional.ofNullable(clazz.cast(throttleSettings()));
        case "features":
            return Optional.ofNullable(clazz.cast(features()));
        case "apiKeyVersion":
            return Optional.ofNullable(clazz.cast(apiKeyVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetAccountResponse, T> g) {
        return obj -> g.apply((GetAccountResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetAccountResponse> {
        /**
         * <p>
         * The ARN of an Amazon CloudWatch role for the current <a>Account</a>.
         * </p>
         * 
         * @param cloudwatchRoleArn
         *        The ARN of an Amazon CloudWatch role for the current <a>Account</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudwatchRoleArn(String cloudwatchRoleArn);

        /**
         * <p>
         * Specifies the API request limits configured for the current <a>Account</a>.
         * </p>
         * 
         * @param throttleSettings
         *        Specifies the API request limits configured for the current <a>Account</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throttleSettings(ThrottleSettings throttleSettings);

        /**
         * <p>
         * Specifies the API request limits configured for the current <a>Account</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link ThrottleSettings.Builder} avoiding the need to
         * create one manually via {@link ThrottleSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link ThrottleSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #throttleSettings(ThrottleSettings)}.
         * 
         * @param throttleSettings
         *        a consumer that will call methods on {@link ThrottleSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #throttleSettings(ThrottleSettings)
         */
        default Builder throttleSettings(Consumer<ThrottleSettings.Builder> throttleSettings) {
            return throttleSettings(ThrottleSettings.builder().applyMutation(throttleSettings).build());
        }

        /**
         * <p>
         * A list of features supported for the account. When usage plans are enabled, the features list will include an
         * entry of <code>"UsagePlans"</code>.
         * </p>
         * 
         * @param features
         *        A list of features supported for the account. When usage plans are enabled, the features list will
         *        include an entry of <code>"UsagePlans"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(Collection<String> features);

        /**
         * <p>
         * A list of features supported for the account. When usage plans are enabled, the features list will include an
         * entry of <code>"UsagePlans"</code>.
         * </p>
         * 
         * @param features
         *        A list of features supported for the account. When usage plans are enabled, the features list will
         *        include an entry of <code>"UsagePlans"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder features(String... features);

        /**
         * <p>
         * The version of the API keys used for the account.
         * </p>
         * 
         * @param apiKeyVersion
         *        The version of the API keys used for the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiKeyVersion(String apiKeyVersion);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String cloudwatchRoleArn;

        private ThrottleSettings throttleSettings;

        private List<String> features = DefaultSdkAutoConstructList.getInstance();

        private String apiKeyVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(GetAccountResponse model) {
            super(model);
            cloudwatchRoleArn(model.cloudwatchRoleArn);
            throttleSettings(model.throttleSettings);
            features(model.features);
            apiKeyVersion(model.apiKeyVersion);
        }

        public final String getCloudwatchRoleArn() {
            return cloudwatchRoleArn;
        }

        public final void setCloudwatchRoleArn(String cloudwatchRoleArn) {
            this.cloudwatchRoleArn = cloudwatchRoleArn;
        }

        @Override
        @Transient
        public final Builder cloudwatchRoleArn(String cloudwatchRoleArn) {
            this.cloudwatchRoleArn = cloudwatchRoleArn;
            return this;
        }

        public final ThrottleSettings.Builder getThrottleSettings() {
            return throttleSettings != null ? throttleSettings.toBuilder() : null;
        }

        public final void setThrottleSettings(ThrottleSettings.BuilderImpl throttleSettings) {
            this.throttleSettings = throttleSettings != null ? throttleSettings.build() : null;
        }

        @Override
        @Transient
        public final Builder throttleSettings(ThrottleSettings throttleSettings) {
            this.throttleSettings = throttleSettings;
            return this;
        }

        public final Collection<String> getFeatures() {
            if (features instanceof SdkAutoConstructList) {
                return null;
            }
            return features;
        }

        public final void setFeatures(Collection<String> features) {
            this.features = ListOfStringCopier.copy(features);
        }

        @Override
        @Transient
        public final Builder features(Collection<String> features) {
            this.features = ListOfStringCopier.copy(features);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder features(String... features) {
            features(Arrays.asList(features));
            return this;
        }

        public final String getApiKeyVersion() {
            return apiKeyVersion;
        }

        public final void setApiKeyVersion(String apiKeyVersion) {
            this.apiKeyVersion = apiKeyVersion;
        }

        @Override
        @Transient
        public final Builder apiKeyVersion(String apiKeyVersion) {
            this.apiKeyVersion = apiKeyVersion;
            return this;
        }

        @Override
        public GetAccountResponse build() {
            return new GetAccountResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
