/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An API Gateway VPC link for a <a>RestApi</a> to access resources in an Amazon Virtual Private Cloud (VPC).
 * </p>
 * <div class="remarks">
 * <p>
 * <p>
 * To enable access to a resource in an Amazon Virtual Private Cloud through Amazon API Gateway, you, as an API
 * developer, create a <a>VpcLink</a> resource targeted for one or more network load balancers of the VPC and then
 * integrate an API method with a private integration that uses the <a>VpcLink</a>. The private integration has an
 * integration type of <code>HTTP</code> or <code>HTTP_PROXY</code> and has a connection type of <code>VPC_LINK</code>.
 * The integration uses the <code>connectionId</code> property to identify the <a>VpcLink</a> used.
 * </p>
 * </p> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateVpcLinkResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<CreateVpcLinkResponse.Builder, CreateVpcLinkResponse> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(CreateVpcLinkResponse::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateVpcLinkResponse::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(CreateVpcLinkResponse::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> TARGET_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("targetArns")
            .getter(getter(CreateVpcLinkResponse::targetArns))
            .setter(setter(Builder::targetArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(CreateVpcLinkResponse::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusMessage").getter(getter(CreateVpcLinkResponse::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusMessage").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateVpcLinkResponse::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            DESCRIPTION_FIELD, TARGET_ARNS_FIELD, STATUS_FIELD, STATUS_MESSAGE_FIELD, TAGS_FIELD));

    private final String id;

    private final String name;

    private final String description;

    private final List<String> targetArns;

    private final String status;

    private final String statusMessage;

    private final Map<String, String> tags;

    private CreateVpcLinkResponse(BuilderImpl builder) {
        super(builder);
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.targetArns = builder.targetArns;
        this.status = builder.status;
        this.statusMessage = builder.statusMessage;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this <a>VpcLink</a>.
     * </p>
     * 
     * @return The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this
     *         <a>VpcLink</a>.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name used to label and identify the VPC link.
     * </p>
     * 
     * @return The name used to label and identify the VPC link.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the VPC link.
     * </p>
     * 
     * @return The description of the VPC link.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargetArns() {
        return targetArns != null && !(targetArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must be owned
     * by the same AWS account of the API owner.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetArns} method.
     * </p>
     * 
     * @return The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must
     *         be owned by the same AWS account of the API owner.
     */
    public final List<String> targetArns() {
        return targetArns;
    }

    /**
     * <p>
     * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is <code>PENDING</code>
     * and will fail if the status is <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcLinkStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     *         <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
     *         <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
     * @see VpcLinkStatus
     */
    public final VpcLinkStatus status() {
        return VpcLinkStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is <code>PENDING</code>
     * and will fail if the status is <code>DELETING</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link VpcLinkStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
     *         <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
     *         <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
     * @see VpcLinkStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A description about the VPC link status.
     * </p>
     * 
     * @return A description about the VPC link status.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The collection of tags. Each tag element is associated with a given resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The collection of tags. Each tag element is associated with a given resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetArns() ? targetArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateVpcLinkResponse)) {
            return false;
        }
        CreateVpcLinkResponse other = (CreateVpcLinkResponse) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && hasTargetArns() == other.hasTargetArns()
                && Objects.equals(targetArns(), other.targetArns()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateVpcLinkResponse").add("Id", id()).add("Name", name()).add("Description", description())
                .add("TargetArns", hasTargetArns() ? targetArns() : null).add("Status", statusAsString())
                .add("StatusMessage", statusMessage()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "targetArns":
            return Optional.ofNullable(clazz.cast(targetArns()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateVpcLinkResponse, T> g) {
        return obj -> g.apply((CreateVpcLinkResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo, CopyableBuilder<Builder, CreateVpcLinkResponse> {
        /**
         * <p>
         * The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this <a>VpcLink</a>.
         * </p>
         * 
         * @param id
         *        The identifier of the <a>VpcLink</a>. It is used in an <a>Integration</a> to reference this
         *        <a>VpcLink</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name used to label and identify the VPC link.
         * </p>
         * 
         * @param name
         *        The name used to label and identify the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the VPC link.
         * </p>
         * 
         * @param description
         *        The description of the VPC link.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must be
         * owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer
         *        must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(Collection<String> targetArns);

        /**
         * <p>
         * The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer must be
         * owned by the same AWS account of the API owner.
         * </p>
         * 
         * @param targetArns
         *        The ARN of the network load balancer of the VPC targeted by the VPC link. The network load balancer
         *        must be owned by the same AWS account of the API owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetArns(String... targetArns);

        /**
         * <p>
         * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         * <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         *        <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         *        <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * @see VpcLinkStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcLinkStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         * <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         * <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * </p>
         * 
         * @param status
         *        The status of the VPC link. The valid values are <code>AVAILABLE</code>, <code>PENDING</code>,
         *        <code>DELETING</code>, or <code>FAILED</code>. Deploying an API will wait if the status is
         *        <code>PENDING</code> and will fail if the status is <code>DELETING</code>.
         * @see VpcLinkStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpcLinkStatus
         */
        Builder status(VpcLinkStatus status);

        /**
         * <p>
         * A description about the VPC link status.
         * </p>
         * 
         * @param statusMessage
         *        A description about the VPC link status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The collection of tags. Each tag element is associated with a given resource.
         * </p>
         * 
         * @param tags
         *        The collection of tags. Each tag element is associated with a given resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String id;

        private String name;

        private String description;

        private List<String> targetArns = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String statusMessage;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateVpcLinkResponse model) {
            super(model);
            id(model.id);
            name(model.name);
            description(model.description);
            targetArns(model.targetArns);
            status(model.status);
            statusMessage(model.statusMessage);
            tags(model.tags);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getTargetArns() {
            if (targetArns instanceof SdkAutoConstructList) {
                return null;
            }
            return targetArns;
        }

        public final void setTargetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
        }

        @Override
        public final Builder targetArns(Collection<String> targetArns) {
            this.targetArns = ListOfStringCopier.copy(targetArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetArns(String... targetArns) {
            targetArns(Arrays.asList(targetArns));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(VpcLinkStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
            return this;
        }

        @Override
        public CreateVpcLinkResponse build() {
            return new CreateVpcLinkResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
