/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a custom domain name as a user-friendly host name of an API (<a>RestApi</a>).
 * </p>
 * <div class="Remarks">
 * <p>
 * When you deploy an API, API Gateway creates a default host name for the API. This default API host name is of the
 * <code>{restapi-id}.execute-api.{region}.amazonaws.com</code> format. With the default host name, you can access the
 * API's root resource with the URL of <code>https://{restapi-id}.execute-api.{region}.amazonaws.com/{stage}/</code>.
 * When you set up a custom domain name of <code>apis.example.com</code> for this API, you can then access the same
 * resource using the URL of the <code>https://apis.examples.com/myApi</code>, where <code>myApi</code> is the base path
 * mapping (<a>BasePathMapping</a>) of your API under the custom domain name.
 * </p>
 * </div> <div class="seeAlso"> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-custom-domains.html">Set a Custom Host Name
 * for an API</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DomainName implements SdkPojo, Serializable, ToCopyableBuilder<DomainName.Builder, DomainName> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(DomainName::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<String> CERTIFICATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateName").getter(getter(DomainName::certificateName)).setter(setter(Builder::certificateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateName").build()).build();

    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateArn").getter(getter(DomainName::certificateArn)).setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateArn").build()).build();

    private static final SdkField<Instant> CERTIFICATE_UPLOAD_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("certificateUploadDate").getter(getter(DomainName::certificateUploadDate))
            .setter(setter(Builder::certificateUploadDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateUploadDate").build())
            .build();

    private static final SdkField<String> REGIONAL_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionalDomainName").getter(getter(DomainName::regionalDomainName))
            .setter(setter(Builder::regionalDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionalDomainName").build())
            .build();

    private static final SdkField<String> REGIONAL_HOSTED_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionalHostedZoneId").getter(getter(DomainName::regionalHostedZoneId))
            .setter(setter(Builder::regionalHostedZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionalHostedZoneId").build())
            .build();

    private static final SdkField<String> REGIONAL_CERTIFICATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionalCertificateName").getter(getter(DomainName::regionalCertificateName))
            .setter(setter(Builder::regionalCertificateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionalCertificateName").build())
            .build();

    private static final SdkField<String> REGIONAL_CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("regionalCertificateArn").getter(getter(DomainName::regionalCertificateArn))
            .setter(setter(Builder::regionalCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regionalCertificateArn").build())
            .build();

    private static final SdkField<String> DISTRIBUTION_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("distributionDomainName").getter(getter(DomainName::distributionDomainName))
            .setter(setter(Builder::distributionDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("distributionDomainName").build())
            .build();

    private static final SdkField<String> DISTRIBUTION_HOSTED_ZONE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("distributionHostedZoneId").getter(getter(DomainName::distributionHostedZoneId))
            .setter(setter(Builder::distributionHostedZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("distributionHostedZoneId").build())
            .build();

    private static final SdkField<EndpointConfiguration> ENDPOINT_CONFIGURATION_FIELD = SdkField
            .<EndpointConfiguration> builder(MarshallingType.SDK_POJO).memberName("endpointConfiguration")
            .getter(getter(DomainName::endpointConfiguration)).setter(setter(Builder::endpointConfiguration))
            .constructor(EndpointConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpointConfiguration").build())
            .build();

    private static final SdkField<String> DOMAIN_NAME_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainNameStatus").getter(getter(DomainName::domainNameStatusAsString))
            .setter(setter(Builder::domainNameStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainNameStatus").build()).build();

    private static final SdkField<String> DOMAIN_NAME_STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainNameStatusMessage").getter(getter(DomainName::domainNameStatusMessage))
            .setter(setter(Builder::domainNameStatusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainNameStatusMessage").build())
            .build();

    private static final SdkField<String> SECURITY_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("securityPolicy").getter(getter(DomainName::securityPolicyAsString))
            .setter(setter(Builder::securityPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityPolicy").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(DomainName::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<MutualTlsAuthentication> MUTUAL_TLS_AUTHENTICATION_FIELD = SdkField
            .<MutualTlsAuthentication> builder(MarshallingType.SDK_POJO).memberName("mutualTlsAuthentication")
            .getter(getter(DomainName::mutualTlsAuthentication)).setter(setter(Builder::mutualTlsAuthentication))
            .constructor(MutualTlsAuthentication::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mutualTlsAuthentication").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            CERTIFICATE_NAME_FIELD, CERTIFICATE_ARN_FIELD, CERTIFICATE_UPLOAD_DATE_FIELD, REGIONAL_DOMAIN_NAME_FIELD,
            REGIONAL_HOSTED_ZONE_ID_FIELD, REGIONAL_CERTIFICATE_NAME_FIELD, REGIONAL_CERTIFICATE_ARN_FIELD,
            DISTRIBUTION_DOMAIN_NAME_FIELD, DISTRIBUTION_HOSTED_ZONE_ID_FIELD, ENDPOINT_CONFIGURATION_FIELD,
            DOMAIN_NAME_STATUS_FIELD, DOMAIN_NAME_STATUS_MESSAGE_FIELD, SECURITY_POLICY_FIELD, TAGS_FIELD,
            MUTUAL_TLS_AUTHENTICATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domainName;

    private final String certificateName;

    private final String certificateArn;

    private final Instant certificateUploadDate;

    private final String regionalDomainName;

    private final String regionalHostedZoneId;

    private final String regionalCertificateName;

    private final String regionalCertificateArn;

    private final String distributionDomainName;

    private final String distributionHostedZoneId;

    private final EndpointConfiguration endpointConfiguration;

    private final String domainNameStatus;

    private final String domainNameStatusMessage;

    private final String securityPolicy;

    private final Map<String, String> tags;

    private final MutualTlsAuthentication mutualTlsAuthentication;

    private DomainName(BuilderImpl builder) {
        this.domainName = builder.domainName;
        this.certificateName = builder.certificateName;
        this.certificateArn = builder.certificateArn;
        this.certificateUploadDate = builder.certificateUploadDate;
        this.regionalDomainName = builder.regionalDomainName;
        this.regionalHostedZoneId = builder.regionalHostedZoneId;
        this.regionalCertificateName = builder.regionalCertificateName;
        this.regionalCertificateArn = builder.regionalCertificateArn;
        this.distributionDomainName = builder.distributionDomainName;
        this.distributionHostedZoneId = builder.distributionHostedZoneId;
        this.endpointConfiguration = builder.endpointConfiguration;
        this.domainNameStatus = builder.domainNameStatus;
        this.domainNameStatusMessage = builder.domainNameStatusMessage;
        this.securityPolicy = builder.securityPolicy;
        this.tags = builder.tags;
        this.mutualTlsAuthentication = builder.mutualTlsAuthentication;
    }

    /**
     * <p>
     * The custom domain name as an API host name, for example, <code>my-api.example.com</code>.
     * </p>
     * 
     * @return The custom domain name as an API host name, for example, <code>my-api.example.com</code>.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The name of the certificate that will be used by edge-optimized endpoint for this domain name.
     * </p>
     * 
     * @return The name of the certificate that will be used by edge-optimized endpoint for this domain name.
     */
    public final String certificateName() {
        return certificateName;
    }

    /**
     * <p>
     * The reference to an AWS-managed certificate that will be used by edge-optimized endpoint for this domain name.
     * AWS Certificate Manager is the only supported source.
     * </p>
     * 
     * @return The reference to an AWS-managed certificate that will be used by edge-optimized endpoint for this domain
     *         name. AWS Certificate Manager is the only supported source.
     */
    public final String certificateArn() {
        return certificateArn;
    }

    /**
     * <p>
     * The timestamp when the certificate that was used by edge-optimized endpoint for this domain name was uploaded.
     * </p>
     * 
     * @return The timestamp when the certificate that was used by edge-optimized endpoint for this domain name was
     *         uploaded.
     */
    public final Instant certificateUploadDate() {
        return certificateUploadDate;
    }

    /**
     * <p>
     * The domain name associated with the regional endpoint for this custom domain name. You set up this association by
     * adding a DNS record that points the custom domain name to this regional domain name. The regional domain name is
     * returned by API Gateway when you create a regional endpoint.
     * </p>
     * 
     * @return The domain name associated with the regional endpoint for this custom domain name. You set up this
     *         association by adding a DNS record that points the custom domain name to this regional domain name. The
     *         regional domain name is returned by API Gateway when you create a regional endpoint.
     */
    public final String regionalDomainName() {
        return regionalDomainName;
    }

    /**
     * <p>
     * The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint. For more information, see <a href=
     * "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
     * >Set up a Regional Custom Domain Name</a> and <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and Endpoints for
     * API Gateway</a>.
     * </p>
     * 
     * @return The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
     *         >Set up a Regional Custom Domain Name</a> and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and
     *         Endpoints for API Gateway</a>.
     */
    public final String regionalHostedZoneId() {
        return regionalHostedZoneId;
    }

    /**
     * <p>
     * The name of the certificate that will be used for validating the regional domain name.
     * </p>
     * 
     * @return The name of the certificate that will be used for validating the regional domain name.
     */
    public final String regionalCertificateName() {
        return regionalCertificateName;
    }

    /**
     * <p>
     * The reference to an AWS-managed certificate that will be used for validating the regional domain name. AWS
     * Certificate Manager is the only supported source.
     * </p>
     * 
     * @return The reference to an AWS-managed certificate that will be used for validating the regional domain name.
     *         AWS Certificate Manager is the only supported source.
     */
    public final String regionalCertificateArn() {
        return regionalCertificateArn;
    }

    /**
     * <p>
     * The domain name of the Amazon CloudFront distribution associated with this custom domain name for an
     * edge-optimized endpoint. You set up this association when adding a DNS record pointing the custom domain name to
     * this distribution name. For more information about CloudFront distributions, see the <a
     * href="https://aws.amazon.com/documentation/cloudfront/" target="_blank">Amazon CloudFront documentation</a>.
     * </p>
     * 
     * @return The domain name of the Amazon CloudFront distribution associated with this custom domain name for an
     *         edge-optimized endpoint. You set up this association when adding a DNS record pointing the custom domain
     *         name to this distribution name. For more information about CloudFront distributions, see the <a
     *         href="https://aws.amazon.com/documentation/cloudfront/" target="_blank">Amazon CloudFront
     *         documentation</a>.
     */
    public final String distributionDomainName() {
        return distributionDomainName;
    }

    /**
     * <p>
     * The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint. The valid value is
     * <code>Z2FDTNDATAQYW2</code> for all the regions. For more information, see <a href=
     * "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
     * >Set up a Regional Custom Domain Name</a> and <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and Endpoints for
     * API Gateway</a>.
     * </p>
     * 
     * @return The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint. The valid value is
     *         <code>Z2FDTNDATAQYW2</code> for all the regions. For more information, see <a href=
     *         "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
     *         >Set up a Regional Custom Domain Name</a> and <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and
     *         Endpoints for API Gateway</a>.
     */
    public final String distributionHostedZoneId() {
        return distributionHostedZoneId;
    }

    /**
     * <p>
     * The endpoint configuration of this <a>DomainName</a> showing the endpoint types of the domain name.
     * </p>
     * 
     * @return The endpoint configuration of this <a>DomainName</a> showing the endpoint types of the domain name.
     */
    public final EndpointConfiguration endpointConfiguration() {
        return endpointConfiguration;
    }

    /**
     * <p>
     * The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
     * <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further until the
     * existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domainNameStatus}
     * will return {@link DomainNameStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #domainNameStatusAsString}.
     * </p>
     * 
     * @return The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
     *         <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further
     *         until the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
     * @see DomainNameStatus
     */
    public final DomainNameStatus domainNameStatus() {
        return DomainNameStatus.fromValue(domainNameStatus);
    }

    /**
     * <p>
     * The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
     * <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further until the
     * existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domainNameStatus}
     * will return {@link DomainNameStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #domainNameStatusAsString}.
     * </p>
     * 
     * @return The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
     *         <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further
     *         until the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
     * @see DomainNameStatus
     */
    public final String domainNameStatusAsString() {
        return domainNameStatus;
    }

    /**
     * <p>
     * An optional text message containing detailed information about status of the <a>DomainName</a> migration.
     * </p>
     * 
     * @return An optional text message containing detailed information about status of the <a>DomainName</a> migration.
     */
    public final String domainNameStatusMessage() {
        return domainNameStatusMessage;
    }

    /**
     * <p>
     * The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values are
     * <code>TLS_1_0</code> and <code>TLS_1_2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #securityPolicy}
     * will return {@link SecurityPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #securityPolicyAsString}.
     * </p>
     * 
     * @return The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values
     *         are <code>TLS_1_0</code> and <code>TLS_1_2</code>.
     * @see SecurityPolicy
     */
    public final SecurityPolicy securityPolicy() {
        return SecurityPolicy.fromValue(securityPolicy);
    }

    /**
     * <p>
     * The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values are
     * <code>TLS_1_0</code> and <code>TLS_1_2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #securityPolicy}
     * will return {@link SecurityPolicy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #securityPolicyAsString}.
     * </p>
     * 
     * @return The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values
     *         are <code>TLS_1_0</code> and <code>TLS_1_2</code>.
     * @see SecurityPolicy
     */
    public final String securityPolicyAsString() {
        return securityPolicy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The collection of tags. Each tag element is associated with a given resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The collection of tags. Each tag element is associated with a given resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs two-way
     * authentication between the client and the server. Clients must present a trusted certificate to access your API.
     * </p>
     * 
     * @return The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs
     *         two-way authentication between the client and the server. Clients must present a trusted certificate to
     *         access your API.
     */
    public final MutualTlsAuthentication mutualTlsAuthentication() {
        return mutualTlsAuthentication;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(certificateName());
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateUploadDate());
        hashCode = 31 * hashCode + Objects.hashCode(regionalDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(regionalHostedZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(regionalCertificateName());
        hashCode = 31 * hashCode + Objects.hashCode(regionalCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(distributionDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(distributionHostedZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(domainNameStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(domainNameStatusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(securityPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mutualTlsAuthentication());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DomainName)) {
            return false;
        }
        DomainName other = (DomainName) obj;
        return Objects.equals(domainName(), other.domainName()) && Objects.equals(certificateName(), other.certificateName())
                && Objects.equals(certificateArn(), other.certificateArn())
                && Objects.equals(certificateUploadDate(), other.certificateUploadDate())
                && Objects.equals(regionalDomainName(), other.regionalDomainName())
                && Objects.equals(regionalHostedZoneId(), other.regionalHostedZoneId())
                && Objects.equals(regionalCertificateName(), other.regionalCertificateName())
                && Objects.equals(regionalCertificateArn(), other.regionalCertificateArn())
                && Objects.equals(distributionDomainName(), other.distributionDomainName())
                && Objects.equals(distributionHostedZoneId(), other.distributionHostedZoneId())
                && Objects.equals(endpointConfiguration(), other.endpointConfiguration())
                && Objects.equals(domainNameStatusAsString(), other.domainNameStatusAsString())
                && Objects.equals(domainNameStatusMessage(), other.domainNameStatusMessage())
                && Objects.equals(securityPolicyAsString(), other.securityPolicyAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(mutualTlsAuthentication(), other.mutualTlsAuthentication());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DomainName").add("DomainName", domainName()).add("CertificateName", certificateName())
                .add("CertificateArn", certificateArn()).add("CertificateUploadDate", certificateUploadDate())
                .add("RegionalDomainName", regionalDomainName()).add("RegionalHostedZoneId", regionalHostedZoneId())
                .add("RegionalCertificateName", regionalCertificateName())
                .add("RegionalCertificateArn", regionalCertificateArn()).add("DistributionDomainName", distributionDomainName())
                .add("DistributionHostedZoneId", distributionHostedZoneId())
                .add("EndpointConfiguration", endpointConfiguration()).add("DomainNameStatus", domainNameStatusAsString())
                .add("DomainNameStatusMessage", domainNameStatusMessage()).add("SecurityPolicy", securityPolicyAsString())
                .add("Tags", hasTags() ? tags() : null).add("MutualTlsAuthentication", mutualTlsAuthentication()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "certificateName":
            return Optional.ofNullable(clazz.cast(certificateName()));
        case "certificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "certificateUploadDate":
            return Optional.ofNullable(clazz.cast(certificateUploadDate()));
        case "regionalDomainName":
            return Optional.ofNullable(clazz.cast(regionalDomainName()));
        case "regionalHostedZoneId":
            return Optional.ofNullable(clazz.cast(regionalHostedZoneId()));
        case "regionalCertificateName":
            return Optional.ofNullable(clazz.cast(regionalCertificateName()));
        case "regionalCertificateArn":
            return Optional.ofNullable(clazz.cast(regionalCertificateArn()));
        case "distributionDomainName":
            return Optional.ofNullable(clazz.cast(distributionDomainName()));
        case "distributionHostedZoneId":
            return Optional.ofNullable(clazz.cast(distributionHostedZoneId()));
        case "endpointConfiguration":
            return Optional.ofNullable(clazz.cast(endpointConfiguration()));
        case "domainNameStatus":
            return Optional.ofNullable(clazz.cast(domainNameStatusAsString()));
        case "domainNameStatusMessage":
            return Optional.ofNullable(clazz.cast(domainNameStatusMessage()));
        case "securityPolicy":
            return Optional.ofNullable(clazz.cast(securityPolicyAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "mutualTlsAuthentication":
            return Optional.ofNullable(clazz.cast(mutualTlsAuthentication()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DomainName, T> g) {
        return obj -> g.apply((DomainName) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DomainName> {
        /**
         * <p>
         * The custom domain name as an API host name, for example, <code>my-api.example.com</code>.
         * </p>
         * 
         * @param domainName
         *        The custom domain name as an API host name, for example, <code>my-api.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The name of the certificate that will be used by edge-optimized endpoint for this domain name.
         * </p>
         * 
         * @param certificateName
         *        The name of the certificate that will be used by edge-optimized endpoint for this domain name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateName(String certificateName);

        /**
         * <p>
         * The reference to an AWS-managed certificate that will be used by edge-optimized endpoint for this domain
         * name. AWS Certificate Manager is the only supported source.
         * </p>
         * 
         * @param certificateArn
         *        The reference to an AWS-managed certificate that will be used by edge-optimized endpoint for this
         *        domain name. AWS Certificate Manager is the only supported source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateArn(String certificateArn);

        /**
         * <p>
         * The timestamp when the certificate that was used by edge-optimized endpoint for this domain name was
         * uploaded.
         * </p>
         * 
         * @param certificateUploadDate
         *        The timestamp when the certificate that was used by edge-optimized endpoint for this domain name was
         *        uploaded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateUploadDate(Instant certificateUploadDate);

        /**
         * <p>
         * The domain name associated with the regional endpoint for this custom domain name. You set up this
         * association by adding a DNS record that points the custom domain name to this regional domain name. The
         * regional domain name is returned by API Gateway when you create a regional endpoint.
         * </p>
         * 
         * @param regionalDomainName
         *        The domain name associated with the regional endpoint for this custom domain name. You set up this
         *        association by adding a DNS record that points the custom domain name to this regional domain name.
         *        The regional domain name is returned by API Gateway when you create a regional endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionalDomainName(String regionalDomainName);

        /**
         * <p>
         * The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint. For more information, see <a
         * href=
         * "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
         * >Set up a Regional Custom Domain Name</a> and <a
         * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and Endpoints
         * for API Gateway</a>.
         * </p>
         * 
         * @param regionalHostedZoneId
         *        The region-specific Amazon Route 53 Hosted Zone ID of the regional endpoint. For more information, see
         *        <a href=
         *        "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
         *        >Set up a Regional Custom Domain Name</a> and <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and
         *        Endpoints for API Gateway</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionalHostedZoneId(String regionalHostedZoneId);

        /**
         * <p>
         * The name of the certificate that will be used for validating the regional domain name.
         * </p>
         * 
         * @param regionalCertificateName
         *        The name of the certificate that will be used for validating the regional domain name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionalCertificateName(String regionalCertificateName);

        /**
         * <p>
         * The reference to an AWS-managed certificate that will be used for validating the regional domain name. AWS
         * Certificate Manager is the only supported source.
         * </p>
         * 
         * @param regionalCertificateArn
         *        The reference to an AWS-managed certificate that will be used for validating the regional domain name.
         *        AWS Certificate Manager is the only supported source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionalCertificateArn(String regionalCertificateArn);

        /**
         * <p>
         * The domain name of the Amazon CloudFront distribution associated with this custom domain name for an
         * edge-optimized endpoint. You set up this association when adding a DNS record pointing the custom domain name
         * to this distribution name. For more information about CloudFront distributions, see the <a
         * href="https://aws.amazon.com/documentation/cloudfront/" target="_blank">Amazon CloudFront documentation</a>.
         * </p>
         * 
         * @param distributionDomainName
         *        The domain name of the Amazon CloudFront distribution associated with this custom domain name for an
         *        edge-optimized endpoint. You set up this association when adding a DNS record pointing the custom
         *        domain name to this distribution name. For more information about CloudFront distributions, see the <a
         *        href="https://aws.amazon.com/documentation/cloudfront/" target="_blank">Amazon CloudFront
         *        documentation</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionDomainName(String distributionDomainName);

        /**
         * <p>
         * The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint. The valid value is
         * <code>Z2FDTNDATAQYW2</code> for all the regions. For more information, see <a href=
         * "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
         * >Set up a Regional Custom Domain Name</a> and <a
         * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and Endpoints
         * for API Gateway</a>.
         * </p>
         * 
         * @param distributionHostedZoneId
         *        The region-agnostic Amazon Route 53 Hosted Zone ID of the edge-optimized endpoint. The valid value is
         *        <code>Z2FDTNDATAQYW2</code> for all the regions. For more information, see <a href=
         *        "https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-regional-api-custom-domain-create.html"
         *        >Set up a Regional Custom Domain Name</a> and <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/rande.html#apigateway_region">AWS Regions and
         *        Endpoints for API Gateway</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distributionHostedZoneId(String distributionHostedZoneId);

        /**
         * <p>
         * The endpoint configuration of this <a>DomainName</a> showing the endpoint types of the domain name.
         * </p>
         * 
         * @param endpointConfiguration
         *        The endpoint configuration of this <a>DomainName</a> showing the endpoint types of the domain name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointConfiguration(EndpointConfiguration endpointConfiguration);

        /**
         * <p>
         * The endpoint configuration of this <a>DomainName</a> showing the endpoint types of the domain name.
         * </p>
         * This is a convenience that creates an instance of the {@link EndpointConfiguration.Builder} avoiding the need
         * to create one manually via {@link EndpointConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link EndpointConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #endpointConfiguration(EndpointConfiguration)}.
         * 
         * @param endpointConfiguration
         *        a consumer that will call methods on {@link EndpointConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpointConfiguration(EndpointConfiguration)
         */
        default Builder endpointConfiguration(Consumer<EndpointConfiguration.Builder> endpointConfiguration) {
            return endpointConfiguration(EndpointConfiguration.builder().applyMutation(endpointConfiguration).build());
        }

        /**
         * <p>
         * The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
         * <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further until
         * the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
         * </p>
         * 
         * @param domainNameStatus
         *        The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
         *        <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further
         *        until the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
         * @see DomainNameStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainNameStatus
         */
        Builder domainNameStatus(String domainNameStatus);

        /**
         * <p>
         * The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
         * <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further until
         * the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
         * </p>
         * 
         * @param domainNameStatus
         *        The status of the <a>DomainName</a> migration. The valid values are <code>AVAILABLE</code> and
         *        <code>UPDATING</code>. If the status is <code>UPDATING</code>, the domain cannot be modified further
         *        until the existing operation is complete. If it is <code>AVAILABLE</code>, the domain can be updated.
         * @see DomainNameStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainNameStatus
         */
        Builder domainNameStatus(DomainNameStatus domainNameStatus);

        /**
         * <p>
         * An optional text message containing detailed information about status of the <a>DomainName</a> migration.
         * </p>
         * 
         * @param domainNameStatusMessage
         *        An optional text message containing detailed information about status of the <a>DomainName</a>
         *        migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainNameStatusMessage(String domainNameStatusMessage);

        /**
         * <p>
         * The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values are
         * <code>TLS_1_0</code> and <code>TLS_1_2</code>.
         * </p>
         * 
         * @param securityPolicy
         *        The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values
         *        are <code>TLS_1_0</code> and <code>TLS_1_2</code>.
         * @see SecurityPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityPolicy
         */
        Builder securityPolicy(String securityPolicy);

        /**
         * <p>
         * The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values are
         * <code>TLS_1_0</code> and <code>TLS_1_2</code>.
         * </p>
         * 
         * @param securityPolicy
         *        The Transport Layer Security (TLS) version + cipher suite for this <a>DomainName</a>. The valid values
         *        are <code>TLS_1_0</code> and <code>TLS_1_2</code>.
         * @see SecurityPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SecurityPolicy
         */
        Builder securityPolicy(SecurityPolicy securityPolicy);

        /**
         * <p>
         * The collection of tags. Each tag element is associated with a given resource.
         * </p>
         * 
         * @param tags
         *        The collection of tags. Each tag element is associated with a given resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs
         * two-way authentication between the client and the server. Clients must present a trusted certificate to
         * access your API.
         * </p>
         * 
         * @param mutualTlsAuthentication
         *        The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway
         *        performs two-way authentication between the client and the server. Clients must present a trusted
         *        certificate to access your API.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mutualTlsAuthentication(MutualTlsAuthentication mutualTlsAuthentication);

        /**
         * <p>
         * The mutual TLS authentication configuration for a custom domain name. If specified, API Gateway performs
         * two-way authentication between the client and the server. Clients must present a trusted certificate to
         * access your API.
         * </p>
         * This is a convenience that creates an instance of the {@link MutualTlsAuthentication.Builder} avoiding the
         * need to create one manually via {@link MutualTlsAuthentication#builder()}.
         *
         * When the {@link Consumer} completes, {@link MutualTlsAuthentication.Builder#build()} is called immediately
         * and its result is passed to {@link #mutualTlsAuthentication(MutualTlsAuthentication)}.
         * 
         * @param mutualTlsAuthentication
         *        a consumer that will call methods on {@link MutualTlsAuthentication.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mutualTlsAuthentication(MutualTlsAuthentication)
         */
        default Builder mutualTlsAuthentication(Consumer<MutualTlsAuthentication.Builder> mutualTlsAuthentication) {
            return mutualTlsAuthentication(MutualTlsAuthentication.builder().applyMutation(mutualTlsAuthentication).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String domainName;

        private String certificateName;

        private String certificateArn;

        private Instant certificateUploadDate;

        private String regionalDomainName;

        private String regionalHostedZoneId;

        private String regionalCertificateName;

        private String regionalCertificateArn;

        private String distributionDomainName;

        private String distributionHostedZoneId;

        private EndpointConfiguration endpointConfiguration;

        private String domainNameStatus;

        private String domainNameStatusMessage;

        private String securityPolicy;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private MutualTlsAuthentication mutualTlsAuthentication;

        private BuilderImpl() {
        }

        private BuilderImpl(DomainName model) {
            domainName(model.domainName);
            certificateName(model.certificateName);
            certificateArn(model.certificateArn);
            certificateUploadDate(model.certificateUploadDate);
            regionalDomainName(model.regionalDomainName);
            regionalHostedZoneId(model.regionalHostedZoneId);
            regionalCertificateName(model.regionalCertificateName);
            regionalCertificateArn(model.regionalCertificateArn);
            distributionDomainName(model.distributionDomainName);
            distributionHostedZoneId(model.distributionHostedZoneId);
            endpointConfiguration(model.endpointConfiguration);
            domainNameStatus(model.domainNameStatus);
            domainNameStatusMessage(model.domainNameStatusMessage);
            securityPolicy(model.securityPolicy);
            tags(model.tags);
            mutualTlsAuthentication(model.mutualTlsAuthentication);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        @Transient
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getCertificateName() {
            return certificateName;
        }

        public final void setCertificateName(String certificateName) {
            this.certificateName = certificateName;
        }

        @Override
        @Transient
        public final Builder certificateName(String certificateName) {
            this.certificateName = certificateName;
            return this;
        }

        public final String getCertificateArn() {
            return certificateArn;
        }

        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        @Override
        @Transient
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final Instant getCertificateUploadDate() {
            return certificateUploadDate;
        }

        public final void setCertificateUploadDate(Instant certificateUploadDate) {
            this.certificateUploadDate = certificateUploadDate;
        }

        @Override
        @Transient
        public final Builder certificateUploadDate(Instant certificateUploadDate) {
            this.certificateUploadDate = certificateUploadDate;
            return this;
        }

        public final String getRegionalDomainName() {
            return regionalDomainName;
        }

        public final void setRegionalDomainName(String regionalDomainName) {
            this.regionalDomainName = regionalDomainName;
        }

        @Override
        @Transient
        public final Builder regionalDomainName(String regionalDomainName) {
            this.regionalDomainName = regionalDomainName;
            return this;
        }

        public final String getRegionalHostedZoneId() {
            return regionalHostedZoneId;
        }

        public final void setRegionalHostedZoneId(String regionalHostedZoneId) {
            this.regionalHostedZoneId = regionalHostedZoneId;
        }

        @Override
        @Transient
        public final Builder regionalHostedZoneId(String regionalHostedZoneId) {
            this.regionalHostedZoneId = regionalHostedZoneId;
            return this;
        }

        public final String getRegionalCertificateName() {
            return regionalCertificateName;
        }

        public final void setRegionalCertificateName(String regionalCertificateName) {
            this.regionalCertificateName = regionalCertificateName;
        }

        @Override
        @Transient
        public final Builder regionalCertificateName(String regionalCertificateName) {
            this.regionalCertificateName = regionalCertificateName;
            return this;
        }

        public final String getRegionalCertificateArn() {
            return regionalCertificateArn;
        }

        public final void setRegionalCertificateArn(String regionalCertificateArn) {
            this.regionalCertificateArn = regionalCertificateArn;
        }

        @Override
        @Transient
        public final Builder regionalCertificateArn(String regionalCertificateArn) {
            this.regionalCertificateArn = regionalCertificateArn;
            return this;
        }

        public final String getDistributionDomainName() {
            return distributionDomainName;
        }

        public final void setDistributionDomainName(String distributionDomainName) {
            this.distributionDomainName = distributionDomainName;
        }

        @Override
        @Transient
        public final Builder distributionDomainName(String distributionDomainName) {
            this.distributionDomainName = distributionDomainName;
            return this;
        }

        public final String getDistributionHostedZoneId() {
            return distributionHostedZoneId;
        }

        public final void setDistributionHostedZoneId(String distributionHostedZoneId) {
            this.distributionHostedZoneId = distributionHostedZoneId;
        }

        @Override
        @Transient
        public final Builder distributionHostedZoneId(String distributionHostedZoneId) {
            this.distributionHostedZoneId = distributionHostedZoneId;
            return this;
        }

        public final EndpointConfiguration.Builder getEndpointConfiguration() {
            return endpointConfiguration != null ? endpointConfiguration.toBuilder() : null;
        }

        public final void setEndpointConfiguration(EndpointConfiguration.BuilderImpl endpointConfiguration) {
            this.endpointConfiguration = endpointConfiguration != null ? endpointConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder endpointConfiguration(EndpointConfiguration endpointConfiguration) {
            this.endpointConfiguration = endpointConfiguration;
            return this;
        }

        public final String getDomainNameStatus() {
            return domainNameStatus;
        }

        public final void setDomainNameStatus(String domainNameStatus) {
            this.domainNameStatus = domainNameStatus;
        }

        @Override
        @Transient
        public final Builder domainNameStatus(String domainNameStatus) {
            this.domainNameStatus = domainNameStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder domainNameStatus(DomainNameStatus domainNameStatus) {
            this.domainNameStatus(domainNameStatus == null ? null : domainNameStatus.toString());
            return this;
        }

        public final String getDomainNameStatusMessage() {
            return domainNameStatusMessage;
        }

        public final void setDomainNameStatusMessage(String domainNameStatusMessage) {
            this.domainNameStatusMessage = domainNameStatusMessage;
        }

        @Override
        @Transient
        public final Builder domainNameStatusMessage(String domainNameStatusMessage) {
            this.domainNameStatusMessage = domainNameStatusMessage;
            return this;
        }

        public final String getSecurityPolicy() {
            return securityPolicy;
        }

        public final void setSecurityPolicy(String securityPolicy) {
            this.securityPolicy = securityPolicy;
        }

        @Override
        @Transient
        public final Builder securityPolicy(String securityPolicy) {
            this.securityPolicy = securityPolicy;
            return this;
        }

        @Override
        @Transient
        public final Builder securityPolicy(SecurityPolicy securityPolicy) {
            this.securityPolicy(securityPolicy == null ? null : securityPolicy.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = MapOfStringToStringCopier.copy(tags);
            return this;
        }

        public final MutualTlsAuthentication.Builder getMutualTlsAuthentication() {
            return mutualTlsAuthentication != null ? mutualTlsAuthentication.toBuilder() : null;
        }

        public final void setMutualTlsAuthentication(MutualTlsAuthentication.BuilderImpl mutualTlsAuthentication) {
            this.mutualTlsAuthentication = mutualTlsAuthentication != null ? mutualTlsAuthentication.build() : null;
        }

        @Override
        @Transient
        public final Builder mutualTlsAuthentication(MutualTlsAuthentication mutualTlsAuthentication) {
            this.mutualTlsAuthentication = mutualTlsAuthentication;
            return this;
        }

        @Override
        public DomainName build() {
            return new DomainName(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
