/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Sets up a method's integration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutIntegrationRequest extends ApiGatewayRequest implements
        ToCopyableBuilder<PutIntegrationRequest.Builder, PutIntegrationRequest> {
    private static final SdkField<String> REST_API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("restApiId").getter(getter(PutIntegrationRequest::restApiId)).setter(setter(Builder::restApiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("restapi_id").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceId").getter(getter(PutIntegrationRequest::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("resource_id").build()).build();

    private static final SdkField<String> HTTP_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("httpMethod").getter(getter(PutIntegrationRequest::httpMethod)).setter(setter(Builder::httpMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("http_method").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(PutIntegrationRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> INTEGRATION_HTTP_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("integrationHttpMethod").getter(getter(PutIntegrationRequest::integrationHttpMethod))
            .setter(setter(Builder::integrationHttpMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("httpMethod").build()).build();

    private static final SdkField<String> URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("uri")
            .getter(getter(PutIntegrationRequest::uri)).setter(setter(Builder::uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("uri").build()).build();

    private static final SdkField<String> CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("connectionType").getter(getter(PutIntegrationRequest::connectionTypeAsString))
            .setter(setter(Builder::connectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectionType").build()).build();

    private static final SdkField<String> CONNECTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("connectionId").getter(getter(PutIntegrationRequest::connectionId)).setter(setter(Builder::connectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("connectionId").build()).build();

    private static final SdkField<String> CREDENTIALS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("credentials").getter(getter(PutIntegrationRequest::credentials)).setter(setter(Builder::credentials))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("credentials").build()).build();

    private static final SdkField<Map<String, String>> REQUEST_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("requestParameters")
            .getter(getter(PutIntegrationRequest::requestParameters))
            .setter(setter(Builder::requestParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> REQUEST_TEMPLATES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("requestTemplates")
            .getter(getter(PutIntegrationRequest::requestTemplates))
            .setter(setter(Builder::requestTemplates))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestTemplates").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> PASSTHROUGH_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("passthroughBehavior").getter(getter(PutIntegrationRequest::passthroughBehavior))
            .setter(setter(Builder::passthroughBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("passthroughBehavior").build())
            .build();

    private static final SdkField<String> CACHE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("cacheNamespace").getter(getter(PutIntegrationRequest::cacheNamespace))
            .setter(setter(Builder::cacheNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cacheNamespace").build()).build();

    private static final SdkField<List<String>> CACHE_KEY_PARAMETERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cacheKeyParameters")
            .getter(getter(PutIntegrationRequest::cacheKeyParameters))
            .setter(setter(Builder::cacheKeyParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cacheKeyParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONTENT_HANDLING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("contentHandling").getter(getter(PutIntegrationRequest::contentHandlingAsString))
            .setter(setter(Builder::contentHandling))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contentHandling").build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_MILLIS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInMillis").getter(getter(PutIntegrationRequest::timeoutInMillis))
            .setter(setter(Builder::timeoutInMillis))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInMillis").build()).build();

    private static final SdkField<TlsConfig> TLS_CONFIG_FIELD = SdkField.<TlsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("tlsConfig").getter(getter(PutIntegrationRequest::tlsConfig)).setter(setter(Builder::tlsConfig))
            .constructor(TlsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tlsConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REST_API_ID_FIELD,
            RESOURCE_ID_FIELD, HTTP_METHOD_FIELD, TYPE_FIELD, INTEGRATION_HTTP_METHOD_FIELD, URI_FIELD, CONNECTION_TYPE_FIELD,
            CONNECTION_ID_FIELD, CREDENTIALS_FIELD, REQUEST_PARAMETERS_FIELD, REQUEST_TEMPLATES_FIELD,
            PASSTHROUGH_BEHAVIOR_FIELD, CACHE_NAMESPACE_FIELD, CACHE_KEY_PARAMETERS_FIELD, CONTENT_HANDLING_FIELD,
            TIMEOUT_IN_MILLIS_FIELD, TLS_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String restApiId;

    private final String resourceId;

    private final String httpMethod;

    private final String type;

    private final String integrationHttpMethod;

    private final String uri;

    private final String connectionType;

    private final String connectionId;

    private final String credentials;

    private final Map<String, String> requestParameters;

    private final Map<String, String> requestTemplates;

    private final String passthroughBehavior;

    private final String cacheNamespace;

    private final List<String> cacheKeyParameters;

    private final String contentHandling;

    private final Integer timeoutInMillis;

    private final TlsConfig tlsConfig;

    private PutIntegrationRequest(BuilderImpl builder) {
        super(builder);
        this.restApiId = builder.restApiId;
        this.resourceId = builder.resourceId;
        this.httpMethod = builder.httpMethod;
        this.type = builder.type;
        this.integrationHttpMethod = builder.integrationHttpMethod;
        this.uri = builder.uri;
        this.connectionType = builder.connectionType;
        this.connectionId = builder.connectionId;
        this.credentials = builder.credentials;
        this.requestParameters = builder.requestParameters;
        this.requestTemplates = builder.requestTemplates;
        this.passthroughBehavior = builder.passthroughBehavior;
        this.cacheNamespace = builder.cacheNamespace;
        this.cacheKeyParameters = builder.cacheKeyParameters;
        this.contentHandling = builder.contentHandling;
        this.timeoutInMillis = builder.timeoutInMillis;
        this.tlsConfig = builder.tlsConfig;
    }

    /**
     * <p>
     * The string identifier of the associated RestApi.
     * </p>
     * 
     * @return The string identifier of the associated RestApi.
     */
    public final String restApiId() {
        return restApiId;
    }

    /**
     * <p>
     * Specifies a put integration request's resource ID.
     * </p>
     * 
     * @return Specifies a put integration request's resource ID.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * Specifies the HTTP method for the integration.
     * </p>
     * 
     * @return Specifies the HTTP method for the integration.
     */
    public final String httpMethod() {
        return httpMethod;
    }

    /**
     * <p>
     * Specifies a put integration input's type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IntegrationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies a put integration input's type.
     * @see IntegrationType
     */
    public final IntegrationType type() {
        return IntegrationType.fromValue(type);
    }

    /**
     * <p>
     * Specifies a put integration input's type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IntegrationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies a put integration input's type.
     * @see IntegrationType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The HTTP method for the integration.
     * </p>
     * 
     * @return The HTTP method for the integration.
     */
    public final String integrationHttpMethod() {
        return integrationHttpMethod;
    }

    /**
     * <p>
     * Specifies Uniform Resource Identifier (URI) of the integration endpoint. For HTTP or <code>HTTP_PROXY</code>
     * integrations, the URI must be a fully formed, encoded HTTP(S) URL according to the RFC-3986 specification, for
     * either standard integration, where <code>connectionType</code> is not <code>VPC_LINK</code>, or private
     * integration, where <code>connectionType</code> is <code>VPC_LINK</code>. For a private HTTP integration, the URI
     * is not used for routing. For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
     * <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api</code> . Here, {Region} is
     * the API Gateway region (e.g., us-east-1); {service} is the name of the integrated Amazon Web Services service
     * (e.g., s3); and {subdomain} is a designated subdomain supported by certain Amazon Web Services service for fast
     * host-name lookup. action can be used for an Amazon Web Services service action-based API, using an
     * Action={name}&amp;{p1}={v1}&amp;p2={v2}... query string. The ensuing {service_api} refers to a supported action
     * {name} plus any required input parameters. Alternatively, path can be used for an Amazon Web Services service
     * path-based API. The ensuing service_api refers to the path to an Amazon Web Services service resource, including
     * the region of the integrated Amazon Web Services service, if applicable. For example, for integration with the S3
     * API of <code>GetObject</code>, the <code>uri</code> can be either
     * <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
     * <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>.
     * </p>
     * 
     * @return Specifies Uniform Resource Identifier (URI) of the integration endpoint. For HTTP or
     *         <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded HTTP(S) URL according to
     *         the RFC-3986 specification, for either standard integration, where <code>connectionType</code> is not
     *         <code>VPC_LINK</code>, or private integration, where <code>connectionType</code> is <code>VPC_LINK</code>
     *         . For a private HTTP integration, the URI is not used for routing. For <code>AWS</code> or
     *         <code>AWS_PROXY</code> integrations, the URI is of the form
     *         <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api</code> . Here,
     *         {Region} is the API Gateway region (e.g., us-east-1); {service} is the name of the integrated Amazon Web
     *         Services service (e.g., s3); and {subdomain} is a designated subdomain supported by certain Amazon Web
     *         Services service for fast host-name lookup. action can be used for an Amazon Web Services service
     *         action-based API, using an Action={name}&amp;{p1}={v1}&amp;p2={v2}... query string. The ensuing
     *         {service_api} refers to a supported action {name} plus any required input parameters. Alternatively, path
     *         can be used for an Amazon Web Services service path-based API. The ensuing service_api refers to the path
     *         to an Amazon Web Services service resource, including the region of the integrated Amazon Web Services
     *         service, if applicable. For example, for integration with the S3 API of <code>GetObject</code>, the
     *         <code>uri</code> can be either
     *         <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
     *         <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>.
     */
    public final String uri() {
        return uri;
    }

    /**
     * <p>
     * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
     * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
     * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code>
     *         for connections through the public routable internet or <code>VPC_LINK</code> for private connections
     *         between API Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * @see ConnectionType
     */
    public final ConnectionType connectionType() {
        return ConnectionType.fromValue(connectionType);
    }

    /**
     * <p>
     * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
     * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
     * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #connectionType}
     * will return {@link ConnectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #connectionTypeAsString}.
     * </p>
     * 
     * @return The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code>
     *         for connections through the public routable internet or <code>VPC_LINK</code> for private connections
     *         between API Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
     * @see ConnectionType
     */
    public final String connectionTypeAsString() {
        return connectionType;
    }

    /**
     * <p>
     * The ID of the VpcLink used for the integration. Specify this value only if you specify <code>VPC_LINK</code> as
     * the connection type.
     * </p>
     * 
     * @return The ID of the VpcLink used for the integration. Specify this value only if you specify
     *         <code>VPC_LINK</code> as the connection type.
     */
    public final String connectionId() {
        return connectionId;
    }

    /**
     * <p>
     * Specifies whether credentials are required for a put integration.
     * </p>
     * 
     * @return Specifies whether credentials are required for a put integration.
     */
    public final String credentials() {
        return credentials;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRequestParameters() {
        return requestParameters != null && !(requestParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key-value map specifying request parameters that are passed from the method request to the back end. The key is
     * an integration request parameter name and the associated value is a method request parameter value or static
     * value that must be enclosed within single quotes and pre-encoded as required by the back end. The method request
     * parameter value must match the pattern of <code>method.request.{location}.{name}</code>, where
     * <code>location</code> is <code>querystring</code>, <code>path</code>, or <code>header</code> and
     * <code>name</code> must be a valid and unique method request parameter name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestParameters} method.
     * </p>
     * 
     * @return A key-value map specifying request parameters that are passed from the method request to the back end.
     *         The key is an integration request parameter name and the associated value is a method request parameter
     *         value or static value that must be enclosed within single quotes and pre-encoded as required by the back
     *         end. The method request parameter value must match the pattern of
     *         <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>,
     *         <code>path</code>, or <code>header</code> and <code>name</code> must be a valid and unique method request
     *         parameter name.
     */
    public final Map<String, String> requestParameters() {
        return requestParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestTemplates property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRequestTemplates() {
        return requestTemplates != null && !(requestTemplates instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Represents a map of Velocity templates that are applied on the request payload based on the value of the
     * Content-Type header sent by the client. The content type value is the key in this map, and the template (as a
     * String) is the value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestTemplates} method.
     * </p>
     * 
     * @return Represents a map of Velocity templates that are applied on the request payload based on the value of the
     *         Content-Type header sent by the client. The content type value is the key in this map, and the template
     *         (as a String) is the value.
     */
    public final Map<String, String> requestTemplates() {
        return requestTemplates;
    }

    /**
     * <p>
     * Specifies the pass-through behavior for incoming requests based on the Content-Type header in the request, and
     * the available mapping templates specified as the <code>requestTemplates</code> property on the Integration
     * resource. There are three valid values: <code>WHEN_NO_MATCH</code>, <code>WHEN_NO_TEMPLATES</code>, and
     * <code>NEVER</code>.
     * </p>
     * 
     * @return Specifies the pass-through behavior for incoming requests based on the Content-Type header in the
     *         request, and the available mapping templates specified as the <code>requestTemplates</code> property on
     *         the Integration resource. There are three valid values: <code>WHEN_NO_MATCH</code>,
     *         <code>WHEN_NO_TEMPLATES</code>, and <code>NEVER</code>.
     */
    public final String passthroughBehavior() {
        return passthroughBehavior;
    }

    /**
     * <p>
     * Specifies a group of related cached parameters. By default, API Gateway uses the resource ID as the
     * <code>cacheNamespace</code>. You can specify the same <code>cacheNamespace</code> across resources to return the
     * same cached data for requests to different resources.
     * </p>
     * 
     * @return Specifies a group of related cached parameters. By default, API Gateway uses the resource ID as the
     *         <code>cacheNamespace</code>. You can specify the same <code>cacheNamespace</code> across resources to
     *         return the same cached data for requests to different resources.
     */
    public final String cacheNamespace() {
        return cacheNamespace;
    }

    /**
     * For responses, this returns true if the service returned a value for the CacheKeyParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCacheKeyParameters() {
        return cacheKeyParameters != null && !(cacheKeyParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of request parameters whose values API Gateway caches. To be valid values for
     * <code>cacheKeyParameters</code>, these parameters must also be specified for Method
     * <code>requestParameters</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCacheKeyParameters} method.
     * </p>
     * 
     * @return A list of request parameters whose values API Gateway caches. To be valid values for
     *         <code>cacheKeyParameters</code>, these parameters must also be specified for Method
     *         <code>requestParameters</code>.
     */
    public final List<String> cacheKeyParameters() {
        return cacheKeyParameters;
    }

    /**
     * <p>
     * Specifies how to handle request payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <p>
     * If this property is not defined, the request payload will be passed through from the method request to
     * integration request without modification, provided that the <code>passthroughBehavior</code> is configured to
     * support payload pass-through.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle request payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <p>
     *         If this property is not defined, the request payload will be passed through from the method request to
     *         integration request without modification, provided that the <code>passthroughBehavior</code> is
     *         configured to support payload pass-through.
     * @see ContentHandlingStrategy
     */
    public final ContentHandlingStrategy contentHandling() {
        return ContentHandlingStrategy.fromValue(contentHandling);
    }

    /**
     * <p>
     * Specifies how to handle request payload content type conversions. Supported values are
     * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
     * </p>
     * <p>
     * If this property is not defined, the request payload will be passed through from the method request to
     * integration request without modification, provided that the <code>passthroughBehavior</code> is configured to
     * support payload pass-through.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentHandling}
     * will return {@link ContentHandlingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #contentHandlingAsString}.
     * </p>
     * 
     * @return Specifies how to handle request payload content type conversions. Supported values are
     *         <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
     *         <p>
     *         If this property is not defined, the request payload will be passed through from the method request to
     *         integration request without modification, provided that the <code>passthroughBehavior</code> is
     *         configured to support payload pass-through.
     * @see ContentHandlingStrategy
     */
    public final String contentHandlingAsString() {
        return contentHandling;
    }

    /**
     * <p>
     * Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
     * </p>
     * 
     * @return Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29
     *         seconds.
     */
    public final Integer timeoutInMillis() {
        return timeoutInMillis;
    }

    /**
     * Returns the value of the TlsConfig property for this object.
     * 
     * @return The value of the TlsConfig property for this object.
     */
    public final TlsConfig tlsConfig() {
        return tlsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(restApiId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(httpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(integrationHttpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(uri());
        hashCode = 31 * hashCode + Objects.hashCode(connectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(connectionId());
        hashCode = 31 * hashCode + Objects.hashCode(credentials());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestParameters() ? requestParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestTemplates() ? requestTemplates() : null);
        hashCode = 31 * hashCode + Objects.hashCode(passthroughBehavior());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNamespace());
        hashCode = 31 * hashCode + Objects.hashCode(hasCacheKeyParameters() ? cacheKeyParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contentHandlingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMillis());
        hashCode = 31 * hashCode + Objects.hashCode(tlsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutIntegrationRequest)) {
            return false;
        }
        PutIntegrationRequest other = (PutIntegrationRequest) obj;
        return Objects.equals(restApiId(), other.restApiId()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(httpMethod(), other.httpMethod()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(integrationHttpMethod(), other.integrationHttpMethod()) && Objects.equals(uri(), other.uri())
                && Objects.equals(connectionTypeAsString(), other.connectionTypeAsString())
                && Objects.equals(connectionId(), other.connectionId()) && Objects.equals(credentials(), other.credentials())
                && hasRequestParameters() == other.hasRequestParameters()
                && Objects.equals(requestParameters(), other.requestParameters())
                && hasRequestTemplates() == other.hasRequestTemplates()
                && Objects.equals(requestTemplates(), other.requestTemplates())
                && Objects.equals(passthroughBehavior(), other.passthroughBehavior())
                && Objects.equals(cacheNamespace(), other.cacheNamespace())
                && hasCacheKeyParameters() == other.hasCacheKeyParameters()
                && Objects.equals(cacheKeyParameters(), other.cacheKeyParameters())
                && Objects.equals(contentHandlingAsString(), other.contentHandlingAsString())
                && Objects.equals(timeoutInMillis(), other.timeoutInMillis()) && Objects.equals(tlsConfig(), other.tlsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutIntegrationRequest").add("RestApiId", restApiId()).add("ResourceId", resourceId())
                .add("HttpMethod", httpMethod()).add("Type", typeAsString())
                .add("IntegrationHttpMethod", integrationHttpMethod()).add("Uri", uri())
                .add("ConnectionType", connectionTypeAsString()).add("ConnectionId", connectionId())
                .add("Credentials", credentials()).add("RequestParameters", hasRequestParameters() ? requestParameters() : null)
                .add("RequestTemplates", hasRequestTemplates() ? requestTemplates() : null)
                .add("PassthroughBehavior", passthroughBehavior()).add("CacheNamespace", cacheNamespace())
                .add("CacheKeyParameters", hasCacheKeyParameters() ? cacheKeyParameters() : null)
                .add("ContentHandling", contentHandlingAsString()).add("TimeoutInMillis", timeoutInMillis())
                .add("TlsConfig", tlsConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "restApiId":
            return Optional.ofNullable(clazz.cast(restApiId()));
        case "resourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "httpMethod":
            return Optional.ofNullable(clazz.cast(httpMethod()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "integrationHttpMethod":
            return Optional.ofNullable(clazz.cast(integrationHttpMethod()));
        case "uri":
            return Optional.ofNullable(clazz.cast(uri()));
        case "connectionType":
            return Optional.ofNullable(clazz.cast(connectionTypeAsString()));
        case "connectionId":
            return Optional.ofNullable(clazz.cast(connectionId()));
        case "credentials":
            return Optional.ofNullable(clazz.cast(credentials()));
        case "requestParameters":
            return Optional.ofNullable(clazz.cast(requestParameters()));
        case "requestTemplates":
            return Optional.ofNullable(clazz.cast(requestTemplates()));
        case "passthroughBehavior":
            return Optional.ofNullable(clazz.cast(passthroughBehavior()));
        case "cacheNamespace":
            return Optional.ofNullable(clazz.cast(cacheNamespace()));
        case "cacheKeyParameters":
            return Optional.ofNullable(clazz.cast(cacheKeyParameters()));
        case "contentHandling":
            return Optional.ofNullable(clazz.cast(contentHandlingAsString()));
        case "timeoutInMillis":
            return Optional.ofNullable(clazz.cast(timeoutInMillis()));
        case "tlsConfig":
            return Optional.ofNullable(clazz.cast(tlsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("restapi_id", REST_API_ID_FIELD);
        map.put("resource_id", RESOURCE_ID_FIELD);
        map.put("http_method", HTTP_METHOD_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("httpMethod", INTEGRATION_HTTP_METHOD_FIELD);
        map.put("uri", URI_FIELD);
        map.put("connectionType", CONNECTION_TYPE_FIELD);
        map.put("connectionId", CONNECTION_ID_FIELD);
        map.put("credentials", CREDENTIALS_FIELD);
        map.put("requestParameters", REQUEST_PARAMETERS_FIELD);
        map.put("requestTemplates", REQUEST_TEMPLATES_FIELD);
        map.put("passthroughBehavior", PASSTHROUGH_BEHAVIOR_FIELD);
        map.put("cacheNamespace", CACHE_NAMESPACE_FIELD);
        map.put("cacheKeyParameters", CACHE_KEY_PARAMETERS_FIELD);
        map.put("contentHandling", CONTENT_HANDLING_FIELD);
        map.put("timeoutInMillis", TIMEOUT_IN_MILLIS_FIELD);
        map.put("tlsConfig", TLS_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PutIntegrationRequest, T> g) {
        return obj -> g.apply((PutIntegrationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutIntegrationRequest> {
        /**
         * <p>
         * The string identifier of the associated RestApi.
         * </p>
         * 
         * @param restApiId
         *        The string identifier of the associated RestApi.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restApiId(String restApiId);

        /**
         * <p>
         * Specifies a put integration request's resource ID.
         * </p>
         * 
         * @param resourceId
         *        Specifies a put integration request's resource ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * Specifies the HTTP method for the integration.
         * </p>
         * 
         * @param httpMethod
         *        Specifies the HTTP method for the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpMethod(String httpMethod);

        /**
         * <p>
         * Specifies a put integration input's type.
         * </p>
         * 
         * @param type
         *        Specifies a put integration input's type.
         * @see IntegrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies a put integration input's type.
         * </p>
         * 
         * @param type
         *        Specifies a put integration input's type.
         * @see IntegrationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntegrationType
         */
        Builder type(IntegrationType type);

        /**
         * <p>
         * The HTTP method for the integration.
         * </p>
         * 
         * @param integrationHttpMethod
         *        The HTTP method for the integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder integrationHttpMethod(String integrationHttpMethod);

        /**
         * <p>
         * Specifies Uniform Resource Identifier (URI) of the integration endpoint. For HTTP or <code>HTTP_PROXY</code>
         * integrations, the URI must be a fully formed, encoded HTTP(S) URL according to the RFC-3986 specification,
         * for either standard integration, where <code>connectionType</code> is not <code>VPC_LINK</code>, or private
         * integration, where <code>connectionType</code> is <code>VPC_LINK</code>. For a private HTTP integration, the
         * URI is not used for routing. For <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the
         * form <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api</code> . Here,
         * {Region} is the API Gateway region (e.g., us-east-1); {service} is the name of the integrated Amazon Web
         * Services service (e.g., s3); and {subdomain} is a designated subdomain supported by certain Amazon Web
         * Services service for fast host-name lookup. action can be used for an Amazon Web Services service
         * action-based API, using an Action={name}&amp;{p1}={v1}&amp;p2={v2}... query string. The ensuing {service_api}
         * refers to a supported action {name} plus any required input parameters. Alternatively, path can be used for
         * an Amazon Web Services service path-based API. The ensuing service_api refers to the path to an Amazon Web
         * Services service resource, including the region of the integrated Amazon Web Services service, if applicable.
         * For example, for integration with the S3 API of <code>GetObject</code>, the <code>uri</code> can be either
         * <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
         * <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>.
         * </p>
         * 
         * @param uri
         *        Specifies Uniform Resource Identifier (URI) of the integration endpoint. For HTTP or
         *        <code>HTTP_PROXY</code> integrations, the URI must be a fully formed, encoded HTTP(S) URL according to
         *        the RFC-3986 specification, for either standard integration, where <code>connectionType</code> is not
         *        <code>VPC_LINK</code>, or private integration, where <code>connectionType</code> is
         *        <code>VPC_LINK</code>. For a private HTTP integration, the URI is not used for routing. For
         *        <code>AWS</code> or <code>AWS_PROXY</code> integrations, the URI is of the form
         *        <code>arn:aws:apigateway:{region}:{subdomain.service|service}:path|action/{service_api</code> . Here,
         *        {Region} is the API Gateway region (e.g., us-east-1); {service} is the name of the integrated Amazon
         *        Web Services service (e.g., s3); and {subdomain} is a designated subdomain supported by certain Amazon
         *        Web Services service for fast host-name lookup. action can be used for an Amazon Web Services service
         *        action-based API, using an Action={name}&amp;{p1}={v1}&amp;p2={v2}... query string. The ensuing
         *        {service_api} refers to a supported action {name} plus any required input parameters. Alternatively,
         *        path can be used for an Amazon Web Services service path-based API. The ensuing service_api refers to
         *        the path to an Amazon Web Services service resource, including the region of the integrated Amazon Web
         *        Services service, if applicable. For example, for integration with the S3 API of
         *        <code>GetObject</code>, the <code>uri</code> can be either
         *        <code>arn:aws:apigateway:us-west-2:s3:action/GetObject&amp;Bucket={bucket}&amp;Key={key}</code> or
         *        <code>arn:aws:apigateway:us-west-2:s3:path/{bucket}/{key}</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uri(String uri);

        /**
         * <p>
         * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
         * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
         * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
         * </p>
         * 
         * @param connectionType
         *        The type of the network connection to the integration endpoint. The valid value is
         *        <code>INTERNET</code> for connections through the public routable internet or <code>VPC_LINK</code>
         *        for private connections between API Gateway and a network load balancer in a VPC. The default value is
         *        <code>INTERNET</code>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * The type of the network connection to the integration endpoint. The valid value is <code>INTERNET</code> for
         * connections through the public routable internet or <code>VPC_LINK</code> for private connections between API
         * Gateway and a network load balancer in a VPC. The default value is <code>INTERNET</code>.
         * </p>
         * 
         * @param connectionType
         *        The type of the network connection to the integration endpoint. The valid value is
         *        <code>INTERNET</code> for connections through the public routable internet or <code>VPC_LINK</code>
         *        for private connections between API Gateway and a network load balancer in a VPC. The default value is
         *        <code>INTERNET</code>.
         * @see ConnectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionType
         */
        Builder connectionType(ConnectionType connectionType);

        /**
         * <p>
         * The ID of the VpcLink used for the integration. Specify this value only if you specify <code>VPC_LINK</code>
         * as the connection type.
         * </p>
         * 
         * @param connectionId
         *        The ID of the VpcLink used for the integration. Specify this value only if you specify
         *        <code>VPC_LINK</code> as the connection type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionId(String connectionId);

        /**
         * <p>
         * Specifies whether credentials are required for a put integration.
         * </p>
         * 
         * @param credentials
         *        Specifies whether credentials are required for a put integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder credentials(String credentials);

        /**
         * <p>
         * A key-value map specifying request parameters that are passed from the method request to the back end. The
         * key is an integration request parameter name and the associated value is a method request parameter value or
         * static value that must be enclosed within single quotes and pre-encoded as required by the back end. The
         * method request parameter value must match the pattern of <code>method.request.{location}.{name}</code>, where
         * <code>location</code> is <code>querystring</code>, <code>path</code>, or <code>header</code> and
         * <code>name</code> must be a valid and unique method request parameter name.
         * </p>
         * 
         * @param requestParameters
         *        A key-value map specifying request parameters that are passed from the method request to the back end.
         *        The key is an integration request parameter name and the associated value is a method request
         *        parameter value or static value that must be enclosed within single quotes and pre-encoded as required
         *        by the back end. The method request parameter value must match the pattern of
         *        <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>
         *        , <code>path</code>, or <code>header</code> and <code>name</code> must be a valid and unique method
         *        request parameter name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestParameters(Map<String, String> requestParameters);

        /**
         * <p>
         * Represents a map of Velocity templates that are applied on the request payload based on the value of the
         * Content-Type header sent by the client. The content type value is the key in this map, and the template (as a
         * String) is the value.
         * </p>
         * 
         * @param requestTemplates
         *        Represents a map of Velocity templates that are applied on the request payload based on the value of
         *        the Content-Type header sent by the client. The content type value is the key in this map, and the
         *        template (as a String) is the value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestTemplates(Map<String, String> requestTemplates);

        /**
         * <p>
         * Specifies the pass-through behavior for incoming requests based on the Content-Type header in the request,
         * and the available mapping templates specified as the <code>requestTemplates</code> property on the
         * Integration resource. There are three valid values: <code>WHEN_NO_MATCH</code>,
         * <code>WHEN_NO_TEMPLATES</code>, and <code>NEVER</code>.
         * </p>
         * 
         * @param passthroughBehavior
         *        Specifies the pass-through behavior for incoming requests based on the Content-Type header in the
         *        request, and the available mapping templates specified as the <code>requestTemplates</code> property
         *        on the Integration resource. There are three valid values: <code>WHEN_NO_MATCH</code>,
         *        <code>WHEN_NO_TEMPLATES</code>, and <code>NEVER</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder passthroughBehavior(String passthroughBehavior);

        /**
         * <p>
         * Specifies a group of related cached parameters. By default, API Gateway uses the resource ID as the
         * <code>cacheNamespace</code>. You can specify the same <code>cacheNamespace</code> across resources to return
         * the same cached data for requests to different resources.
         * </p>
         * 
         * @param cacheNamespace
         *        Specifies a group of related cached parameters. By default, API Gateway uses the resource ID as the
         *        <code>cacheNamespace</code>. You can specify the same <code>cacheNamespace</code> across resources to
         *        return the same cached data for requests to different resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNamespace(String cacheNamespace);

        /**
         * <p>
         * A list of request parameters whose values API Gateway caches. To be valid values for
         * <code>cacheKeyParameters</code>, these parameters must also be specified for Method
         * <code>requestParameters</code>.
         * </p>
         * 
         * @param cacheKeyParameters
         *        A list of request parameters whose values API Gateway caches. To be valid values for
         *        <code>cacheKeyParameters</code>, these parameters must also be specified for Method
         *        <code>requestParameters</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheKeyParameters(Collection<String> cacheKeyParameters);

        /**
         * <p>
         * A list of request parameters whose values API Gateway caches. To be valid values for
         * <code>cacheKeyParameters</code>, these parameters must also be specified for Method
         * <code>requestParameters</code>.
         * </p>
         * 
         * @param cacheKeyParameters
         *        A list of request parameters whose values API Gateway caches. To be valid values for
         *        <code>cacheKeyParameters</code>, these parameters must also be specified for Method
         *        <code>requestParameters</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheKeyParameters(String... cacheKeyParameters);

        /**
         * <p>
         * Specifies how to handle request payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <p>
         * If this property is not defined, the request payload will be passed through from the method request to
         * integration request without modification, provided that the <code>passthroughBehavior</code> is configured to
         * support payload pass-through.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle request payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <p>
         *        If this property is not defined, the request payload will be passed through from the method request to
         *        integration request without modification, provided that the <code>passthroughBehavior</code> is
         *        configured to support payload pass-through.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(String contentHandling);

        /**
         * <p>
         * Specifies how to handle request payload content type conversions. Supported values are
         * <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:
         * </p>
         * <p>
         * If this property is not defined, the request payload will be passed through from the method request to
         * integration request without modification, provided that the <code>passthroughBehavior</code> is configured to
         * support payload pass-through.
         * </p>
         * 
         * @param contentHandling
         *        Specifies how to handle request payload content type conversions. Supported values are
         *        <code>CONVERT_TO_BINARY</code> and <code>CONVERT_TO_TEXT</code>, with the following behaviors:</p>
         *        <p>
         *        If this property is not defined, the request payload will be passed through from the method request to
         *        integration request without modification, provided that the <code>passthroughBehavior</code> is
         *        configured to support payload pass-through.
         * @see ContentHandlingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentHandlingStrategy
         */
        Builder contentHandling(ContentHandlingStrategy contentHandling);

        /**
         * <p>
         * Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29 seconds.
         * </p>
         * 
         * @param timeoutInMillis
         *        Custom timeout between 50 and 29,000 milliseconds. The default value is 29,000 milliseconds or 29
         *        seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMillis(Integer timeoutInMillis);

        /**
         * Sets the value of the TlsConfig property for this object.
         *
         * @param tlsConfig
         *        The new value for the TlsConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tlsConfig(TlsConfig tlsConfig);

        /**
         * Sets the value of the TlsConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link TlsConfig.Builder} avoiding the need to
         * create one manually via {@link TlsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TlsConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #tlsConfig(TlsConfig)}.
         * 
         * @param tlsConfig
         *        a consumer that will call methods on {@link TlsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tlsConfig(TlsConfig)
         */
        default Builder tlsConfig(Consumer<TlsConfig.Builder> tlsConfig) {
            return tlsConfig(TlsConfig.builder().applyMutation(tlsConfig).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayRequest.BuilderImpl implements Builder {
        private String restApiId;

        private String resourceId;

        private String httpMethod;

        private String type;

        private String integrationHttpMethod;

        private String uri;

        private String connectionType;

        private String connectionId;

        private String credentials;

        private Map<String, String> requestParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> requestTemplates = DefaultSdkAutoConstructMap.getInstance();

        private String passthroughBehavior;

        private String cacheNamespace;

        private List<String> cacheKeyParameters = DefaultSdkAutoConstructList.getInstance();

        private String contentHandling;

        private Integer timeoutInMillis;

        private TlsConfig tlsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(PutIntegrationRequest model) {
            super(model);
            restApiId(model.restApiId);
            resourceId(model.resourceId);
            httpMethod(model.httpMethod);
            type(model.type);
            integrationHttpMethod(model.integrationHttpMethod);
            uri(model.uri);
            connectionType(model.connectionType);
            connectionId(model.connectionId);
            credentials(model.credentials);
            requestParameters(model.requestParameters);
            requestTemplates(model.requestTemplates);
            passthroughBehavior(model.passthroughBehavior);
            cacheNamespace(model.cacheNamespace);
            cacheKeyParameters(model.cacheKeyParameters);
            contentHandling(model.contentHandling);
            timeoutInMillis(model.timeoutInMillis);
            tlsConfig(model.tlsConfig);
        }

        public final String getRestApiId() {
            return restApiId;
        }

        public final void setRestApiId(String restApiId) {
            this.restApiId = restApiId;
        }

        @Override
        public final Builder restApiId(String restApiId) {
            this.restApiId = restApiId;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getHttpMethod() {
            return httpMethod;
        }

        public final void setHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
        }

        @Override
        public final Builder httpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(IntegrationType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getIntegrationHttpMethod() {
            return integrationHttpMethod;
        }

        public final void setIntegrationHttpMethod(String integrationHttpMethod) {
            this.integrationHttpMethod = integrationHttpMethod;
        }

        @Override
        public final Builder integrationHttpMethod(String integrationHttpMethod) {
            this.integrationHttpMethod = integrationHttpMethod;
            return this;
        }

        public final String getUri() {
            return uri;
        }

        public final void setUri(String uri) {
            this.uri = uri;
        }

        @Override
        public final Builder uri(String uri) {
            this.uri = uri;
            return this;
        }

        public final String getConnectionType() {
            return connectionType;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        @Override
        public final Builder connectionType(ConnectionType connectionType) {
            this.connectionType(connectionType == null ? null : connectionType.toString());
            return this;
        }

        public final String getConnectionId() {
            return connectionId;
        }

        public final void setConnectionId(String connectionId) {
            this.connectionId = connectionId;
        }

        @Override
        public final Builder connectionId(String connectionId) {
            this.connectionId = connectionId;
            return this;
        }

        public final String getCredentials() {
            return credentials;
        }

        public final void setCredentials(String credentials) {
            this.credentials = credentials;
        }

        @Override
        public final Builder credentials(String credentials) {
            this.credentials = credentials;
            return this;
        }

        public final Map<String, String> getRequestParameters() {
            if (requestParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return requestParameters;
        }

        public final void setRequestParameters(Map<String, String> requestParameters) {
            this.requestParameters = MapOfStringToStringCopier.copy(requestParameters);
        }

        @Override
        public final Builder requestParameters(Map<String, String> requestParameters) {
            this.requestParameters = MapOfStringToStringCopier.copy(requestParameters);
            return this;
        }

        public final Map<String, String> getRequestTemplates() {
            if (requestTemplates instanceof SdkAutoConstructMap) {
                return null;
            }
            return requestTemplates;
        }

        public final void setRequestTemplates(Map<String, String> requestTemplates) {
            this.requestTemplates = MapOfStringToStringCopier.copy(requestTemplates);
        }

        @Override
        public final Builder requestTemplates(Map<String, String> requestTemplates) {
            this.requestTemplates = MapOfStringToStringCopier.copy(requestTemplates);
            return this;
        }

        public final String getPassthroughBehavior() {
            return passthroughBehavior;
        }

        public final void setPassthroughBehavior(String passthroughBehavior) {
            this.passthroughBehavior = passthroughBehavior;
        }

        @Override
        public final Builder passthroughBehavior(String passthroughBehavior) {
            this.passthroughBehavior = passthroughBehavior;
            return this;
        }

        public final String getCacheNamespace() {
            return cacheNamespace;
        }

        public final void setCacheNamespace(String cacheNamespace) {
            this.cacheNamespace = cacheNamespace;
        }

        @Override
        public final Builder cacheNamespace(String cacheNamespace) {
            this.cacheNamespace = cacheNamespace;
            return this;
        }

        public final Collection<String> getCacheKeyParameters() {
            if (cacheKeyParameters instanceof SdkAutoConstructList) {
                return null;
            }
            return cacheKeyParameters;
        }

        public final void setCacheKeyParameters(Collection<String> cacheKeyParameters) {
            this.cacheKeyParameters = ListOfStringCopier.copy(cacheKeyParameters);
        }

        @Override
        public final Builder cacheKeyParameters(Collection<String> cacheKeyParameters) {
            this.cacheKeyParameters = ListOfStringCopier.copy(cacheKeyParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheKeyParameters(String... cacheKeyParameters) {
            cacheKeyParameters(Arrays.asList(cacheKeyParameters));
            return this;
        }

        public final String getContentHandling() {
            return contentHandling;
        }

        public final void setContentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
        }

        @Override
        public final Builder contentHandling(String contentHandling) {
            this.contentHandling = contentHandling;
            return this;
        }

        @Override
        public final Builder contentHandling(ContentHandlingStrategy contentHandling) {
            this.contentHandling(contentHandling == null ? null : contentHandling.toString());
            return this;
        }

        public final Integer getTimeoutInMillis() {
            return timeoutInMillis;
        }

        public final void setTimeoutInMillis(Integer timeoutInMillis) {
            this.timeoutInMillis = timeoutInMillis;
        }

        @Override
        public final Builder timeoutInMillis(Integer timeoutInMillis) {
            this.timeoutInMillis = timeoutInMillis;
            return this;
        }

        public final TlsConfig.Builder getTlsConfig() {
            return tlsConfig != null ? tlsConfig.toBuilder() : null;
        }

        public final void setTlsConfig(TlsConfig.BuilderImpl tlsConfig) {
            this.tlsConfig = tlsConfig != null ? tlsConfig.build() : null;
        }

        @Override
        public final Builder tlsConfig(TlsConfig tlsConfig) {
            this.tlsConfig = tlsConfig;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutIntegrationRequest build() {
            return new PutIntegrationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
