/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration settings of a canary deployment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CanarySettings implements SdkPojo, Serializable, ToCopyableBuilder<CanarySettings.Builder, CanarySettings> {
    private static final SdkField<Double> PERCENT_TRAFFIC_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("percentTraffic").getter(getter(CanarySettings::percentTraffic)).setter(setter(Builder::percentTraffic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("percentTraffic").build()).build();

    private static final SdkField<String> DEPLOYMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deploymentId").getter(getter(CanarySettings::deploymentId)).setter(setter(Builder::deploymentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentId").build()).build();

    private static final SdkField<Map<String, String>> STAGE_VARIABLE_OVERRIDES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("stageVariableOverrides")
            .getter(getter(CanarySettings::stageVariableOverrides))
            .setter(setter(Builder::stageVariableOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stageVariableOverrides").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> USE_STAGE_CACHE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("useStageCache").getter(getter(CanarySettings::useStageCache)).setter(setter(Builder::useStageCache))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("useStageCache").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PERCENT_TRAFFIC_FIELD,
            DEPLOYMENT_ID_FIELD, STAGE_VARIABLE_OVERRIDES_FIELD, USE_STAGE_CACHE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Double percentTraffic;

    private final String deploymentId;

    private final Map<String, String> stageVariableOverrides;

    private final Boolean useStageCache;

    private CanarySettings(BuilderImpl builder) {
        this.percentTraffic = builder.percentTraffic;
        this.deploymentId = builder.deploymentId;
        this.stageVariableOverrides = builder.stageVariableOverrides;
        this.useStageCache = builder.useStageCache;
    }

    /**
     * <p>
     * The percent (0-100) of traffic diverted to a canary deployment.
     * </p>
     * 
     * @return The percent (0-100) of traffic diverted to a canary deployment.
     */
    public final Double percentTraffic() {
        return percentTraffic;
    }

    /**
     * <p>
     * The ID of the canary deployment.
     * </p>
     * 
     * @return The ID of the canary deployment.
     */
    public final String deploymentId() {
        return deploymentId;
    }

    /**
     * For responses, this returns true if the service returned a value for the StageVariableOverrides property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasStageVariableOverrides() {
        return stageVariableOverrides != null && !(stageVariableOverrides instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Stage variables overridden for a canary release deployment, including new stage variables introduced in the
     * canary. These stage variables are represented as a string-to-string map between stage variable names and their
     * values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStageVariableOverrides} method.
     * </p>
     * 
     * @return Stage variables overridden for a canary release deployment, including new stage variables introduced in
     *         the canary. These stage variables are represented as a string-to-string map between stage variable names
     *         and their values.
     */
    public final Map<String, String> stageVariableOverrides() {
        return stageVariableOverrides;
    }

    /**
     * <p>
     * A Boolean flag to indicate whether the canary deployment uses the stage cache or not.
     * </p>
     * 
     * @return A Boolean flag to indicate whether the canary deployment uses the stage cache or not.
     */
    public final Boolean useStageCache() {
        return useStageCache;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(percentTraffic());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentId());
        hashCode = 31 * hashCode + Objects.hashCode(hasStageVariableOverrides() ? stageVariableOverrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(useStageCache());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CanarySettings)) {
            return false;
        }
        CanarySettings other = (CanarySettings) obj;
        return Objects.equals(percentTraffic(), other.percentTraffic()) && Objects.equals(deploymentId(), other.deploymentId())
                && hasStageVariableOverrides() == other.hasStageVariableOverrides()
                && Objects.equals(stageVariableOverrides(), other.stageVariableOverrides())
                && Objects.equals(useStageCache(), other.useStageCache());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CanarySettings").add("PercentTraffic", percentTraffic()).add("DeploymentId", deploymentId())
                .add("StageVariableOverrides", hasStageVariableOverrides() ? stageVariableOverrides() : null)
                .add("UseStageCache", useStageCache()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "percentTraffic":
            return Optional.ofNullable(clazz.cast(percentTraffic()));
        case "deploymentId":
            return Optional.ofNullable(clazz.cast(deploymentId()));
        case "stageVariableOverrides":
            return Optional.ofNullable(clazz.cast(stageVariableOverrides()));
        case "useStageCache":
            return Optional.ofNullable(clazz.cast(useStageCache()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("percentTraffic", PERCENT_TRAFFIC_FIELD);
        map.put("deploymentId", DEPLOYMENT_ID_FIELD);
        map.put("stageVariableOverrides", STAGE_VARIABLE_OVERRIDES_FIELD);
        map.put("useStageCache", USE_STAGE_CACHE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CanarySettings, T> g) {
        return obj -> g.apply((CanarySettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CanarySettings> {
        /**
         * <p>
         * The percent (0-100) of traffic diverted to a canary deployment.
         * </p>
         * 
         * @param percentTraffic
         *        The percent (0-100) of traffic diverted to a canary deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentTraffic(Double percentTraffic);

        /**
         * <p>
         * The ID of the canary deployment.
         * </p>
         * 
         * @param deploymentId
         *        The ID of the canary deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentId(String deploymentId);

        /**
         * <p>
         * Stage variables overridden for a canary release deployment, including new stage variables introduced in the
         * canary. These stage variables are represented as a string-to-string map between stage variable names and
         * their values.
         * </p>
         * 
         * @param stageVariableOverrides
         *        Stage variables overridden for a canary release deployment, including new stage variables introduced
         *        in the canary. These stage variables are represented as a string-to-string map between stage variable
         *        names and their values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stageVariableOverrides(Map<String, String> stageVariableOverrides);

        /**
         * <p>
         * A Boolean flag to indicate whether the canary deployment uses the stage cache or not.
         * </p>
         * 
         * @param useStageCache
         *        A Boolean flag to indicate whether the canary deployment uses the stage cache or not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useStageCache(Boolean useStageCache);
    }

    static final class BuilderImpl implements Builder {
        private Double percentTraffic;

        private String deploymentId;

        private Map<String, String> stageVariableOverrides = DefaultSdkAutoConstructMap.getInstance();

        private Boolean useStageCache;

        private BuilderImpl() {
        }

        private BuilderImpl(CanarySettings model) {
            percentTraffic(model.percentTraffic);
            deploymentId(model.deploymentId);
            stageVariableOverrides(model.stageVariableOverrides);
            useStageCache(model.useStageCache);
        }

        public final Double getPercentTraffic() {
            return percentTraffic;
        }

        public final void setPercentTraffic(Double percentTraffic) {
            this.percentTraffic = percentTraffic;
        }

        @Override
        public final Builder percentTraffic(Double percentTraffic) {
            this.percentTraffic = percentTraffic;
            return this;
        }

        public final String getDeploymentId() {
            return deploymentId;
        }

        public final void setDeploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
        }

        @Override
        public final Builder deploymentId(String deploymentId) {
            this.deploymentId = deploymentId;
            return this;
        }

        public final Map<String, String> getStageVariableOverrides() {
            if (stageVariableOverrides instanceof SdkAutoConstructMap) {
                return null;
            }
            return stageVariableOverrides;
        }

        public final void setStageVariableOverrides(Map<String, String> stageVariableOverrides) {
            this.stageVariableOverrides = MapOfStringToStringCopier.copy(stageVariableOverrides);
        }

        @Override
        public final Builder stageVariableOverrides(Map<String, String> stageVariableOverrides) {
            this.stageVariableOverrides = MapOfStringToStringCopier.copy(stageVariableOverrides);
            return this;
        }

        public final Boolean getUseStageCache() {
            return useStageCache;
        }

        public final void setUseStageCache(Boolean useStageCache) {
            this.useStageCache = useStageCache;
        }

        @Override
        public final Builder useStageCache(Boolean useStageCache) {
            this.useStageCache = useStageCache;
            return this;
        }

        @Override
        public CanarySettings build() {
            return new CanarySettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
