/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.apigateway.ApiGatewayClient;
import software.amazon.awssdk.services.apigateway.model.ClientCertificate;
import software.amazon.awssdk.services.apigateway.model.GetClientCertificatesRequest;
import software.amazon.awssdk.services.apigateway.model.GetClientCertificatesResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.apigateway.ApiGatewayClient#getClientCertificatesPaginator(software.amazon.awssdk.services.apigateway.model.GetClientCertificatesRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.apigateway.model.GetClientCertificatesResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.apigateway.paginators.GetClientCertificatesIterable responses = client.getClientCertificatesPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.apigateway.paginators.GetClientCertificatesIterable responses = client
 *             .getClientCertificatesPaginator(request);
 *     for (software.amazon.awssdk.services.apigateway.model.GetClientCertificatesResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.apigateway.paginators.GetClientCertificatesIterable responses = client.getClientCertificatesPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #getClientCertificates(software.amazon.awssdk.services.apigateway.model.GetClientCertificatesRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class GetClientCertificatesIterable implements SdkIterable<GetClientCertificatesResponse> {
    private final ApiGatewayClient client;

    private final GetClientCertificatesRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public GetClientCertificatesIterable(ApiGatewayClient client, GetClientCertificatesRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new GetClientCertificatesResponseFetcher();
    }

    @Override
    public Iterator<GetClientCertificatesResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated {@link GetClientCertificatesResponse#items()} member. The
     * returned iterable is used to iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<ClientCertificate> items() {
        Function<GetClientCertificatesResponse, Iterator<ClientCertificate>> getIterator = response -> {
            if (response != null && response.items() != null) {
                return response.items().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.builder().pagesIterable(this).itemIteratorFunction(getIterator).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link GetClientCertificatesIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final GetClientCertificatesIterable resume(GetClientCertificatesResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new GetClientCertificatesIterable(client, firstRequest.toBuilder().position(lastSuccessfulPage.position())
                    .build());
        }
        return new GetClientCertificatesIterable(client, firstRequest) {
            @Override
            public Iterator<GetClientCertificatesResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class GetClientCertificatesResponseFetcher implements SyncPageFetcher<GetClientCertificatesResponse> {
        @Override
        public boolean hasNextPage(GetClientCertificatesResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.position());
        }

        @Override
        public GetClientCertificatesResponse nextPage(GetClientCertificatesResponse previousPage) {
            if (previousPage == null) {
                return client.getClientCertificates(firstRequest);
            }
            return client.getClientCertificates(firstRequest.toBuilder().position(previousPage.position()).build());
        }
    }
}
