/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a usage plan than can specify who can assess associated API stages with specified request limits and
 * quotas.
 * </p>
 * <div class="remarks">
 * <p>
 * In a usage plan, you associate an API by specifying the API's Id and a stage name of the specified API. You add plan
 * customers by adding API keys to the plan.
 * </p>
 * </div> <div class="seeAlso"> <a
 * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html">Create and Use
 * Usage Plans</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UsagePlan implements SdkPojo, Serializable, ToCopyableBuilder<UsagePlan.Builder, UsagePlan> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UsagePlan::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UsagePlan::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UsagePlan::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<ApiStage>> API_STAGES_FIELD = SdkField
            .<List<ApiStage>> builder(MarshallingType.LIST)
            .getter(getter(UsagePlan::apiStages))
            .setter(setter(Builder::apiStages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("apiStages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ApiStage> builder(MarshallingType.SDK_POJO)
                                            .constructor(ApiStage::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ThrottleSettings> THROTTLE_FIELD = SdkField
            .<ThrottleSettings> builder(MarshallingType.SDK_POJO).getter(getter(UsagePlan::throttle))
            .setter(setter(Builder::throttle)).constructor(ThrottleSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("throttle").build()).build();

    private static final SdkField<QuotaSettings> QUOTA_FIELD = SdkField.<QuotaSettings> builder(MarshallingType.SDK_POJO)
            .getter(getter(UsagePlan::quota)).setter(setter(Builder::quota)).constructor(QuotaSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("quota").build()).build();

    private static final SdkField<String> PRODUCT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(UsagePlan::productCode)).setter(setter(Builder::productCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("productCode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, NAME_FIELD,
            DESCRIPTION_FIELD, API_STAGES_FIELD, THROTTLE_FIELD, QUOTA_FIELD, PRODUCT_CODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String name;

    private final String description;

    private final List<ApiStage> apiStages;

    private final ThrottleSettings throttle;

    private final QuotaSettings quota;

    private final String productCode;

    private UsagePlan(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.apiStages = builder.apiStages;
        this.throttle = builder.throttle;
        this.quota = builder.quota;
        this.productCode = builder.productCode;
    }

    /**
     * <p>
     * The identifier of a <a>UsagePlan</a> resource.
     * </p>
     * 
     * @return The identifier of a <a>UsagePlan</a> resource.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of a usage plan.
     * </p>
     * 
     * @return The name of a usage plan.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The description of a usage plan.
     * </p>
     * 
     * @return The description of a usage plan.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The associated API stages of a usage plan.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The associated API stages of a usage plan.
     */
    public List<ApiStage> apiStages() {
        return apiStages;
    }

    /**
     * <p>
     * The request throttle limits of a usage plan.
     * </p>
     * 
     * @return The request throttle limits of a usage plan.
     */
    public ThrottleSettings throttle() {
        return throttle;
    }

    /**
     * <p>
     * The maximum number of permitted requests per a given unit time interval.
     * </p>
     * 
     * @return The maximum number of permitted requests per a given unit time interval.
     */
    public QuotaSettings quota() {
        return quota;
    }

    /**
     * <p>
     * The AWS Markeplace product identifier to associate with the usage plan as a SaaS product on AWS Marketplace.
     * </p>
     * 
     * @return The AWS Markeplace product identifier to associate with the usage plan as a SaaS product on AWS
     *         Marketplace.
     */
    public String productCode() {
        return productCode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(apiStages());
        hashCode = 31 * hashCode + Objects.hashCode(throttle());
        hashCode = 31 * hashCode + Objects.hashCode(quota());
        hashCode = 31 * hashCode + Objects.hashCode(productCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UsagePlan)) {
            return false;
        }
        UsagePlan other = (UsagePlan) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && Objects.equals(apiStages(), other.apiStages())
                && Objects.equals(throttle(), other.throttle()) && Objects.equals(quota(), other.quota())
                && Objects.equals(productCode(), other.productCode());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("UsagePlan").add("Id", id()).add("Name", name()).add("Description", description())
                .add("ApiStages", apiStages()).add("Throttle", throttle()).add("Quota", quota())
                .add("ProductCode", productCode()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "apiStages":
            return Optional.ofNullable(clazz.cast(apiStages()));
        case "throttle":
            return Optional.ofNullable(clazz.cast(throttle()));
        case "quota":
            return Optional.ofNullable(clazz.cast(quota()));
        case "productCode":
            return Optional.ofNullable(clazz.cast(productCode()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UsagePlan, T> g) {
        return obj -> g.apply((UsagePlan) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UsagePlan> {
        /**
         * <p>
         * The identifier of a <a>UsagePlan</a> resource.
         * </p>
         * 
         * @param id
         *        The identifier of a <a>UsagePlan</a> resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of a usage plan.
         * </p>
         * 
         * @param name
         *        The name of a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of a usage plan.
         * </p>
         * 
         * @param description
         *        The description of a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The associated API stages of a usage plan.
         * </p>
         * 
         * @param apiStages
         *        The associated API stages of a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiStages(Collection<ApiStage> apiStages);

        /**
         * <p>
         * The associated API stages of a usage plan.
         * </p>
         * 
         * @param apiStages
         *        The associated API stages of a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiStages(ApiStage... apiStages);

        /**
         * <p>
         * The associated API stages of a usage plan.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ApiStage>.Builder} avoiding the need to
         * create one manually via {@link List<ApiStage>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ApiStage>.Builder#build()} is called immediately and its
         * result is passed to {@link #apiStages(List<ApiStage>)}.
         * 
         * @param apiStages
         *        a consumer that will call methods on {@link List<ApiStage>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #apiStages(List<ApiStage>)
         */
        Builder apiStages(Consumer<ApiStage.Builder>... apiStages);

        /**
         * <p>
         * The request throttle limits of a usage plan.
         * </p>
         * 
         * @param throttle
         *        The request throttle limits of a usage plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throttle(ThrottleSettings throttle);

        /**
         * <p>
         * The request throttle limits of a usage plan.
         * </p>
         * This is a convenience that creates an instance of the {@link ThrottleSettings.Builder} avoiding the need to
         * create one manually via {@link ThrottleSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link ThrottleSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #throttle(ThrottleSettings)}.
         * 
         * @param throttle
         *        a consumer that will call methods on {@link ThrottleSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #throttle(ThrottleSettings)
         */
        default Builder throttle(Consumer<ThrottleSettings.Builder> throttle) {
            return throttle(ThrottleSettings.builder().applyMutation(throttle).build());
        }

        /**
         * <p>
         * The maximum number of permitted requests per a given unit time interval.
         * </p>
         * 
         * @param quota
         *        The maximum number of permitted requests per a given unit time interval.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quota(QuotaSettings quota);

        /**
         * <p>
         * The maximum number of permitted requests per a given unit time interval.
         * </p>
         * This is a convenience that creates an instance of the {@link QuotaSettings.Builder} avoiding the need to
         * create one manually via {@link QuotaSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link QuotaSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #quota(QuotaSettings)}.
         * 
         * @param quota
         *        a consumer that will call methods on {@link QuotaSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #quota(QuotaSettings)
         */
        default Builder quota(Consumer<QuotaSettings.Builder> quota) {
            return quota(QuotaSettings.builder().applyMutation(quota).build());
        }

        /**
         * <p>
         * The AWS Markeplace product identifier to associate with the usage plan as a SaaS product on AWS Marketplace.
         * </p>
         * 
         * @param productCode
         *        The AWS Markeplace product identifier to associate with the usage plan as a SaaS product on AWS
         *        Marketplace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productCode(String productCode);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private String description;

        private List<ApiStage> apiStages = DefaultSdkAutoConstructList.getInstance();

        private ThrottleSettings throttle;

        private QuotaSettings quota;

        private String productCode;

        private BuilderImpl() {
        }

        private BuilderImpl(UsagePlan model) {
            id(model.id);
            name(model.name);
            description(model.description);
            apiStages(model.apiStages);
            throttle(model.throttle);
            quota(model.quota);
            productCode(model.productCode);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<ApiStage.Builder> getApiStages() {
            return apiStages != null ? apiStages.stream().map(ApiStage::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder apiStages(Collection<ApiStage> apiStages) {
            this.apiStages = ListOfApiStageCopier.copy(apiStages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder apiStages(ApiStage... apiStages) {
            apiStages(Arrays.asList(apiStages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder apiStages(Consumer<ApiStage.Builder>... apiStages) {
            apiStages(Stream.of(apiStages).map(c -> ApiStage.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setApiStages(Collection<ApiStage.BuilderImpl> apiStages) {
            this.apiStages = ListOfApiStageCopier.copyFromBuilder(apiStages);
        }

        public final ThrottleSettings.Builder getThrottle() {
            return throttle != null ? throttle.toBuilder() : null;
        }

        @Override
        public final Builder throttle(ThrottleSettings throttle) {
            this.throttle = throttle;
            return this;
        }

        public final void setThrottle(ThrottleSettings.BuilderImpl throttle) {
            this.throttle = throttle != null ? throttle.build() : null;
        }

        public final QuotaSettings.Builder getQuota() {
            return quota != null ? quota.toBuilder() : null;
        }

        @Override
        public final Builder quota(QuotaSettings quota) {
            this.quota = quota;
            return this;
        }

        public final void setQuota(QuotaSettings.BuilderImpl quota) {
            this.quota = quota != null ? quota.build() : null;
        }

        public final String getProductCode() {
            return productCode;
        }

        @Override
        public final Builder productCode(String productCode) {
            this.productCode = productCode;
            return this;
        }

        public final void setProductCode(String productCode) {
            this.productCode = productCode;
        }

        @Override
        public UsagePlan build() {
            return new UsagePlan(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
