/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a client-facing interface by which the client calls the API to access back-end resources. A <b>Method</b>
 * resource is integrated with an <a>Integration</a> resource. Both consist of a request and one or more responses. The
 * method request takes the client input that is passed to the back end through the integration request. A method
 * response returns the output from the back end to the client through an integration response. A method request is
 * embodied in a <b>Method</b> resource, whereas an integration request is embodied in an <a>Integration</a> resource.
 * On the other hand, a method response is represented by a <a>MethodResponse</a> resource, whereas an integration
 * response is represented by an <a>IntegrationResponse</a> resource.
 * </p>
 * <div class="remarks">
 * <p/>
 * <h4>Example: Retrive the GET method on a specified resource</h4>
 * <h5>Request</h5>
 * <p>
 * The following example request retrieves the information about the GET method on an API resource (
 * <code>3kzxbg5sa2</code>) of an API (<code>fugvjdxtri</code>).
 * </p>
 * 
 * <pre>
 * <code>GET /restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com X-Amz-Date: 20160603T210259Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160603/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
 * </pre>
 * 
 * <h5>Response</h5>
 * <p>
 * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
 * </p>
 * 
 * <pre>
 * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-{rel}.html", "name": "method", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true } ], "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET", "name": "GET", "title": "GET" }, "integration:put": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration" }, "method:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET" }, "method:integration": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration" }, "method:responses": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200", "name": "200", "title": "200" }, "method:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET" }, "methodresponse:put": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/{status_code}", "templated": true } }, "apiKeyRequired": true, "authorizationType": "NONE", "httpMethod": "GET", "_embedded": { "method:integration": { "_links": { "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration" }, "integration:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration" }, "integration:responses": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "3kzxbg5sa2", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestParameters": { "integration.request.header.Content-Type": "'application/x-amz-json-1.1'" }, "requestTemplates": { "application/json": "{\n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-east-1:kinesis:action/ListStreams", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.Content-Type": "'application/xml'" }, "responseTemplates": { "application/json": "$util.urlDecode(\"%3CkinesisStreams%3E%23foreach(%24stream%20in%20%24input.path(%27%24.StreamNames%27))%3Cstream%3E%3Cname%3E%24stream%3C%2Fname%3E%3C%2Fstream%3E%23end%3C%2FkinesisStreams%3E\")" }, "statusCode": "200" } } }, "method:responses": { "_links": { "self": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200", "name": "200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/fugvjdxtri/resources/3kzxbg5sa2/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.Content-Type": false }, "statusCode": "200" } } }</code>
 * </pre>
 * <p>
 * In the example above, the response template for the <code>200 OK</code> response maps the JSON output from the
 * <code>ListStreams</code> action in the back end to an XML output. The mapping template is URL-encoded as
 * <code>%3CkinesisStreams%3E%23foreach(%24stream%20in%20%24input.path(%27%24.StreamNames%27))%3Cstream%3E%3Cname%3E%24stream%3C%2Fname%3E%3C%2Fstream%3E%23end%3C%2FkinesisStreams%3E</code>
 * and the output is decoded using the <a href=
 * "https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-mapping-template-reference.html#util-templat-reference"
 * >$util.urlDecode()</a> helper function.
 * </p>
 * </div> <div class="seeAlso"> <a>MethodResponse</a>, <a>Integration</a>, <a>IntegrationResponse</a>, <a>Resource</a>,
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/how-to-method-settings.html">Set up an API's
 * method</a> </div>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetMethodResponse extends ApiGatewayResponse implements
        ToCopyableBuilder<GetMethodResponse.Builder, GetMethodResponse> {
    private static final SdkField<String> HTTP_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetMethodResponse::httpMethod)).setter(setter(Builder::httpMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("httpMethod").build()).build();

    private static final SdkField<String> AUTHORIZATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetMethodResponse::authorizationType)).setter(setter(Builder::authorizationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizationType").build()).build();

    private static final SdkField<String> AUTHORIZER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetMethodResponse::authorizerId)).setter(setter(Builder::authorizerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerId").build()).build();

    private static final SdkField<Boolean> API_KEY_REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(GetMethodResponse::apiKeyRequired)).setter(setter(Builder::apiKeyRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("apiKeyRequired").build()).build();

    private static final SdkField<String> REQUEST_VALIDATOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetMethodResponse::requestValidatorId)).setter(setter(Builder::requestValidatorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestValidatorId").build())
            .build();

    private static final SdkField<String> OPERATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GetMethodResponse::operationName)).setter(setter(Builder::operationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("operationName").build()).build();

    private static final SdkField<Map<String, Boolean>> REQUEST_PARAMETERS_FIELD = SdkField
            .<Map<String, Boolean>> builder(MarshallingType.MAP)
            .getter(getter(GetMethodResponse::requestParameters))
            .setter(setter(Builder::requestParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> REQUEST_MODELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(GetMethodResponse::requestModels))
            .setter(setter(Builder::requestModels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requestModels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, MethodResponse>> METHOD_RESPONSES_FIELD = SdkField
            .<Map<String, MethodResponse>> builder(MarshallingType.MAP)
            .getter(getter(GetMethodResponse::methodResponses))
            .setter(setter(Builder::methodResponses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("methodResponses").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MethodResponse> builder(MarshallingType.SDK_POJO)
                                            .constructor(MethodResponse::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integration> METHOD_INTEGRATION_FIELD = SdkField
            .<Integration> builder(MarshallingType.SDK_POJO).getter(getter(GetMethodResponse::methodIntegration))
            .setter(setter(Builder::methodIntegration)).constructor(Integration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("methodIntegration").build()).build();

    private static final SdkField<List<String>> AUTHORIZATION_SCOPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GetMethodResponse::authorizationScopes))
            .setter(setter(Builder::authorizationScopes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizationScopes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HTTP_METHOD_FIELD,
            AUTHORIZATION_TYPE_FIELD, AUTHORIZER_ID_FIELD, API_KEY_REQUIRED_FIELD, REQUEST_VALIDATOR_ID_FIELD,
            OPERATION_NAME_FIELD, REQUEST_PARAMETERS_FIELD, REQUEST_MODELS_FIELD, METHOD_RESPONSES_FIELD,
            METHOD_INTEGRATION_FIELD, AUTHORIZATION_SCOPES_FIELD));

    private final String httpMethod;

    private final String authorizationType;

    private final String authorizerId;

    private final Boolean apiKeyRequired;

    private final String requestValidatorId;

    private final String operationName;

    private final Map<String, Boolean> requestParameters;

    private final Map<String, String> requestModels;

    private final Map<String, MethodResponse> methodResponses;

    private final Integration methodIntegration;

    private final List<String> authorizationScopes;

    private GetMethodResponse(BuilderImpl builder) {
        super(builder);
        this.httpMethod = builder.httpMethod;
        this.authorizationType = builder.authorizationType;
        this.authorizerId = builder.authorizerId;
        this.apiKeyRequired = builder.apiKeyRequired;
        this.requestValidatorId = builder.requestValidatorId;
        this.operationName = builder.operationName;
        this.requestParameters = builder.requestParameters;
        this.requestModels = builder.requestModels;
        this.methodResponses = builder.methodResponses;
        this.methodIntegration = builder.methodIntegration;
        this.authorizationScopes = builder.authorizationScopes;
    }

    /**
     * <p>
     * The method's HTTP verb.
     * </p>
     * 
     * @return The method's HTTP verb.
     */
    public String httpMethod() {
        return httpMethod;
    }

    /**
     * <p>
     * The method's authorization type. Valid values are <code>NONE</code> for open access, <code>AWS_IAM</code> for
     * using AWS IAM permissions, <code>CUSTOM</code> for using a custom authorizer, or <code>COGNITO_USER_POOLS</code>
     * for using a Cognito user pool.
     * </p>
     * 
     * @return The method's authorization type. Valid values are <code>NONE</code> for open access, <code>AWS_IAM</code>
     *         for using AWS IAM permissions, <code>CUSTOM</code> for using a custom authorizer, or
     *         <code>COGNITO_USER_POOLS</code> for using a Cognito user pool.
     */
    public String authorizationType() {
        return authorizationType;
    }

    /**
     * <p>
     * The identifier of an <a>Authorizer</a> to use on this method. The <code>authorizationType</code> must be
     * <code>CUSTOM</code>.
     * </p>
     * 
     * @return The identifier of an <a>Authorizer</a> to use on this method. The <code>authorizationType</code> must be
     *         <code>CUSTOM</code>.
     */
    public String authorizerId() {
        return authorizerId;
    }

    /**
     * <p>
     * A boolean flag specifying whether a valid <a>ApiKey</a> is required to invoke this method.
     * </p>
     * 
     * @return A boolean flag specifying whether a valid <a>ApiKey</a> is required to invoke this method.
     */
    public Boolean apiKeyRequired() {
        return apiKeyRequired;
    }

    /**
     * <p>
     * The identifier of a <a>RequestValidator</a> for request validation.
     * </p>
     * 
     * @return The identifier of a <a>RequestValidator</a> for request validation.
     */
    public String requestValidatorId() {
        return requestValidatorId;
    }

    /**
     * <p>
     * A human-friendly operation identifier for the method. For example, you can assign the <code>operationName</code>
     * of <code>ListPets</code> for the <code>GET /pets</code> method in the <code>PetStore</code> example.
     * </p>
     * 
     * @return A human-friendly operation identifier for the method. For example, you can assign the
     *         <code>operationName</code> of <code>ListPets</code> for the <code>GET /pets</code> method in the
     *         <code>PetStore</code> example.
     */
    public String operationName() {
        return operationName;
    }

    /**
     * <p>
     * A key-value map defining required or optional method request parameters that can be accepted by API Gateway. A
     * key is a method request parameter name matching the pattern of <code>method.request.{location}.{name}</code>,
     * where <code>location</code> is <code>querystring</code>, <code>path</code>, or <code>header</code> and
     * <code>name</code> is a valid and unique parameter name. The value associated with the key is a Boolean flag
     * indicating whether the parameter is required (<code>true</code>) or optional (<code>false</code>). The method
     * request parameter names defined here are available in <a>Integration</a> to be mapped to integration request
     * parameters or templates.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A key-value map defining required or optional method request parameters that can be accepted by API
     *         Gateway. A key is a method request parameter name matching the pattern of
     *         <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>,
     *         <code>path</code>, or <code>header</code> and <code>name</code> is a valid and unique parameter name. The
     *         value associated with the key is a Boolean flag indicating whether the parameter is required (
     *         <code>true</code>) or optional (<code>false</code>). The method request parameter names defined here are
     *         available in <a>Integration</a> to be mapped to integration request parameters or templates.
     */
    public Map<String, Boolean> requestParameters() {
        return requestParameters;
    }

    /**
     * <p>
     * A key-value map specifying data schemas, represented by <a>Model</a> resources, (as the mapped value) of the
     * request payloads of given content types (as the mapping key).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A key-value map specifying data schemas, represented by <a>Model</a> resources, (as the mapped value) of
     *         the request payloads of given content types (as the mapping key).
     */
    public Map<String, String> requestModels() {
        return requestModels;
    }

    /**
     * <p>
     * Gets a method response associated with a given HTTP status code.
     * </p>
     * <div class="remarks">
     * <p>
     * The collection of method responses are encapsulated in a key-value map, where the key is a response's HTTP status
     * code and the value is a <a>MethodResponse</a> resource that specifies the response returned to the caller from
     * the back end through the integration response.
     * </p>
     * <h4>Example: Get a 200 OK response of a GET method</h4> <h5>Request</h5>
     * <p/>
     * 
     * <pre>
     * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T215008Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
     * </pre>
     * 
     * <h5>Response</h5>
     * <p>
     * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
     * </p>
     * 
     * <pre>
     * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true }, "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.operator": false, "method.response.header.operand_2": false, "method.response.header.operand_1": false }, "statusCode": "200" }</code>
     * </pre>
     * <p/>
     * </div> <div class="seeAlso"> <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-method-response.html">AWS CLI</a> </div>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Gets a method response associated with a given HTTP status code. </p> <div class="remarks">
     *         <p>
     *         The collection of method responses are encapsulated in a key-value map, where the key is a response's
     *         HTTP status code and the value is a <a>MethodResponse</a> resource that specifies the response returned
     *         to the caller from the back end through the integration response.
     *         </p>
     *         <h4>Example: Get a 200 OK response of a GET method</h4> <h5>Request</h5>
     *         <p/>
     * 
     *         <pre>
     * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T215008Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
     * </pre>
     * 
     *         <h5>Response</h5>
     *         <p>
     *         The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
     *         </p>
     * 
     *         <pre>
     * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true }, "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.operator": false, "method.response.header.operand_2": false, "method.response.header.operand_1": false }, "statusCode": "200" }</code>
     * </pre>
     *         <p/>
     *         </div> <div class="seeAlso"> <a
     *         href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-method-response.html">AWS CLI</a>
     */
    public Map<String, MethodResponse> methodResponses() {
        return methodResponses;
    }

    /**
     * <p>
     * Gets the method's integration responsible for passing the client-submitted request to the back end and performing
     * necessary transformations to make the request compliant with the back end.
     * </p>
     * <div class="remarks">
     * <p/>
     * <h4>Example:</h4>
     * <h5>Request</h5>
     * <p/>
     * 
     * <pre>
     * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T213210Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
     * </pre>
     * 
     * <h5>Response</h5>
     * <p>
     * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
     * </p>
     * 
     * <pre>
     * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true } ], "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:responses": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "0cjtch", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestTemplates": { "application/json": "{\n \"a\": \"$input.params('operand1')\",\n \"b\": \"$input.params('operand2')\", \n \"op\": \"$input.params('operator')\" \n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-west-2:lambda:path//2015-03-31/functions/arn:aws:lambda:us-west-2:123456789012:function:Calc/invocations", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.operator": "integration.response.body.op", "method.response.header.operand_2": "integration.response.body.b", "method.response.header.operand_1": "integration.response.body.a" }, "responseTemplates": { "application/json": "#set($res = $input.path('$'))\n{\n \"result\": \"$res.a, $res.b, $res.op => $res.c\",\n \"a\" : \"$res.a\",\n \"b\" : \"$res.b\",\n \"op\" : \"$res.op\",\n \"c\" : \"$res.c\"\n}" }, "selectionPattern": "", "statusCode": "200" } } }</code>
     * </pre>
     * <p/>
     * </div> <div class="seeAlso"> <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-integration.html">AWS CLI</a> </div>
     * 
     * @return Gets the method's integration responsible for passing the client-submitted request to the back end and
     *         performing necessary transformations to make the request compliant with the back end.
     *         </p>
     *         <div class="remarks">
     *         <p/>
     *         <h4>Example:</h4>
     *         <h5>Request</h5>
     *         <p/>
     * 
     *         <pre>
     * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T213210Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
     * </pre>
     * 
     *         <h5>Response</h5>
     *         <p>
     *         The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
     *         </p>
     * 
     *         <pre>
     * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true } ], "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:responses": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "0cjtch", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestTemplates": { "application/json": "{\n \"a\": \"$input.params('operand1')\",\n \"b\": \"$input.params('operand2')\", \n \"op\": \"$input.params('operator')\" \n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-west-2:lambda:path//2015-03-31/functions/arn:aws:lambda:us-west-2:123456789012:function:Calc/invocations", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.operator": "integration.response.body.op", "method.response.header.operand_2": "integration.response.body.b", "method.response.header.operand_1": "integration.response.body.a" }, "responseTemplates": { "application/json": "#set($res = $input.path('$'))\n{\n \"result\": \"$res.a, $res.b, $res.op => $res.c\",\n \"a\" : \"$res.a\",\n \"b\" : \"$res.b\",\n \"op\" : \"$res.op\",\n \"c\" : \"$res.c\"\n}" }, "selectionPattern": "", "statusCode": "200" } } }</code>
     * </pre>
     *         <p/>
     *         </div> <div class="seeAlso"> <a
     *         href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-integration.html">AWS CLI</a>
     */
    public Integration methodIntegration() {
        return methodIntegration;
    }

    /**
     * <p>
     * A list of authorization scopes configured on the method. The scopes are used with a
     * <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works by
     * matching the method scopes against the scopes parsed from the access token in the incoming request. The method
     * invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise, the
     * invocation is not authorized. When the method scope is configured, the client must provide an access token
     * instead of an identity token for authorization purposes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of authorization scopes configured on the method. The scopes are used with a
     *         <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works by
     *         matching the method scopes against the scopes parsed from the access token in the incoming request. The
     *         method invocation is authorized if any method scopes matches a claimed scope in the access token.
     *         Otherwise, the invocation is not authorized. When the method scope is configured, the client must provide
     *         an access token instead of an identity token for authorization purposes.
     */
    public List<String> authorizationScopes() {
        return authorizationScopes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(httpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(authorizationType());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerId());
        hashCode = 31 * hashCode + Objects.hashCode(apiKeyRequired());
        hashCode = 31 * hashCode + Objects.hashCode(requestValidatorId());
        hashCode = 31 * hashCode + Objects.hashCode(operationName());
        hashCode = 31 * hashCode + Objects.hashCode(requestParameters());
        hashCode = 31 * hashCode + Objects.hashCode(requestModels());
        hashCode = 31 * hashCode + Objects.hashCode(methodResponses());
        hashCode = 31 * hashCode + Objects.hashCode(methodIntegration());
        hashCode = 31 * hashCode + Objects.hashCode(authorizationScopes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetMethodResponse)) {
            return false;
        }
        GetMethodResponse other = (GetMethodResponse) obj;
        return Objects.equals(httpMethod(), other.httpMethod()) && Objects.equals(authorizationType(), other.authorizationType())
                && Objects.equals(authorizerId(), other.authorizerId())
                && Objects.equals(apiKeyRequired(), other.apiKeyRequired())
                && Objects.equals(requestValidatorId(), other.requestValidatorId())
                && Objects.equals(operationName(), other.operationName())
                && Objects.equals(requestParameters(), other.requestParameters())
                && Objects.equals(requestModels(), other.requestModels())
                && Objects.equals(methodResponses(), other.methodResponses())
                && Objects.equals(methodIntegration(), other.methodIntegration())
                && Objects.equals(authorizationScopes(), other.authorizationScopes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetMethodResponse").add("HttpMethod", httpMethod())
                .add("AuthorizationType", authorizationType()).add("AuthorizerId", authorizerId())
                .add("ApiKeyRequired", apiKeyRequired()).add("RequestValidatorId", requestValidatorId())
                .add("OperationName", operationName()).add("RequestParameters", requestParameters())
                .add("RequestModels", requestModels()).add("MethodResponses", methodResponses())
                .add("MethodIntegration", methodIntegration()).add("AuthorizationScopes", authorizationScopes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "httpMethod":
            return Optional.ofNullable(clazz.cast(httpMethod()));
        case "authorizationType":
            return Optional.ofNullable(clazz.cast(authorizationType()));
        case "authorizerId":
            return Optional.ofNullable(clazz.cast(authorizerId()));
        case "apiKeyRequired":
            return Optional.ofNullable(clazz.cast(apiKeyRequired()));
        case "requestValidatorId":
            return Optional.ofNullable(clazz.cast(requestValidatorId()));
        case "operationName":
            return Optional.ofNullable(clazz.cast(operationName()));
        case "requestParameters":
            return Optional.ofNullable(clazz.cast(requestParameters()));
        case "requestModels":
            return Optional.ofNullable(clazz.cast(requestModels()));
        case "methodResponses":
            return Optional.ofNullable(clazz.cast(methodResponses()));
        case "methodIntegration":
            return Optional.ofNullable(clazz.cast(methodIntegration()));
        case "authorizationScopes":
            return Optional.ofNullable(clazz.cast(authorizationScopes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetMethodResponse, T> g) {
        return obj -> g.apply((GetMethodResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetMethodResponse> {
        /**
         * <p>
         * The method's HTTP verb.
         * </p>
         * 
         * @param httpMethod
         *        The method's HTTP verb.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpMethod(String httpMethod);

        /**
         * <p>
         * The method's authorization type. Valid values are <code>NONE</code> for open access, <code>AWS_IAM</code> for
         * using AWS IAM permissions, <code>CUSTOM</code> for using a custom authorizer, or
         * <code>COGNITO_USER_POOLS</code> for using a Cognito user pool.
         * </p>
         * 
         * @param authorizationType
         *        The method's authorization type. Valid values are <code>NONE</code> for open access,
         *        <code>AWS_IAM</code> for using AWS IAM permissions, <code>CUSTOM</code> for using a custom authorizer,
         *        or <code>COGNITO_USER_POOLS</code> for using a Cognito user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizationType(String authorizationType);

        /**
         * <p>
         * The identifier of an <a>Authorizer</a> to use on this method. The <code>authorizationType</code> must be
         * <code>CUSTOM</code>.
         * </p>
         * 
         * @param authorizerId
         *        The identifier of an <a>Authorizer</a> to use on this method. The <code>authorizationType</code> must
         *        be <code>CUSTOM</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerId(String authorizerId);

        /**
         * <p>
         * A boolean flag specifying whether a valid <a>ApiKey</a> is required to invoke this method.
         * </p>
         * 
         * @param apiKeyRequired
         *        A boolean flag specifying whether a valid <a>ApiKey</a> is required to invoke this method.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiKeyRequired(Boolean apiKeyRequired);

        /**
         * <p>
         * The identifier of a <a>RequestValidator</a> for request validation.
         * </p>
         * 
         * @param requestValidatorId
         *        The identifier of a <a>RequestValidator</a> for request validation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestValidatorId(String requestValidatorId);

        /**
         * <p>
         * A human-friendly operation identifier for the method. For example, you can assign the
         * <code>operationName</code> of <code>ListPets</code> for the <code>GET /pets</code> method in the
         * <code>PetStore</code> example.
         * </p>
         * 
         * @param operationName
         *        A human-friendly operation identifier for the method. For example, you can assign the
         *        <code>operationName</code> of <code>ListPets</code> for the <code>GET /pets</code> method in the
         *        <code>PetStore</code> example.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationName(String operationName);

        /**
         * <p>
         * A key-value map defining required or optional method request parameters that can be accepted by API Gateway.
         * A key is a method request parameter name matching the pattern of
         * <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>,
         * <code>path</code>, or <code>header</code> and <code>name</code> is a valid and unique parameter name. The
         * value associated with the key is a Boolean flag indicating whether the parameter is required (
         * <code>true</code>) or optional (<code>false</code>). The method request parameter names defined here are
         * available in <a>Integration</a> to be mapped to integration request parameters or templates.
         * </p>
         * 
         * @param requestParameters
         *        A key-value map defining required or optional method request parameters that can be accepted by API
         *        Gateway. A key is a method request parameter name matching the pattern of
         *        <code>method.request.{location}.{name}</code>, where <code>location</code> is <code>querystring</code>
         *        , <code>path</code>, or <code>header</code> and <code>name</code> is a valid and unique parameter
         *        name. The value associated with the key is a Boolean flag indicating whether the parameter is required
         *        (<code>true</code>) or optional (<code>false</code>). The method request parameter names defined here
         *        are available in <a>Integration</a> to be mapped to integration request parameters or templates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestParameters(Map<String, Boolean> requestParameters);

        /**
         * <p>
         * A key-value map specifying data schemas, represented by <a>Model</a> resources, (as the mapped value) of the
         * request payloads of given content types (as the mapping key).
         * </p>
         * 
         * @param requestModels
         *        A key-value map specifying data schemas, represented by <a>Model</a> resources, (as the mapped value)
         *        of the request payloads of given content types (as the mapping key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestModels(Map<String, String> requestModels);

        /**
         * <p>
         * Gets a method response associated with a given HTTP status code.
         * </p>
         * <div class="remarks">
         * <p>
         * The collection of method responses are encapsulated in a key-value map, where the key is a response's HTTP
         * status code and the value is a <a>MethodResponse</a> resource that specifies the response returned to the
         * caller from the back end through the integration response.
         * </p>
         * <h4>Example: Get a 200 OK response of a GET method</h4> <h5>Request</h5>
         * <p/>
         * 
         * <pre>
         * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T215008Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
         * </pre>
         * 
         * <h5>Response</h5>
         * <p>
         * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
         * </p>
         * 
         * <pre>
         * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true }, "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.operator": false, "method.response.header.operand_2": false, "method.response.header.operand_1": false }, "statusCode": "200" }</code>
         * </pre>
         * <p/>
         * </div> <div class="seeAlso"> <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-method-response.html">AWS CLI</a>
         * </div>
         * 
         * @param methodResponses
         *        Gets a method response associated with a given HTTP status code.
         *        </p>
         *        <div class="remarks">
         *        <p>
         *        The collection of method responses are encapsulated in a key-value map, where the key is a response's
         *        HTTP status code and the value is a <a>MethodResponse</a> resource that specifies the response
         *        returned to the caller from the back end through the integration response.
         *        </p>
         *        <h4>Example: Get a 200 OK response of a GET method</h4> <h5>Request</h5>
         *        <p/>
         * 
         *        <pre>
         * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200 HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T215008Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
         * </pre>
         * 
         *        <h5>Response</h5>
         *        <p>
         *        The successful response returns a <code>200 OK</code> status code and a payload similar to the
         *        following:
         *        </p>
         * 
         *        <pre>
         * <code>{ "_links": { "curies": { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-method-response-{rel}.html", "name": "methodresponse", "templated": true }, "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200", "title": "200" }, "methodresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" }, "methodresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/responses/200" } }, "responseModels": { "application/json": "Empty" }, "responseParameters": { "method.response.header.operator": false, "method.response.header.operand_2": false, "method.response.header.operand_1": false }, "statusCode": "200" }</code>
         * </pre>
         *        <p/>
         *        </div> <div class="seeAlso"> <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-method-response.html">AWS
         *        CLI</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder methodResponses(Map<String, MethodResponse> methodResponses);

        /**
         * <p>
         * Gets the method's integration responsible for passing the client-submitted request to the back end and
         * performing necessary transformations to make the request compliant with the back end.
         * </p>
         * <div class="remarks">
         * <p/>
         * <h4>Example:</h4>
         * <h5>Request</h5>
         * <p/>
         * 
         * <pre>
         * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T213210Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
         * </pre>
         * 
         * <h5>Response</h5>
         * <p>
         * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
         * </p>
         * 
         * <pre>
         * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true } ], "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:responses": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "0cjtch", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestTemplates": { "application/json": "{\n \"a\": \"$input.params('operand1')\",\n \"b\": \"$input.params('operand2')\", \n \"op\": \"$input.params('operator')\" \n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-west-2:lambda:path//2015-03-31/functions/arn:aws:lambda:us-west-2:123456789012:function:Calc/invocations", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.operator": "integration.response.body.op", "method.response.header.operand_2": "integration.response.body.b", "method.response.header.operand_1": "integration.response.body.a" }, "responseTemplates": { "application/json": "#set($res = $input.path('$'))\n{\n \"result\": \"$res.a, $res.b, $res.op => $res.c\",\n \"a\" : \"$res.a\",\n \"b\" : \"$res.b\",\n \"op\" : \"$res.op\",\n \"c\" : \"$res.c\"\n}" }, "selectionPattern": "", "statusCode": "200" } } }</code>
         * </pre>
         * <p/>
         * </div> <div class="seeAlso"> <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-integration.html">AWS CLI</a> </div>
         * 
         * @param methodIntegration
         *        Gets the method's integration responsible for passing the client-submitted request to the back end and
         *        performing necessary transformations to make the request compliant with the back end.
         *        </p>
         *        <div class="remarks">
         *        <p/>
         *        <h4>Example:</h4>
         *        <h5>Request</h5>
         *        <p/>
         * 
         *        <pre>
         * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T213210Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
         * </pre>
         * 
         *        <h5>Response</h5>
         *        <p>
         *        The successful response returns a <code>200 OK</code> status code and a payload similar to the
         *        following:
         *        </p>
         * 
         *        <pre>
         * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true } ], "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:responses": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "0cjtch", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestTemplates": { "application/json": "{\n \"a\": \"$input.params('operand1')\",\n \"b\": \"$input.params('operand2')\", \n \"op\": \"$input.params('operator')\" \n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-west-2:lambda:path//2015-03-31/functions/arn:aws:lambda:us-west-2:123456789012:function:Calc/invocations", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.operator": "integration.response.body.op", "method.response.header.operand_2": "integration.response.body.b", "method.response.header.operand_1": "integration.response.body.a" }, "responseTemplates": { "application/json": "#set($res = $input.path('$'))\n{\n \"result\": \"$res.a, $res.b, $res.op => $res.c\",\n \"a\" : \"$res.a\",\n \"b\" : \"$res.b\",\n \"op\" : \"$res.op\",\n \"c\" : \"$res.c\"\n}" }, "selectionPattern": "", "statusCode": "200" } } }</code>
         * </pre>
         *        <p/>
         *        </div> <div class="seeAlso"> <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-integration.html">AWS CLI</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder methodIntegration(Integration methodIntegration);

        /**
         * <p>
         * Gets the method's integration responsible for passing the client-submitted request to the back end and
         * performing necessary transformations to make the request compliant with the back end.
         * </p>
         * <div class="remarks">
         * <p/>
         * <h4>Example:</h4>
         * <h5>Request</h5>
         * <p/>
         * 
         * <pre>
         * <code>GET /restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration HTTP/1.1 Content-Type: application/json Host: apigateway.us-east-1.amazonaws.com Content-Length: 117 X-Amz-Date: 20160613T213210Z Authorization: AWS4-HMAC-SHA256 Credential={access_key_ID}/20160613/us-east-1/apigateway/aws4_request, SignedHeaders=content-type;host;x-amz-date, Signature={sig4_hash}</code>
         * </pre>
         * 
         * <h5>Response</h5>
         * <p>
         * The successful response returns a <code>200 OK</code> status code and a payload similar to the following:
         * </p>
         * 
         * <pre>
         * <code>{ "_links": { "curies": [ { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-{rel}.html", "name": "integration", "templated": true }, { "href": "https://docs.aws.amazon.com/apigateway/latest/developerguide/restapi-integration-response-{rel}.html", "name": "integrationresponse", "templated": true } ], "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integration:responses": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integration:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration" }, "integrationresponse:put": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/{status_code}", "templated": true } }, "cacheKeyParameters": [], "cacheNamespace": "0cjtch", "credentials": "arn:aws:iam::123456789012:role/apigAwsProxyRole", "httpMethod": "POST", "passthroughBehavior": "WHEN_NO_MATCH", "requestTemplates": { "application/json": "{\n \"a\": \"$input.params('operand1')\",\n \"b\": \"$input.params('operand2')\", \n \"op\": \"$input.params('operator')\" \n}" }, "type": "AWS", "uri": "arn:aws:apigateway:us-west-2:lambda:path//2015-03-31/functions/arn:aws:lambda:us-west-2:123456789012:function:Calc/invocations", "_embedded": { "integration:responses": { "_links": { "self": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200", "name": "200", "title": "200" }, "integrationresponse:delete": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" }, "integrationresponse:update": { "href": "/restapis/uojnr9hd57/resources/0cjtch/methods/GET/integration/responses/200" } }, "responseParameters": { "method.response.header.operator": "integration.response.body.op", "method.response.header.operand_2": "integration.response.body.b", "method.response.header.operand_1": "integration.response.body.a" }, "responseTemplates": { "application/json": "#set($res = $input.path('$'))\n{\n \"result\": \"$res.a, $res.b, $res.op => $res.c\",\n \"a\" : \"$res.a\",\n \"b\" : \"$res.b\",\n \"op\" : \"$res.op\",\n \"c\" : \"$res.c\"\n}" }, "selectionPattern": "", "statusCode": "200" } } }</code>
         * </pre>
         * <p/>
         * </div> <div class="seeAlso"> <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/apigateway/get-integration.html">AWS CLI</a> </div>
         * This is a convenience that creates an instance of the {@link Integration.Builder} avoiding the need to create
         * one manually via {@link Integration#builder()}.
         *
         * When the {@link Consumer} completes, {@link Integration.Builder#build()} is called immediately and its result
         * is passed to {@link #methodIntegration(Integration)}.
         * 
         * @param methodIntegration
         *        a consumer that will call methods on {@link Integration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #methodIntegration(Integration)
         */
        default Builder methodIntegration(Consumer<Integration.Builder> methodIntegration) {
            return methodIntegration(Integration.builder().applyMutation(methodIntegration).build());
        }

        /**
         * <p>
         * A list of authorization scopes configured on the method. The scopes are used with a
         * <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works by
         * matching the method scopes against the scopes parsed from the access token in the incoming request. The
         * method invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise,
         * the invocation is not authorized. When the method scope is configured, the client must provide an access
         * token instead of an identity token for authorization purposes.
         * </p>
         * 
         * @param authorizationScopes
         *        A list of authorization scopes configured on the method. The scopes are used with a
         *        <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works
         *        by matching the method scopes against the scopes parsed from the access token in the incoming request.
         *        The method invocation is authorized if any method scopes matches a claimed scope in the access token.
         *        Otherwise, the invocation is not authorized. When the method scope is configured, the client must
         *        provide an access token instead of an identity token for authorization purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizationScopes(Collection<String> authorizationScopes);

        /**
         * <p>
         * A list of authorization scopes configured on the method. The scopes are used with a
         * <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works by
         * matching the method scopes against the scopes parsed from the access token in the incoming request. The
         * method invocation is authorized if any method scopes matches a claimed scope in the access token. Otherwise,
         * the invocation is not authorized. When the method scope is configured, the client must provide an access
         * token instead of an identity token for authorization purposes.
         * </p>
         * 
         * @param authorizationScopes
         *        A list of authorization scopes configured on the method. The scopes are used with a
         *        <code>COGNITO_USER_POOLS</code> authorizer to authorize the method invocation. The authorization works
         *        by matching the method scopes against the scopes parsed from the access token in the incoming request.
         *        The method invocation is authorized if any method scopes matches a claimed scope in the access token.
         *        Otherwise, the invocation is not authorized. When the method scope is configured, the client must
         *        provide an access token instead of an identity token for authorization purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizationScopes(String... authorizationScopes);
    }

    static final class BuilderImpl extends ApiGatewayResponse.BuilderImpl implements Builder {
        private String httpMethod;

        private String authorizationType;

        private String authorizerId;

        private Boolean apiKeyRequired;

        private String requestValidatorId;

        private String operationName;

        private Map<String, Boolean> requestParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> requestModels = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, MethodResponse> methodResponses = DefaultSdkAutoConstructMap.getInstance();

        private Integration methodIntegration;

        private List<String> authorizationScopes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetMethodResponse model) {
            super(model);
            httpMethod(model.httpMethod);
            authorizationType(model.authorizationType);
            authorizerId(model.authorizerId);
            apiKeyRequired(model.apiKeyRequired);
            requestValidatorId(model.requestValidatorId);
            operationName(model.operationName);
            requestParameters(model.requestParameters);
            requestModels(model.requestModels);
            methodResponses(model.methodResponses);
            methodIntegration(model.methodIntegration);
            authorizationScopes(model.authorizationScopes);
        }

        public final String getHttpMethod() {
            return httpMethod;
        }

        @Override
        public final Builder httpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public final void setHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
        }

        public final String getAuthorizationType() {
            return authorizationType;
        }

        @Override
        public final Builder authorizationType(String authorizationType) {
            this.authorizationType = authorizationType;
            return this;
        }

        public final void setAuthorizationType(String authorizationType) {
            this.authorizationType = authorizationType;
        }

        public final String getAuthorizerId() {
            return authorizerId;
        }

        @Override
        public final Builder authorizerId(String authorizerId) {
            this.authorizerId = authorizerId;
            return this;
        }

        public final void setAuthorizerId(String authorizerId) {
            this.authorizerId = authorizerId;
        }

        public final Boolean getApiKeyRequired() {
            return apiKeyRequired;
        }

        @Override
        public final Builder apiKeyRequired(Boolean apiKeyRequired) {
            this.apiKeyRequired = apiKeyRequired;
            return this;
        }

        public final void setApiKeyRequired(Boolean apiKeyRequired) {
            this.apiKeyRequired = apiKeyRequired;
        }

        public final String getRequestValidatorId() {
            return requestValidatorId;
        }

        @Override
        public final Builder requestValidatorId(String requestValidatorId) {
            this.requestValidatorId = requestValidatorId;
            return this;
        }

        public final void setRequestValidatorId(String requestValidatorId) {
            this.requestValidatorId = requestValidatorId;
        }

        public final String getOperationName() {
            return operationName;
        }

        @Override
        public final Builder operationName(String operationName) {
            this.operationName = operationName;
            return this;
        }

        public final void setOperationName(String operationName) {
            this.operationName = operationName;
        }

        public final Map<String, Boolean> getRequestParameters() {
            return requestParameters;
        }

        @Override
        public final Builder requestParameters(Map<String, Boolean> requestParameters) {
            this.requestParameters = MapOfStringToBooleanCopier.copy(requestParameters);
            return this;
        }

        public final void setRequestParameters(Map<String, Boolean> requestParameters) {
            this.requestParameters = MapOfStringToBooleanCopier.copy(requestParameters);
        }

        public final Map<String, String> getRequestModels() {
            return requestModels;
        }

        @Override
        public final Builder requestModels(Map<String, String> requestModels) {
            this.requestModels = MapOfStringToStringCopier.copy(requestModels);
            return this;
        }

        public final void setRequestModels(Map<String, String> requestModels) {
            this.requestModels = MapOfStringToStringCopier.copy(requestModels);
        }

        public final Map<String, MethodResponse.Builder> getMethodResponses() {
            return methodResponses != null ? CollectionUtils.mapValues(methodResponses, MethodResponse::toBuilder) : null;
        }

        @Override
        public final Builder methodResponses(Map<String, MethodResponse> methodResponses) {
            this.methodResponses = MapOfMethodResponseCopier.copy(methodResponses);
            return this;
        }

        public final void setMethodResponses(Map<String, MethodResponse.BuilderImpl> methodResponses) {
            this.methodResponses = MapOfMethodResponseCopier.copyFromBuilder(methodResponses);
        }

        public final Integration.Builder getMethodIntegration() {
            return methodIntegration != null ? methodIntegration.toBuilder() : null;
        }

        @Override
        public final Builder methodIntegration(Integration methodIntegration) {
            this.methodIntegration = methodIntegration;
            return this;
        }

        public final void setMethodIntegration(Integration.BuilderImpl methodIntegration) {
            this.methodIntegration = methodIntegration != null ? methodIntegration.build() : null;
        }

        public final Collection<String> getAuthorizationScopes() {
            return authorizationScopes;
        }

        @Override
        public final Builder authorizationScopes(Collection<String> authorizationScopes) {
            this.authorizationScopes = ListOfStringCopier.copy(authorizationScopes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder authorizationScopes(String... authorizationScopes) {
            authorizationScopes(Arrays.asList(authorizationScopes));
            return this;
        }

        public final void setAuthorizationScopes(Collection<String> authorizationScopes) {
            this.authorizationScopes = ListOfStringCopier.copy(authorizationScopes);
        }

        @Override
        public GetMethodResponse build() {
            return new GetMethodResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
