/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigatewayv2.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a new RouteResponse resource to represent a route response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateRouteResponseRequest extends ApiGatewayV2Request implements
        ToCopyableBuilder<CreateRouteResponseRequest.Builder, CreateRouteResponseRequest> {
    private static final SdkField<String> API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ApiId")
            .getter(getter(CreateRouteResponseRequest::apiId)).setter(setter(Builder::apiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("apiId").build()).build();

    private static final SdkField<String> MODEL_SELECTION_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelSelectionExpression").getter(getter(CreateRouteResponseRequest::modelSelectionExpression))
            .setter(setter(Builder::modelSelectionExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modelSelectionExpression").build())
            .build();

    private static final SdkField<Map<String, String>> RESPONSE_MODELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ResponseModels")
            .getter(getter(CreateRouteResponseRequest::responseModels))
            .setter(setter(Builder::responseModels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseModels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, ParameterConstraints>> RESPONSE_PARAMETERS_FIELD = SdkField
            .<Map<String, ParameterConstraints>> builder(MarshallingType.MAP)
            .memberName("ResponseParameters")
            .getter(getter(CreateRouteResponseRequest::responseParameters))
            .setter(setter(Builder::responseParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ParameterConstraints> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterConstraints::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ROUTE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RouteId").getter(getter(CreateRouteResponseRequest::routeId)).setter(setter(Builder::routeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("routeId").build()).build();

    private static final SdkField<String> ROUTE_RESPONSE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RouteResponseKey").getter(getter(CreateRouteResponseRequest::routeResponseKey))
            .setter(setter(Builder::routeResponseKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("routeResponseKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(API_ID_FIELD,
            MODEL_SELECTION_EXPRESSION_FIELD, RESPONSE_MODELS_FIELD, RESPONSE_PARAMETERS_FIELD, ROUTE_ID_FIELD,
            ROUTE_RESPONSE_KEY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("apiId", API_ID_FIELD);
                    put("modelSelectionExpression", MODEL_SELECTION_EXPRESSION_FIELD);
                    put("responseModels", RESPONSE_MODELS_FIELD);
                    put("responseParameters", RESPONSE_PARAMETERS_FIELD);
                    put("routeId", ROUTE_ID_FIELD);
                    put("routeResponseKey", ROUTE_RESPONSE_KEY_FIELD);
                }
            });

    private final String apiId;

    private final String modelSelectionExpression;

    private final Map<String, String> responseModels;

    private final Map<String, ParameterConstraints> responseParameters;

    private final String routeId;

    private final String routeResponseKey;

    private CreateRouteResponseRequest(BuilderImpl builder) {
        super(builder);
        this.apiId = builder.apiId;
        this.modelSelectionExpression = builder.modelSelectionExpression;
        this.responseModels = builder.responseModels;
        this.responseParameters = builder.responseParameters;
        this.routeId = builder.routeId;
        this.routeResponseKey = builder.routeResponseKey;
    }

    /**
     * <p>
     * The API identifier.
     * </p>
     * 
     * @return The API identifier.
     */
    public final String apiId() {
        return apiId;
    }

    /**
     * <p>
     * The model selection expression for the route response. Supported only for WebSocket APIs.
     * </p>
     * 
     * @return The model selection expression for the route response. Supported only for WebSocket APIs.
     */
    public final String modelSelectionExpression() {
        return modelSelectionExpression;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseModels property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseModels() {
        return responseModels != null && !(responseModels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The response models for the route response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseModels} method.
     * </p>
     * 
     * @return The response models for the route response.
     */
    public final Map<String, String> responseModels() {
        return responseModels;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseParameters() {
        return responseParameters != null && !(responseParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The route response parameters.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseParameters} method.
     * </p>
     * 
     * @return The route response parameters.
     */
    public final Map<String, ParameterConstraints> responseParameters() {
        return responseParameters;
    }

    /**
     * <p>
     * The route ID.
     * </p>
     * 
     * @return The route ID.
     */
    public final String routeId() {
        return routeId;
    }

    /**
     * <p>
     * The route response key.
     * </p>
     * 
     * @return The route response key.
     */
    public final String routeResponseKey() {
        return routeResponseKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(apiId());
        hashCode = 31 * hashCode + Objects.hashCode(modelSelectionExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseModels() ? responseModels() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseParameters() ? responseParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(routeId());
        hashCode = 31 * hashCode + Objects.hashCode(routeResponseKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateRouteResponseRequest)) {
            return false;
        }
        CreateRouteResponseRequest other = (CreateRouteResponseRequest) obj;
        return Objects.equals(apiId(), other.apiId())
                && Objects.equals(modelSelectionExpression(), other.modelSelectionExpression())
                && hasResponseModels() == other.hasResponseModels() && Objects.equals(responseModels(), other.responseModels())
                && hasResponseParameters() == other.hasResponseParameters()
                && Objects.equals(responseParameters(), other.responseParameters()) && Objects.equals(routeId(), other.routeId())
                && Objects.equals(routeResponseKey(), other.routeResponseKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateRouteResponseRequest").add("ApiId", apiId())
                .add("ModelSelectionExpression", modelSelectionExpression())
                .add("ResponseModels", hasResponseModels() ? responseModels() : null)
                .add("ResponseParameters", hasResponseParameters() ? responseParameters() : null).add("RouteId", routeId())
                .add("RouteResponseKey", routeResponseKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApiId":
            return Optional.ofNullable(clazz.cast(apiId()));
        case "ModelSelectionExpression":
            return Optional.ofNullable(clazz.cast(modelSelectionExpression()));
        case "ResponseModels":
            return Optional.ofNullable(clazz.cast(responseModels()));
        case "ResponseParameters":
            return Optional.ofNullable(clazz.cast(responseParameters()));
        case "RouteId":
            return Optional.ofNullable(clazz.cast(routeId()));
        case "RouteResponseKey":
            return Optional.ofNullable(clazz.cast(routeResponseKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateRouteResponseRequest, T> g) {
        return obj -> g.apply((CreateRouteResponseRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayV2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateRouteResponseRequest> {
        /**
         * <p>
         * The API identifier.
         * </p>
         * 
         * @param apiId
         *        The API identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiId(String apiId);

        /**
         * <p>
         * The model selection expression for the route response. Supported only for WebSocket APIs.
         * </p>
         * 
         * @param modelSelectionExpression
         *        The model selection expression for the route response. Supported only for WebSocket APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelSelectionExpression(String modelSelectionExpression);

        /**
         * <p>
         * The response models for the route response.
         * </p>
         * 
         * @param responseModels
         *        The response models for the route response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseModels(Map<String, String> responseModels);

        /**
         * <p>
         * The route response parameters.
         * </p>
         * 
         * @param responseParameters
         *        The route response parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseParameters(Map<String, ParameterConstraints> responseParameters);

        /**
         * <p>
         * The route ID.
         * </p>
         * 
         * @param routeId
         *        The route ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeId(String routeId);

        /**
         * <p>
         * The route response key.
         * </p>
         * 
         * @param routeResponseKey
         *        The route response key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeResponseKey(String routeResponseKey);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayV2Request.BuilderImpl implements Builder {
        private String apiId;

        private String modelSelectionExpression;

        private Map<String, String> responseModels = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, ParameterConstraints> responseParameters = DefaultSdkAutoConstructMap.getInstance();

        private String routeId;

        private String routeResponseKey;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateRouteResponseRequest model) {
            super(model);
            apiId(model.apiId);
            modelSelectionExpression(model.modelSelectionExpression);
            responseModels(model.responseModels);
            responseParameters(model.responseParameters);
            routeId(model.routeId);
            routeResponseKey(model.routeResponseKey);
        }

        public final String getApiId() {
            return apiId;
        }

        public final void setApiId(String apiId) {
            this.apiId = apiId;
        }

        @Override
        public final Builder apiId(String apiId) {
            this.apiId = apiId;
            return this;
        }

        public final String getModelSelectionExpression() {
            return modelSelectionExpression;
        }

        public final void setModelSelectionExpression(String modelSelectionExpression) {
            this.modelSelectionExpression = modelSelectionExpression;
        }

        @Override
        public final Builder modelSelectionExpression(String modelSelectionExpression) {
            this.modelSelectionExpression = modelSelectionExpression;
            return this;
        }

        public final Map<String, String> getResponseModels() {
            if (responseModels instanceof SdkAutoConstructMap) {
                return null;
            }
            return responseModels;
        }

        public final void setResponseModels(Map<String, String> responseModels) {
            this.responseModels = RouteModelsCopier.copy(responseModels);
        }

        @Override
        public final Builder responseModels(Map<String, String> responseModels) {
            this.responseModels = RouteModelsCopier.copy(responseModels);
            return this;
        }

        public final Map<String, ParameterConstraints.Builder> getResponseParameters() {
            Map<String, ParameterConstraints.Builder> result = RouteParametersCopier.copyToBuilder(this.responseParameters);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setResponseParameters(Map<String, ParameterConstraints.BuilderImpl> responseParameters) {
            this.responseParameters = RouteParametersCopier.copyFromBuilder(responseParameters);
        }

        @Override
        public final Builder responseParameters(Map<String, ParameterConstraints> responseParameters) {
            this.responseParameters = RouteParametersCopier.copy(responseParameters);
            return this;
        }

        public final String getRouteId() {
            return routeId;
        }

        public final void setRouteId(String routeId) {
            this.routeId = routeId;
        }

        @Override
        public final Builder routeId(String routeId) {
            this.routeId = routeId;
            return this;
        }

        public final String getRouteResponseKey() {
            return routeResponseKey;
        }

        public final void setRouteResponseKey(String routeResponseKey) {
            this.routeResponseKey = routeResponseKey;
        }

        @Override
        public final Builder routeResponseKey(String routeResponseKey) {
            this.routeResponseKey = routeResponseKey;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateRouteResponseRequest build() {
            return new CreateRouteResponseRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
