/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appconfig.model.AppConfigException;
import software.amazon.awssdk.services.appconfig.model.AppConfigRequest;
import software.amazon.awssdk.services.appconfig.model.BadRequestException;
import software.amazon.awssdk.services.appconfig.model.ConflictException;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.GetApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.GetApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.InternalServerException;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appconfig.model.PayloadTooLargeException;
import software.amazon.awssdk.services.appconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.TagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.TagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UntagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.UntagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationResponse;
import software.amazon.awssdk.services.appconfig.paginators.ListApplicationsIterable;
import software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesIterable;
import software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesIterable;
import software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsIterable;
import software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsIterable;
import software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsIterable;
import software.amazon.awssdk.services.appconfig.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListConfigurationProfilesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListHostedConfigurationVersionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StopDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ValidateConfigurationRequestMarshaller;

/**
 * Internal implementation of {@link AppConfigClient}.
 *
 * @see AppConfigClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppConfigClient implements AppConfigClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppConfigClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * An application in AppConfig is a logical unit of code that provides capabilities for your customers. For example,
     * an application can be a microservice that runs on Amazon EC2 instances, a mobile application installed by your
     * users, a serverless application using Amazon API Gateway and AWS Lambda, or any system you run on behalf of
     * others.
     * </p>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                .withOperationName("CreateApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationRequest)
                .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Information that enables AppConfig to access the configuration source. Valid configuration sources include
     * Systems Manager (SSM) documents, SSM Parameter Store parameters, and Amazon S3 objects. A configuration profile
     * includes the following information.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Uri location of the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * The AWS Identity and Access Management (IAM) role that provides access to the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * A validator for the configuration data. Available validators include either a JSON Schema or an AWS Lambda
     * function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-configuration-and-profile.html"
     * >Create a Configuration and a Configuration Profile</a> in the <i>AWS AppConfig User Guide</i>.
     * </p>
     *
     * @param createConfigurationProfileRequest
     * @return Result of the CreateConfigurationProfile operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConfigurationProfileResponse createConfigurationProfile(
            CreateConfigurationProfileRequest createConfigurationProfileRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateConfigurationProfileRequest, CreateConfigurationProfileResponse>()
                        .withOperationName("CreateConfigurationProfile").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createConfigurationProfileRequest)
                        .withMarshaller(new CreateConfigurationProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * A deployment strategy defines important criteria for rolling out your configuration to the designated targets. A
     * deployment strategy includes: the overall duration required, a percentage of targets to receive the deployment
     * during each interval, an algorithm that defines how percentage grows, and bake time.
     * </p>
     *
     * @param createDeploymentStrategyRequest
     * @return Result of the CreateDeploymentStrategy operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDeploymentStrategyResponse createDeploymentStrategy(
            CreateDeploymentStrategyRequest createDeploymentStrategyRequest) throws InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateDeploymentStrategyRequest, CreateDeploymentStrategyResponse>()
                        .withOperationName("CreateDeploymentStrategy").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createDeploymentStrategyRequest)
                        .withMarshaller(new CreateDeploymentStrategyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * For each application, you define one or more environments. An environment is a logical deployment group of
     * AppConfig targets, such as applications in a <code>Beta</code> or <code>Production</code> environment. You can
     * also define environments for application subcomponents such as the <code>Web</code>, <code>Mobile</code> and
     * <code>Back-end</code> components for your application. You can configure Amazon CloudWatch alarms for each
     * environment. The system monitors alarms during a configuration deployment. If an alarm is triggered, the system
     * rolls back the configuration.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws InternalServerException, ResourceNotFoundException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                .withOperationName("CreateEnvironment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createEnvironmentRequest)
                .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Create a new configuration in the AppConfig configuration store.
     * </p>
     *
     * @param createHostedConfigurationVersionRequest
     * @return Result of the CreateHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ServiceQuotaExceededException
     *         The number of hosted configuration versions exceeds the limit for the AppConfig configuration store.
     *         Delete one or more versions and try again.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws PayloadTooLargeException
     *         The configuration size is too large.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHostedConfigurationVersionResponse createHostedConfigurationVersion(
            CreateHostedConfigurationVersionRequest createHostedConfigurationVersionRequest) throws BadRequestException,
            ServiceQuotaExceededException, ResourceNotFoundException, ConflictException, PayloadTooLargeException,
            InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<CreateHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateHostedConfigurationVersionRequest, CreateHostedConfigurationVersionResponse>()
                        .withOperationName("CreateHostedConfigurationVersion").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createHostedConfigurationVersionRequest)
                        .withMarshaller(new CreateHostedConfigurationVersionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete an application. Deleting an application does not delete a configuration from a host.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return Result of the DeleteApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteApplicationResponse deleteApplication(DeleteApplicationRequest deleteApplicationRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                .withOperationName("DeleteApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteApplicationRequest)
                .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete a configuration profile. Deleting a configuration profile does not delete a configuration from a host.
     * </p>
     *
     * @param deleteConfigurationProfileRequest
     * @return Result of the DeleteConfigurationProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConfigurationProfileResponse deleteConfigurationProfile(
            DeleteConfigurationProfileRequest deleteConfigurationProfileRequest) throws ResourceNotFoundException,
            ConflictException, InternalServerException, BadRequestException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteConfigurationProfileRequest, DeleteConfigurationProfileResponse>()
                        .withOperationName("DeleteConfigurationProfile").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteConfigurationProfileRequest)
                        .withMarshaller(new DeleteConfigurationProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete a deployment strategy. Deleting a deployment strategy does not delete a configuration from a host.
     * </p>
     *
     * @param deleteDeploymentStrategyRequest
     * @return Result of the DeleteDeploymentStrategy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDeploymentStrategyResponse deleteDeploymentStrategy(
            DeleteDeploymentStrategyRequest deleteDeploymentStrategyRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteDeploymentStrategyRequest, DeleteDeploymentStrategyResponse>()
                        .withOperationName("DeleteDeploymentStrategy").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteDeploymentStrategyRequest)
                        .withMarshaller(new DeleteDeploymentStrategyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete an environment. Deleting an environment does not delete a configuration from a host.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, ConflictException, InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                .withOperationName("DeleteEnvironment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteEnvironmentRequest)
                .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Delete a version of a configuration from the AppConfig configuration store.
     * </p>
     *
     * @param deleteHostedConfigurationVersionRequest
     * @return Result of the DeleteHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHostedConfigurationVersionResponse deleteHostedConfigurationVersion(
            DeleteHostedConfigurationVersionRequest deleteHostedConfigurationVersionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteHostedConfigurationVersionRequest, DeleteHostedConfigurationVersionResponse>()
                        .withOperationName("DeleteHostedConfigurationVersion").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteHostedConfigurationVersionRequest)
                        .withMarshaller(new DeleteHostedConfigurationVersionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieve information about an application.
     * </p>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                .withOperationName("GetApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationRequest)
                .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Receive information about a configuration.
     * </p>
     * <important>
     * <p>
     * AWS AppConfig uses the value of the <code>ClientConfigurationVersion</code> parameter to identify the
     * configuration version on your clients. If you don’t send <code>ClientConfigurationVersion</code> with each call
     * to <code>GetConfiguration</code>, your clients receive the current configuration. You are charged each time your
     * clients receive a configuration.
     * </p>
     * <p>
     * To avoid excess charges, we recommend that you include the <code>ClientConfigurationVersion</code> value with
     * every call to <code>GetConfiguration</code>. This value must be saved on your client. Subsequent calls to
     * <code>GetConfiguration</code> must pass this value by using the <code>ClientConfigurationVersion</code>
     * parameter.
     * </p>
     * </important>
     *
     * @param getConfigurationRequest
     * @return Result of the GetConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetConfigurationResponse getConfiguration(GetConfigurationRequest getConfigurationRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetConfigurationRequest, GetConfigurationResponse>()
                .withOperationName("GetConfiguration").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getConfigurationRequest)
                .withMarshaller(new GetConfigurationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieve information about a configuration profile.
     * </p>
     *
     * @param getConfigurationProfileRequest
     * @return Result of the GetConfigurationProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConfigurationProfileResponse getConfigurationProfile(GetConfigurationProfileRequest getConfigurationProfileRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetConfigurationProfileRequest, GetConfigurationProfileResponse>()
                .withOperationName("GetConfigurationProfile").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getConfigurationProfileRequest)
                .withMarshaller(new GetConfigurationProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieve information about a configuration deployment.
     * </p>
     *
     * @param getDeploymentRequest
     * @return Result of the GetDeployment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDeploymentResponse getDeployment(GetDeploymentRequest getDeploymentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                .withOperationName("GetDeployment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDeploymentRequest)
                .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieve information about a deployment strategy. A deployment strategy defines important criteria for rolling
     * out your configuration to the designated targets. A deployment strategy includes: the overall duration required,
     * a percentage of targets to receive the deployment during each interval, an algorithm that defines how percentage
     * grows, and bake time.
     * </p>
     *
     * @param getDeploymentStrategyRequest
     * @return Result of the GetDeploymentStrategy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDeploymentStrategyResponse getDeploymentStrategy(GetDeploymentStrategyRequest getDeploymentStrategyRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDeploymentStrategyRequest, GetDeploymentStrategyResponse>()
                .withOperationName("GetDeploymentStrategy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDeploymentStrategyRequest)
                .withMarshaller(new GetDeploymentStrategyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieve information about an environment. An environment is a logical deployment group of AppConfig
     * applications, such as applications in a <code>Production</code> environment or in an <code>EU_Region</code>
     * environment. Each configuration deployment targets an environment. You can enable one or more Amazon CloudWatch
     * alarms for an environment. If an alarm is triggered during a deployment, AppConfig roles back the configuration.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                .withOperationName("GetEnvironment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getEnvironmentRequest)
                .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Get information about a specific configuration version.
     * </p>
     *
     * @param getHostedConfigurationVersionRequest
     * @return Result of the GetHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHostedConfigurationVersionResponse getHostedConfigurationVersion(
            GetHostedConfigurationVersionRequest getHostedConfigurationVersionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetHostedConfigurationVersionRequest, GetHostedConfigurationVersionResponse>()
                        .withOperationName("GetHostedConfigurationVersion").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getHostedConfigurationVersionRequest)
                        .withMarshaller(new GetHostedConfigurationVersionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all applications in your AWS account.
     * </p>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest)
            throws InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                .withOperationName("ListApplications").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationsRequest)
                .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List all applications in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListApplicationsIterable responses = client
     *             .listApplicationsPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListApplicationsIterable responses = client.listApplicationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest)} operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListApplicationsIterable listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest)
            throws InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        return new ListApplicationsIterable(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return Result of the ListConfigurationProfiles operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListConfigurationProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConfigurationProfilesResponse listConfigurationProfiles(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListConfigurationProfilesRequest, ListConfigurationProfilesResponse>()
                        .withOperationName("ListConfigurationProfiles").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listConfigurationProfilesRequest)
                        .withMarshaller(new ListConfigurationProfilesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConfigurationProfiles(software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesIterable responses = client.listConfigurationProfilesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesIterable responses = client
     *             .listConfigurationProfilesPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesIterable responses = client.listConfigurationProfilesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConfigurationProfiles(software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListConfigurationProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConfigurationProfilesIterable listConfigurationProfilesPaginator(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        return new ListConfigurationProfilesIterable(this, applyPaginatorUserAgent(listConfigurationProfilesRequest));
    }

    /**
     * <p>
     * List deployment strategies.
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return Result of the ListDeploymentStrategies operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeploymentStrategies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeploymentStrategiesResponse listDeploymentStrategies(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) throws InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeploymentStrategiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListDeploymentStrategiesRequest, ListDeploymentStrategiesResponse>()
                        .withOperationName("ListDeploymentStrategies").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listDeploymentStrategiesRequest)
                        .withMarshaller(new ListDeploymentStrategiesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List deployment strategies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeploymentStrategies(software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesIterable responses = client.listDeploymentStrategiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesIterable responses = client
     *             .listDeploymentStrategiesPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesIterable responses = client.listDeploymentStrategiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeploymentStrategies(software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeploymentStrategies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeploymentStrategiesIterable listDeploymentStrategiesPaginator(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) throws InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        return new ListDeploymentStrategiesIterable(this, applyPaginatorUserAgent(listDeploymentStrategiesRequest));
    }

    /**
     * <p>
     * Lists the deployments for an environment.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return Result of the ListDeployments operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeployments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDeploymentsResponse listDeployments(ListDeploymentsRequest listDeploymentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDeploymentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                .withOperationName("ListDeployments").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDeploymentsRequest)
                .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the deployments for an environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeployments(software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsIterable responses = client.listDeploymentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsIterable responses = client
     *             .listDeploymentsPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsIterable responses = client.listDeploymentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeployments(software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest)} operation.</b>
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeployments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDeploymentsIterable listDeploymentsPaginator(ListDeploymentsRequest listDeploymentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        return new ListDeploymentsIterable(this, applyPaginatorUserAgent(listDeploymentsRequest));
    }

    /**
     * <p>
     * List the environments for an application.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                .withOperationName("ListEnvironments").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listEnvironmentsRequest)
                .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the environments for an application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsIterable responses = client
     *             .listEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListEnvironmentsIterable listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        return new ListEnvironmentsIterable(this, applyPaginatorUserAgent(listEnvironmentsRequest));
    }

    /**
     * <p>
     * View a list of configurations stored in the AppConfig configuration store by version.
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return Result of the ListHostedConfigurationVersions operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListHostedConfigurationVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHostedConfigurationVersionsResponse listHostedConfigurationVersions(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHostedConfigurationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListHostedConfigurationVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListHostedConfigurationVersionsRequest, ListHostedConfigurationVersionsResponse>()
                        .withOperationName("ListHostedConfigurationVersions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listHostedConfigurationVersionsRequest)
                        .withMarshaller(new ListHostedConfigurationVersionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * View a list of configurations stored in the AppConfig configuration store by version.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHostedConfigurationVersions(software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsIterable responses = client.listHostedConfigurationVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsIterable responses = client
     *             .listHostedConfigurationVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsIterable responses = client.listHostedConfigurationVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHostedConfigurationVersions(software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListHostedConfigurationVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHostedConfigurationVersionsIterable listHostedConfigurationVersionsPaginator(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        return new ListHostedConfigurationVersionsIterable(this, applyPaginatorUserAgent(listHostedConfigurationVersionsRequest));
    }

    /**
     * <p>
     * Retrieves the list of key-value tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, BadRequestException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Starts a deployment.
     * </p>
     *
     * @param startDeploymentRequest
     * @return Result of the StartDeployment operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.StartDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartDeploymentResponse startDeployment(StartDeploymentRequest startDeploymentRequest) throws BadRequestException,
            ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                .withOperationName("StartDeployment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startDeploymentRequest)
                .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stops a deployment. This API action works only on deployments that have a status of <code>DEPLOYING</code>. This
     * action moves the deployment to a status of <code>ROLLED_BACK</code>.
     * </p>
     *
     * @param stopDeploymentRequest
     * @return Result of the StopDeployment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.StopDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StopDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopDeploymentResponse stopDeployment(StopDeploymentRequest stopDeploymentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopDeploymentRequest, StopDeploymentResponse>()
                .withOperationName("StopDeployment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopDeploymentRequest)
                .withMarshaller(new StopDeploymentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Metadata to assign to an AppConfig resource. Tags help organize and categorize your AppConfig resources. Each tag
     * consists of a key and an optional value, both of which you define. You can specify a maximum of 50 tags for a
     * resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a tag key and value from an AppConfig resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                .withOperationName("UpdateApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationRequest)
                .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a configuration profile.
     * </p>
     *
     * @param updateConfigurationProfileRequest
     * @return Result of the UpdateConfigurationProfile operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConfigurationProfileResponse updateConfigurationProfile(
            UpdateConfigurationProfileRequest updateConfigurationProfileRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateConfigurationProfileRequest, UpdateConfigurationProfileResponse>()
                        .withOperationName("UpdateConfigurationProfile").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateConfigurationProfileRequest)
                        .withMarshaller(new UpdateConfigurationProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a deployment strategy.
     * </p>
     *
     * @param updateDeploymentStrategyRequest
     * @return Result of the UpdateDeploymentStrategy operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDeploymentStrategyResponse updateDeploymentStrategy(
            UpdateDeploymentStrategyRequest updateDeploymentStrategyRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateDeploymentStrategyRequest, UpdateDeploymentStrategyResponse>()
                        .withOperationName("UpdateDeploymentStrategy").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateDeploymentStrategyRequest)
                        .withMarshaller(new UpdateDeploymentStrategyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates an environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                .withOperationName("UpdateEnvironment").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateEnvironmentRequest)
                .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Uses the validators in a configuration profile to validate a configuration.
     * </p>
     *
     * @param validateConfigurationRequest
     * @return Result of the ValidateConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ValidateConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ValidateConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidateConfigurationResponse validateConfiguration(ValidateConfigurationRequest validateConfigurationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidateConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ValidateConfigurationRequest, ValidateConfigurationResponse>()
                .withOperationName("ValidateConfiguration").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(validateConfigurationRequest)
                .withMarshaller(new ValidateConfigurationRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PayloadTooLargeException")
                                .exceptionBuilderSupplier(PayloadTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends AppConfigRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
