/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appconfig.model.AppConfigException;
import software.amazon.awssdk.services.appconfig.model.AppConfigRequest;
import software.amazon.awssdk.services.appconfig.model.BadRequestException;
import software.amazon.awssdk.services.appconfig.model.ConflictException;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.GetApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.GetApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.InternalServerException;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appconfig.model.PayloadTooLargeException;
import software.amazon.awssdk.services.appconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.TagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.TagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UntagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.UntagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationResponse;
import software.amazon.awssdk.services.appconfig.paginators.ListApplicationsPublisher;
import software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesPublisher;
import software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesPublisher;
import software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsPublisher;
import software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsPublisher;
import software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsPublisher;
import software.amazon.awssdk.services.appconfig.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListConfigurationProfilesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListHostedConfigurationVersionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StopDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ValidateConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppConfigAsyncClient}.
 *
 * @see AppConfigAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppConfigAsyncClient implements AppConfigAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppConfigAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppConfigAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * An application in AppConfig is a logical unit of code that provides capabilities for your customers. For example,
     * an application can be a microservice that runs on Amazon EC2 instances, a mobile application installed by your
     * users, a serverless application using Amazon API Gateway and AWS Lambda, or any system you run on behalf of
     * others.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication")
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Information that enables AppConfig to access the configuration source. Valid configuration sources include
     * Systems Manager (SSM) documents, SSM Parameter Store parameters, and Amazon S3 objects. A configuration profile
     * includes the following information.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Uri location of the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * The AWS Identity and Access Management (IAM) role that provides access to the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * A validator for the configuration data. Available validators include either a JSON Schema or an AWS Lambda
     * function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/systems-manager/latest/userguide/appconfig-creating-configuration-and-profile.html"
     * >Create a Configuration and a Configuration Profile</a> in the <i>AWS AppConfig User Guide</i>.
     * </p>
     *
     * @param createConfigurationProfileRequest
     * @return A Java Future containing the result of the CreateConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationProfileResponse> createConfigurationProfile(
            CreateConfigurationProfileRequest createConfigurationProfileRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationProfileRequest, CreateConfigurationProfileResponse>()
                            .withOperationName("CreateConfigurationProfile")
                            .withMarshaller(new CreateConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConfigurationProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createConfigurationProfileRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    createConfigurationProfileRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A deployment strategy defines important criteria for rolling out your configuration to the designated targets. A
     * deployment strategy includes: the overall duration required, a percentage of targets to receive the deployment
     * during each interval, an algorithm that defines how percentage grows, and bake time.
     * </p>
     *
     * @param createDeploymentStrategyRequest
     * @return A Java Future containing the result of the CreateDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentStrategyResponse> createDeploymentStrategy(
            CreateDeploymentStrategyRequest createDeploymentStrategyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentStrategyRequest, CreateDeploymentStrategyResponse>()
                            .withOperationName("CreateDeploymentStrategy")
                            .withMarshaller(new CreateDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDeploymentStrategyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDeploymentStrategyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentStrategyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For each application, you define one or more environments. An environment is a logical deployment group of
     * AppConfig targets, such as applications in a <code>Beta</code> or <code>Production</code> environment. You can
     * also define environments for application subcomponents such as the <code>Web</code>, <code>Mobile</code> and
     * <code>Back-end</code> components for your application. You can configure Amazon CloudWatch alarms for each
     * environment. The system monitors alarms during a configuration deployment. If an alarm is triggered, the system
     * rolls back the configuration.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                            .withOperationName("CreateEnvironment")
                            .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createEnvironmentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create a new configuration in the AppConfig configuration store.
     * </p>
     *
     * @param createHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the CreateHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ServiceQuotaExceededException The number of hosted configuration versions exceeds the limit for the
     *         AppConfig configuration store. Delete one or more versions and try again.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>PayloadTooLargeException The configuration size is too large.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHostedConfigurationVersionResponse> createHostedConfigurationVersion(
            CreateHostedConfigurationVersionRequest createHostedConfigurationVersionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<CreateHostedConfigurationVersionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHostedConfigurationVersionRequest, CreateHostedConfigurationVersionResponse>()
                            .withOperationName("CreateHostedConfigurationVersion")
                            .withMarshaller(new CreateHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createHostedConfigurationVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createHostedConfigurationVersionRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<CreateHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    createHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an application. Deleting an application does not delete a configuration from a host.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication")
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a configuration profile. Deleting a configuration profile does not delete a configuration from a host.
     * </p>
     *
     * @param deleteConfigurationProfileRequest
     * @return A Java Future containing the result of the DeleteConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationProfileResponse> deleteConfigurationProfile(
            DeleteConfigurationProfileRequest deleteConfigurationProfileRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationProfileRequest, DeleteConfigurationProfileResponse>()
                            .withOperationName("DeleteConfigurationProfile")
                            .withMarshaller(new DeleteConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConfigurationProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteConfigurationProfileRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    deleteConfigurationProfileRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a deployment strategy. Deleting a deployment strategy does not delete a configuration from a host.
     * </p>
     *
     * @param deleteDeploymentStrategyRequest
     * @return A Java Future containing the result of the DeleteDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeploymentStrategyResponse> deleteDeploymentStrategy(
            DeleteDeploymentStrategyRequest deleteDeploymentStrategyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeploymentStrategyRequest, DeleteDeploymentStrategyResponse>()
                            .withOperationName("DeleteDeploymentStrategy")
                            .withMarshaller(new DeleteDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDeploymentStrategyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDeploymentStrategyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeploymentStrategyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an environment. Deleting an environment does not delete a configuration from a host.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment")
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteEnvironmentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete a version of a configuration from the AppConfig configuration store.
     * </p>
     *
     * @param deleteHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the DeleteHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHostedConfigurationVersionResponse> deleteHostedConfigurationVersion(
            DeleteHostedConfigurationVersionRequest deleteHostedConfigurationVersionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHostedConfigurationVersionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHostedConfigurationVersionRequest, DeleteHostedConfigurationVersionResponse>()
                            .withOperationName("DeleteHostedConfigurationVersion")
                            .withMarshaller(new DeleteHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteHostedConfigurationVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteHostedConfigurationVersionRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    deleteHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about an application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication")
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Receive information about a configuration.
     * </p>
     * <important>
     * <p>
     * AWS AppConfig uses the value of the <code>ClientConfigurationVersion</code> parameter to identify the
     * configuration version on your clients. If you don’t send <code>ClientConfigurationVersion</code> with each call
     * to <code>GetConfiguration</code>, your clients receive the current configuration. You are charged each time your
     * clients receive a configuration.
     * </p>
     * <p>
     * To avoid excess charges, we recommend that you include the <code>ClientConfigurationVersion</code> value with
     * every call to <code>GetConfiguration</code>. This value must be saved on your client. Subsequent calls to
     * <code>GetConfiguration</code> must pass this value by using the <code>ClientConfigurationVersion</code>
     * parameter.
     * </p>
     * </important>
     *
     * @param getConfigurationRequest
     * @return A Java Future containing the result of the GetConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationResponse> getConfiguration(GetConfigurationRequest getConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationRequest, GetConfigurationResponse>()
                            .withOperationName("GetConfiguration")
                            .withMarshaller(new GetConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getConfigurationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about a configuration profile.
     * </p>
     *
     * @param getConfigurationProfileRequest
     * @return A Java Future containing the result of the GetConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationProfileResponse> getConfigurationProfile(
            GetConfigurationProfileRequest getConfigurationProfileRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationProfileRequest, GetConfigurationProfileResponse>()
                            .withOperationName("GetConfigurationProfile")
                            .withMarshaller(new GetConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfigurationProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getConfigurationProfileRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationProfileRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about a configuration deployment.
     * </p>
     *
     * @param getDeploymentRequest
     * @return A Java Future containing the result of the GetDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentResponse> getDeployment(GetDeploymentRequest getDeploymentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                            .withOperationName("GetDeployment")
                            .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about a deployment strategy. A deployment strategy defines important criteria for rolling
     * out your configuration to the designated targets. A deployment strategy includes: the overall duration required,
     * a percentage of targets to receive the deployment during each interval, an algorithm that defines how percentage
     * grows, and bake time.
     * </p>
     *
     * @param getDeploymentStrategyRequest
     * @return A Java Future containing the result of the GetDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentStrategyResponse> getDeploymentStrategy(
            GetDeploymentStrategyRequest getDeploymentStrategyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentStrategyRequest, GetDeploymentStrategyResponse>()
                            .withOperationName("GetDeploymentStrategy")
                            .withMarshaller(new GetDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDeploymentStrategyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDeploymentStrategyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentStrategyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve information about an environment. An environment is a logical deployment group of AppConfig
     * applications, such as applications in a <code>Production</code> environment or in an <code>EU_Region</code>
     * environment. Each configuration deployment targets an environment. You can enable one or more Amazon CloudWatch
     * alarms for an environment. If an alarm is triggered during a deployment, AppConfig roles back the configuration.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                            .withOperationName("GetEnvironment")
                            .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEnvironmentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about a specific configuration version.
     * </p>
     *
     * @param getHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the GetHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetHostedConfigurationVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHostedConfigurationVersionResponse> getHostedConfigurationVersion(
            GetHostedConfigurationVersionRequest getHostedConfigurationVersionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHostedConfigurationVersionRequest, GetHostedConfigurationVersionResponse>()
                            .withOperationName("GetHostedConfigurationVersion")
                            .withMarshaller(new GetHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getHostedConfigurationVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getHostedConfigurationVersionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    getHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all applications in your AWS account.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications")
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listApplicationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listApplicationsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all applications in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listApplications(software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListApplicationsPublisher publisher = client.listApplicationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listApplications(software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest)} operation.</b>
     * </p>
     *
     * @param listApplicationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListApplications
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS API
     *      Documentation</a>
     */
    public ListApplicationsPublisher listApplicationsPaginator(ListApplicationsRequest listApplicationsRequest) {
        return new ListApplicationsPublisher(this, applyPaginatorUserAgent(listApplicationsRequest));
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return A Java Future containing the result of the ListConfigurationProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListConfigurationProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationProfilesResponse> listConfigurationProfiles(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationProfilesRequest, ListConfigurationProfilesResponse>()
                            .withOperationName("ListConfigurationProfiles")
                            .withMarshaller(new ListConfigurationProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConfigurationProfilesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listConfigurationProfilesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListConfigurationProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listConfigurationProfilesRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConfigurationProfiles(software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesPublisher publisher = client.listConfigurationProfilesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListConfigurationProfilesPublisher publisher = client.listConfigurationProfilesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConfigurationProfiles(software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListConfigurationProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    public ListConfigurationProfilesPublisher listConfigurationProfilesPaginator(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) {
        return new ListConfigurationProfilesPublisher(this, applyPaginatorUserAgent(listConfigurationProfilesRequest));
    }

    /**
     * <p>
     * List deployment strategies.
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return A Java Future containing the result of the ListDeploymentStrategies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeploymentStrategies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentStrategiesResponse> listDeploymentStrategies(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeploymentStrategies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentStrategiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeploymentStrategiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentStrategiesRequest, ListDeploymentStrategiesResponse>()
                            .withOperationName("ListDeploymentStrategies")
                            .withMarshaller(new ListDeploymentStrategiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeploymentStrategiesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDeploymentStrategiesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListDeploymentStrategiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentStrategiesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List deployment strategies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeploymentStrategies(software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesPublisher publisher = client.listDeploymentStrategiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentStrategiesPublisher publisher = client.listDeploymentStrategiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeploymentStrategies(software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeploymentStrategies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDeploymentStrategiesPublisher listDeploymentStrategiesPaginator(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) {
        return new ListDeploymentStrategiesPublisher(this, applyPaginatorUserAgent(listDeploymentStrategiesRequest));
    }

    /**
     * <p>
     * Lists the deployments for an environment.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A Java Future containing the result of the ListDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeployments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentsResponse> listDeployments(ListDeploymentsRequest listDeploymentsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                            .withOperationName("ListDeployments")
                            .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeploymentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDeploymentsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the deployments for an environment.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeployments(software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsPublisher publisher = client.listDeploymentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListDeploymentsPublisher publisher = client.listDeploymentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeployments(software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest)} operation.</b>
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeployments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    public ListDeploymentsPublisher listDeploymentsPaginator(ListDeploymentsRequest listDeploymentsRequest) {
        return new ListDeploymentsPublisher(this, applyPaginatorUserAgent(listDeploymentsRequest));
    }

    /**
     * <p>
     * List the environments for an application.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments")
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEnvironmentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEnvironmentsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List the environments for an application.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEnvironments(software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListEnvironmentsPublisher publisher = client.listEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEnvironments(software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest)} operation.</b>
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    public ListEnvironmentsPublisher listEnvironmentsPaginator(ListEnvironmentsRequest listEnvironmentsRequest) {
        return new ListEnvironmentsPublisher(this, applyPaginatorUserAgent(listEnvironmentsRequest));
    }

    /**
     * <p>
     * View a list of configurations stored in the AppConfig configuration store by version.
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return A Java Future containing the result of the ListHostedConfigurationVersions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListHostedConfigurationVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHostedConfigurationVersionsResponse> listHostedConfigurationVersions(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHostedConfigurationVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHostedConfigurationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHostedConfigurationVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHostedConfigurationVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHostedConfigurationVersionsRequest, ListHostedConfigurationVersionsResponse>()
                            .withOperationName("ListHostedConfigurationVersions")
                            .withMarshaller(new ListHostedConfigurationVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listHostedConfigurationVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listHostedConfigurationVersionsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListHostedConfigurationVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    listHostedConfigurationVersionsRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * View a list of configurations stored in the AppConfig configuration store by version.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHostedConfigurationVersions(software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsPublisher publisher = client.listHostedConfigurationVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appconfig.paginators.ListHostedConfigurationVersionsPublisher publisher = client.listHostedConfigurationVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHostedConfigurationVersions(software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListHostedConfigurationVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListHostedConfigurationVersionsPublisher listHostedConfigurationVersionsPaginator(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) {
        return new ListHostedConfigurationVersionsPublisher(this, applyPaginatorUserAgent(listHostedConfigurationVersionsRequest));
    }

    /**
     * <p>
     * Retrieves the list of key-value tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a deployment.
     * </p>
     *
     * @param startDeploymentRequest
     * @return A Java Future containing the result of the StartDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.StartDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeploymentResponse> startDeployment(StartDeploymentRequest startDeploymentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                            .withOperationName("StartDeployment")
                            .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StartDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a deployment. This API action works only on deployments that have a status of <code>DEPLOYING</code>. This
     * action moves the deployment to a status of <code>ROLLED_BACK</code>.
     * </p>
     *
     * @param stopDeploymentRequest
     * @return A Java Future containing the result of the StopDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.StopDeployment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StopDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopDeploymentResponse> stopDeployment(StopDeploymentRequest stopDeploymentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopDeploymentRequest, StopDeploymentResponse>()
                            .withOperationName("StopDeployment")
                            .withMarshaller(new StopDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = stopDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<StopDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDeploymentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Metadata to assign to an AppConfig resource. Tags help organize and categorize your AppConfig resources. Each tag
     * consists of a key and an optional value, both of which you define. You can specify a maximum of 50 tags for a
     * resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tag key and value from an AppConfig resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateApplication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication")
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateApplicationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateApplicationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a configuration profile.
     * </p>
     *
     * @param updateConfigurationProfileRequest
     * @return A Java Future containing the result of the UpdateConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateConfigurationProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationProfileResponse> updateConfigurationProfile(
            UpdateConfigurationProfileRequest updateConfigurationProfileRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationProfileRequest, UpdateConfigurationProfileResponse>()
                            .withOperationName("UpdateConfigurationProfile")
                            .withMarshaller(new UpdateConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConfigurationProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateConfigurationProfileRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    updateConfigurationProfileRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a deployment strategy.
     * </p>
     *
     * @param updateDeploymentStrategyRequest
     * @return A Java Future containing the result of the UpdateDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateDeploymentStrategy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDeploymentStrategyResponse> updateDeploymentStrategy(
            UpdateDeploymentStrategyRequest updateDeploymentStrategyRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDeploymentStrategyRequest, UpdateDeploymentStrategyResponse>()
                            .withOperationName("UpdateDeploymentStrategy")
                            .withMarshaller(new UpdateDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDeploymentStrategyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDeploymentStrategyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeploymentStrategyRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment")
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateEnvironmentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateEnvironmentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uses the validators in a configuration profile to validate a configuration.
     * </p>
     *
     * @param validateConfigurationRequest
     * @return A Java Future containing the result of the ValidateConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ValidateConfiguration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ValidateConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateConfigurationResponse> validateConfiguration(
            ValidateConfigurationRequest validateConfigurationRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidateConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateConfigurationRequest, ValidateConfigurationResponse>()
                            .withOperationName("ValidateConfiguration")
                            .withMarshaller(new ValidateConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(validateConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = validateConfigurationRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ValidateConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validateConfigurationRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PayloadTooLargeException")
                                .exceptionBuilderSupplier(PayloadTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AppConfigRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
