/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that describes a deployment event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentEvent implements SdkPojo, Serializable, ToCopyableBuilder<DeploymentEvent.Builder, DeploymentEvent> {
    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventType").getter(getter(DeploymentEvent::eventTypeAsString)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventType").build()).build();

    private static final SdkField<String> TRIGGERED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TriggeredBy").getter(getter(DeploymentEvent::triggeredByAsString)).setter(setter(Builder::triggeredBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TriggeredBy").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DeploymentEvent::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> OCCURRED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("OccurredAt")
            .getter(getter(DeploymentEvent::occurredAt))
            .setter(setter(Builder::occurredAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OccurredAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_TYPE_FIELD,
            TRIGGERED_BY_FIELD, DESCRIPTION_FIELD, OCCURRED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventType;

    private final String triggeredBy;

    private final String description;

    private final Instant occurredAt;

    private DeploymentEvent(BuilderImpl builder) {
        this.eventType = builder.eventType;
        this.triggeredBy = builder.triggeredBy;
        this.description = builder.description;
        this.occurredAt = builder.occurredAt;
    }

    /**
     * <p>
     * The type of deployment event. Deployment event types include the start, stop, or completion of a deployment; a
     * percentage update; the start or stop of a bake period; and the start or completion of a rollback.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link DeploymentEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of deployment event. Deployment event types include the start, stop, or completion of a
     *         deployment; a percentage update; the start or stop of a bake period; and the start or completion of a
     *         rollback.
     * @see DeploymentEventType
     */
    public final DeploymentEventType eventType() {
        return DeploymentEventType.fromValue(eventType);
    }

    /**
     * <p>
     * The type of deployment event. Deployment event types include the start, stop, or completion of a deployment; a
     * percentage update; the start or stop of a bake period; and the start or completion of a rollback.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link DeploymentEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of deployment event. Deployment event types include the start, stop, or completion of a
     *         deployment; a percentage update; the start or stop of a bake period; and the start or completion of a
     *         rollback.
     * @see DeploymentEventType
     */
    public final String eventTypeAsString() {
        return eventType;
    }

    /**
     * <p>
     * The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
     * CloudWatch alarm, or an internal error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #triggeredBy} will
     * return {@link TriggeredBy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #triggeredByAsString}.
     * </p>
     * 
     * @return The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
     *         CloudWatch alarm, or an internal error.
     * @see TriggeredBy
     */
    public final TriggeredBy triggeredBy() {
        return TriggeredBy.fromValue(triggeredBy);
    }

    /**
     * <p>
     * The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
     * CloudWatch alarm, or an internal error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #triggeredBy} will
     * return {@link TriggeredBy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #triggeredByAsString}.
     * </p>
     * 
     * @return The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
     *         CloudWatch alarm, or an internal error.
     * @see TriggeredBy
     */
    public final String triggeredByAsString() {
        return triggeredBy;
    }

    /**
     * <p>
     * A description of the deployment event. Descriptions include, but are not limited to, the user account or the
     * Amazon CloudWatch alarm ARN that initiated a rollback, the percentage of hosts that received the deployment, or
     * in the case of an internal error, a recommendation to attempt a new deployment.
     * </p>
     * 
     * @return A description of the deployment event. Descriptions include, but are not limited to, the user account or
     *         the Amazon CloudWatch alarm ARN that initiated a rollback, the percentage of hosts that received the
     *         deployment, or in the case of an internal error, a recommendation to attempt a new deployment.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The date and time the event occurred.
     * </p>
     * 
     * @return The date and time the event occurred.
     */
    public final Instant occurredAt() {
        return occurredAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(triggeredByAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(occurredAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentEvent)) {
            return false;
        }
        DeploymentEvent other = (DeploymentEvent) obj;
        return Objects.equals(eventTypeAsString(), other.eventTypeAsString())
                && Objects.equals(triggeredByAsString(), other.triggeredByAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(occurredAt(), other.occurredAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentEvent").add("EventType", eventTypeAsString())
                .add("TriggeredBy", triggeredByAsString()).add("Description", description()).add("OccurredAt", occurredAt())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventType":
            return Optional.ofNullable(clazz.cast(eventTypeAsString()));
        case "TriggeredBy":
            return Optional.ofNullable(clazz.cast(triggeredByAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OccurredAt":
            return Optional.ofNullable(clazz.cast(occurredAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentEvent, T> g) {
        return obj -> g.apply((DeploymentEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentEvent> {
        /**
         * <p>
         * The type of deployment event. Deployment event types include the start, stop, or completion of a deployment;
         * a percentage update; the start or stop of a bake period; and the start or completion of a rollback.
         * </p>
         * 
         * @param eventType
         *        The type of deployment event. Deployment event types include the start, stop, or completion of a
         *        deployment; a percentage update; the start or stop of a bake period; and the start or completion of a
         *        rollback.
         * @see DeploymentEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentEventType
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * The type of deployment event. Deployment event types include the start, stop, or completion of a deployment;
         * a percentage update; the start or stop of a bake period; and the start or completion of a rollback.
         * </p>
         * 
         * @param eventType
         *        The type of deployment event. Deployment event types include the start, stop, or completion of a
         *        deployment; a percentage update; the start or stop of a bake period; and the start or completion of a
         *        rollback.
         * @see DeploymentEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentEventType
         */
        Builder eventType(DeploymentEventType eventType);

        /**
         * <p>
         * The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
         * CloudWatch alarm, or an internal error.
         * </p>
         * 
         * @param triggeredBy
         *        The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an
         *        Amazon CloudWatch alarm, or an internal error.
         * @see TriggeredBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TriggeredBy
         */
        Builder triggeredBy(String triggeredBy);

        /**
         * <p>
         * The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an Amazon
         * CloudWatch alarm, or an internal error.
         * </p>
         * 
         * @param triggeredBy
         *        The entity that triggered the deployment event. Events can be triggered by a user, AppConfig, an
         *        Amazon CloudWatch alarm, or an internal error.
         * @see TriggeredBy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TriggeredBy
         */
        Builder triggeredBy(TriggeredBy triggeredBy);

        /**
         * <p>
         * A description of the deployment event. Descriptions include, but are not limited to, the user account or the
         * Amazon CloudWatch alarm ARN that initiated a rollback, the percentage of hosts that received the deployment,
         * or in the case of an internal error, a recommendation to attempt a new deployment.
         * </p>
         * 
         * @param description
         *        A description of the deployment event. Descriptions include, but are not limited to, the user account
         *        or the Amazon CloudWatch alarm ARN that initiated a rollback, the percentage of hosts that received
         *        the deployment, or in the case of an internal error, a recommendation to attempt a new deployment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The date and time the event occurred.
         * </p>
         * 
         * @param occurredAt
         *        The date and time the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder occurredAt(Instant occurredAt);
    }

    static final class BuilderImpl implements Builder {
        private String eventType;

        private String triggeredBy;

        private String description;

        private Instant occurredAt;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentEvent model) {
            eventType(model.eventType);
            triggeredBy(model.triggeredBy);
            description(model.description);
            occurredAt(model.occurredAt);
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        @Override
        public final Builder eventType(DeploymentEventType eventType) {
            this.eventType(eventType == null ? null : eventType.toString());
            return this;
        }

        public final String getTriggeredBy() {
            return triggeredBy;
        }

        public final void setTriggeredBy(String triggeredBy) {
            this.triggeredBy = triggeredBy;
        }

        @Override
        public final Builder triggeredBy(String triggeredBy) {
            this.triggeredBy = triggeredBy;
            return this;
        }

        @Override
        public final Builder triggeredBy(TriggeredBy triggeredBy) {
            this.triggeredBy(triggeredBy == null ? null : triggeredBy.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getOccurredAt() {
            return occurredAt;
        }

        public final void setOccurredAt(Instant occurredAt) {
            this.occurredAt = occurredAt;
        }

        @Override
        public final Builder occurredAt(Instant occurredAt) {
            this.occurredAt = occurredAt;
            return this;
        }

        @Override
        public DeploymentEvent build() {
            return new DeploymentEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
