/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appconfig.internal.AppConfigServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.appconfig.model.AppConfigException;
import software.amazon.awssdk.services.appconfig.model.BadRequestException;
import software.amazon.awssdk.services.appconfig.model.ConflictException;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.GetApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.GetApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.GetExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.GetExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.GetExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.GetExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.InternalServerException;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListExtensionAssociationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListExtensionAssociationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListExtensionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListExtensionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appconfig.model.PayloadTooLargeException;
import software.amazon.awssdk.services.appconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.TagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.TagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UntagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.UntagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationResponse;
import software.amazon.awssdk.services.appconfig.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListConfigurationProfilesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListExtensionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListExtensionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListHostedConfigurationVersionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StopDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ValidateConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppConfigAsyncClient}.
 *
 * @see AppConfigAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppConfigAsyncClient implements AppConfigAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppConfigAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AppConfigServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAppConfigAsyncClient(AppConfigServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an application. In AppConfig, an application is simply an organizational construct like a folder. This
     * organizational construct has a relationship with some unit of executable code. For example, you could create an
     * application called MyMobileApp to organize and manage configuration data for a mobile application installed by
     * your users.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication")
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createApplicationRequest));
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a configuration profile, which is information that enables AppConfig to access the configuration source.
     * Valid configuration sources include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Configuration data in YAML, JSON, and other formats stored in the AppConfig hosted configuration store
     * </p>
     * </li>
     * <li>
     * <p>
     * Configuration data stored as objects in an Amazon Simple Storage Service (Amazon S3) bucket
     * </p>
     * </li>
     * <li>
     * <p>
     * Pipelines stored in CodePipeline
     * </p>
     * </li>
     * <li>
     * <p>
     * Secrets stored in Secrets Manager
     * </p>
     * </li>
     * <li>
     * <p>
     * Standard and secure string parameters stored in Amazon Web Services Systems Manager Parameter Store
     * </p>
     * </li>
     * <li>
     * <p>
     * Configuration data in SSM documents stored in the Systems Manager document store
     * </p>
     * </li>
     * </ul>
     * <p>
     * A configuration profile includes the following information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The URI location of the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Identity and Access Management (IAM) role that provides access to the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * A validator for the configuration data. Available validators include either a JSON Schema or an Amazon Web
     * Services Lambda function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-configuration-and-profile.html"
     * >Create a Configuration and a Configuration Profile</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createConfigurationProfileRequest
     * @return A Java Future containing the result of the CreateConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationProfileResponse> createConfigurationProfile(
            CreateConfigurationProfileRequest createConfigurationProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationProfileRequest, CreateConfigurationProfileResponse>()
                            .withOperationName("CreateConfigurationProfile")
                            .withMarshaller(new CreateConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationProfileRequest));
            CompletableFuture<CreateConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a deployment strategy that defines important criteria for rolling out your configuration to the
     * designated targets. A deployment strategy includes the overall duration required, a percentage of targets to
     * receive the deployment during each interval, an algorithm that defines how percentage grows, and bake time.
     * </p>
     *
     * @param createDeploymentStrategyRequest
     * @return A Java Future containing the result of the CreateDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentStrategyResponse> createDeploymentStrategy(
            CreateDeploymentStrategyRequest createDeploymentStrategyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeploymentStrategyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentStrategyRequest, CreateDeploymentStrategyResponse>()
                            .withOperationName("CreateDeploymentStrategy")
                            .withMarshaller(new CreateDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDeploymentStrategyRequest));
            CompletableFuture<CreateDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an environment. For each application, you define one or more environments. An environment is a deployment
     * group of AppConfig targets, such as applications in a <code>Beta</code> or <code>Production</code> environment.
     * You can also define environments for application subcomponents such as the <code>Web</code>, <code>Mobile</code>
     * and <code>Back-end</code> components for your application. You can configure Amazon CloudWatch alarms for each
     * environment. The system monitors alarms during a configuration deployment. If an alarm is triggered, the system
     * rolls back the configuration.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                            .withOperationName("CreateEnvironment")
                            .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEnvironmentRequest));
            CompletableFuture<CreateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AppConfig extension. An extension augments your ability to inject logic or behavior at different
     * points during the AppConfig workflow of creating or deploying a configuration.
     * </p>
     * <p>
     * You can create your own extensions or use the Amazon Web Services authored extensions provided by AppConfig. For
     * an AppConfig extension that uses Lambda, you must create a Lambda function to perform any computation and
     * processing defined in the extension. If you plan to create custom versions of the Amazon Web Services authored
     * notification extensions, you only need to specify an Amazon Resource Name (ARN) in the <code>Uri</code> field for
     * the new extension version.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a custom EventBridge notification extension, enter the ARN of the EventBridge default events in the
     * <code>Uri</code> field.
     * </p>
     * </li>
     * <li>
     * <p>
     * For a custom Amazon SNS notification extension, enter the ARN of an Amazon SNS topic in the <code>Uri</code>
     * field.
     * </p>
     * </li>
     * <li>
     * <p>
     * For a custom Amazon SQS notification extension, enter the ARN of an Amazon SQS message queue in the
     * <code>Uri</code> field.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createExtensionRequest
     * @return A Java Future containing the result of the CreateExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExtensionResponse> createExtension(CreateExtensionRequest createExtensionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExtensionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExtensionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExtensionRequest, CreateExtensionResponse>()
                            .withOperationName("CreateExtension")
                            .withMarshaller(new CreateExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createExtensionRequest));
            CompletableFuture<CreateExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you create an extension or configure an Amazon Web Services authored extension, you associate the extension
     * with an AppConfig application, environment, or configuration profile. For example, you can choose to run the
     * <code>AppConfig deployment events to Amazon SNS</code> Amazon Web Services authored extension and receive
     * notifications on an Amazon SNS topic anytime a configuration deployment is started for a specific application.
     * Defining which extension to associate with an AppConfig resource is called an <i>extension association</i>. An
     * extension association is a specified relationship between an extension and an AppConfig resource, such as an
     * application or a configuration profile. For more information about extensions and associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createExtensionAssociationRequest
     * @return A Java Future containing the result of the CreateExtensionAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateExtensionAssociationResponse> createExtensionAssociation(
            CreateExtensionAssociationRequest createExtensionAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExtensionAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExtensionAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateExtensionAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateExtensionAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateExtensionAssociationRequest, CreateExtensionAssociationResponse>()
                            .withOperationName("CreateExtensionAssociation")
                            .withMarshaller(new CreateExtensionAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createExtensionAssociationRequest));
            CompletableFuture<CreateExtensionAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new configuration in the AppConfig hosted configuration store.
     * </p>
     *
     * @param createHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the CreateHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ServiceQuotaExceededException The number of one more AppConfig resources exceeds the maximum allowed.
     *         Verify that your environment doesn't exceed the following service quotas:</p>
     *         <p>
     *         Applications: 100 max
     *         </p>
     *         <p>
     *         Deployment strategies: 20 max
     *         </p>
     *         <p>
     *         Configuration profiles: 100 max per application
     *         </p>
     *         <p>
     *         Environments: 20 max per application
     *         </p>
     *         <p>
     *         To resolve this issue, you can delete one or more resources and try again. Or, you can request a quota
     *         increase. For more information about quotas and to request an increase, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/appconfig.html#limits_appconfig">Service quotas for
     *         AppConfig</a> in the Amazon Web Services General Reference.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>PayloadTooLargeException The configuration size is too large.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.CreateHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHostedConfigurationVersionResponse> createHostedConfigurationVersion(
            CreateHostedConfigurationVersionRequest createHostedConfigurationVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHostedConfigurationVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<CreateHostedConfigurationVersionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHostedConfigurationVersionRequest, CreateHostedConfigurationVersionResponse>()
                            .withOperationName("CreateHostedConfigurationVersion")
                            .withMarshaller(new CreateHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHostedConfigurationVersionRequest));
            CompletableFuture<CreateHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an application. Deleting an application does not delete a configuration from a host.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication")
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationRequest));
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a configuration profile. Deleting a configuration profile does not delete a configuration from a host.
     * </p>
     *
     * @param deleteConfigurationProfileRequest
     * @return A Java Future containing the result of the DeleteConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigurationProfileResponse> deleteConfigurationProfile(
            DeleteConfigurationProfileRequest deleteConfigurationProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigurationProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationProfileRequest, DeleteConfigurationProfileResponse>()
                            .withOperationName("DeleteConfigurationProfile")
                            .withMarshaller(new DeleteConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigurationProfileRequest));
            CompletableFuture<DeleteConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a deployment strategy. Deleting a deployment strategy does not delete a configuration from a host.
     * </p>
     *
     * @param deleteDeploymentStrategyRequest
     * @return A Java Future containing the result of the DeleteDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDeploymentStrategyResponse> deleteDeploymentStrategy(
            DeleteDeploymentStrategyRequest deleteDeploymentStrategyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeploymentStrategyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDeploymentStrategyRequest, DeleteDeploymentStrategyResponse>()
                            .withOperationName("DeleteDeploymentStrategy")
                            .withMarshaller(new DeleteDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDeploymentStrategyRequest));
            CompletableFuture<DeleteDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an environment. Deleting an environment does not delete a configuration from a host.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment")
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEnvironmentRequest));
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AppConfig extension. You must delete all associations to an extension before you delete the extension.
     * </p>
     *
     * @param deleteExtensionRequest
     * @return A Java Future containing the result of the DeleteExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExtensionResponse> deleteExtension(DeleteExtensionRequest deleteExtensionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExtensionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExtensionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExtensionRequest, DeleteExtensionResponse>()
                            .withOperationName("DeleteExtension")
                            .withMarshaller(new DeleteExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteExtensionRequest));
            CompletableFuture<DeleteExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an extension association. This action doesn't delete extensions defined in the association.
     * </p>
     *
     * @param deleteExtensionAssociationRequest
     * @return A Java Future containing the result of the DeleteExtensionAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteExtensionAssociationResponse> deleteExtensionAssociation(
            DeleteExtensionAssociationRequest deleteExtensionAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExtensionAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExtensionAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteExtensionAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteExtensionAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteExtensionAssociationRequest, DeleteExtensionAssociationResponse>()
                            .withOperationName("DeleteExtensionAssociation")
                            .withMarshaller(new DeleteExtensionAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteExtensionAssociationRequest));
            CompletableFuture<DeleteExtensionAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a version of a configuration from the AppConfig hosted configuration store.
     * </p>
     *
     * @param deleteHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the DeleteHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.DeleteHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHostedConfigurationVersionResponse> deleteHostedConfigurationVersion(
            DeleteHostedConfigurationVersionRequest deleteHostedConfigurationVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHostedConfigurationVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHostedConfigurationVersionResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHostedConfigurationVersionRequest, DeleteHostedConfigurationVersionResponse>()
                            .withOperationName("DeleteHostedConfigurationVersion")
                            .withMarshaller(new DeleteHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHostedConfigurationVersionRequest));
            CompletableFuture<DeleteHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApplicationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication")
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getApplicationRequest));
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * (Deprecated) Retrieves the latest deployed configuration.
     * </p>
     * <important>
     * <p>
     * Note the following important information.
     * </p>
     * <ul>
     * <li>
     * <p>
     * This API action is deprecated. Calls to receive configuration data should use the <a href=
     * "https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_appconfigdata_StartConfigurationSession.html"
     * >StartConfigurationSession</a> and <a href=
     * "https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_appconfigdata_GetLatestConfiguration.html"
     * >GetLatestConfiguration</a> APIs instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GetConfiguration</code> is a priced call. For more information, see <a
     * href="https://aws.amazon.com/systems-manager/pricing/">Pricing</a>.
     * </p>
     * </li>
     * </ul>
     * </important>
     *
     * @param getConfigurationRequest
     * @return A Java Future containing the result of the GetConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationResponse> getConfiguration(GetConfigurationRequest getConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationRequest, GetConfigurationResponse>()
                            .withOperationName("GetConfiguration")
                            .withMarshaller(new GetConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationRequest));
            CompletableFuture<GetConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a configuration profile.
     * </p>
     *
     * @param getConfigurationProfileRequest
     * @return A Java Future containing the result of the GetConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigurationProfileResponse> getConfigurationProfile(
            GetConfigurationProfileRequest getConfigurationProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigurationProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationProfileRequest, GetConfigurationProfileResponse>()
                            .withOperationName("GetConfigurationProfile")
                            .withMarshaller(new GetConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getConfigurationProfileRequest));
            CompletableFuture<GetConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a configuration deployment.
     * </p>
     *
     * @param getDeploymentRequest
     * @return A Java Future containing the result of the GetDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentResponse> getDeployment(GetDeploymentRequest getDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeploymentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                            .withOperationName("GetDeployment")
                            .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeploymentRequest));
            CompletableFuture<GetDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a deployment strategy. A deployment strategy defines important criteria for rolling
     * out your configuration to the designated targets. A deployment strategy includes the overall duration required, a
     * percentage of targets to receive the deployment during each interval, an algorithm that defines how percentage
     * grows, and bake time.
     * </p>
     *
     * @param getDeploymentStrategyRequest
     * @return A Java Future containing the result of the GetDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentStrategyResponse> getDeploymentStrategy(
            GetDeploymentStrategyRequest getDeploymentStrategyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeploymentStrategyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentStrategyRequest, GetDeploymentStrategyResponse>()
                            .withOperationName("GetDeploymentStrategy")
                            .withMarshaller(new GetDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDeploymentStrategyRequest));
            CompletableFuture<GetDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an environment. An environment is a deployment group of AppConfig applications, such
     * as applications in a <code>Production</code> environment or in an <code>EU_Region</code> environment. Each
     * configuration deployment targets an environment. You can enable one or more Amazon CloudWatch alarms for an
     * environment. If an alarm is triggered during a deployment, AppConfig roles back the configuration.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                            .withOperationName("GetEnvironment")
                            .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnvironmentRequest));
            CompletableFuture<GetEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an AppConfig extension.
     * </p>
     *
     * @param getExtensionRequest
     * @return A Java Future containing the result of the GetExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetExtensionResponse> getExtension(GetExtensionRequest getExtensionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExtensionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExtensionRequest, GetExtensionResponse>()
                            .withOperationName("GetExtension").withMarshaller(new GetExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getExtensionRequest));
            CompletableFuture<GetExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an AppConfig extension association. For more information about extensions and
     * associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param getExtensionAssociationRequest
     * @return A Java Future containing the result of the GetExtensionAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExtensionAssociationResponse> getExtensionAssociation(
            GetExtensionAssociationRequest getExtensionAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExtensionAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtensionAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExtensionAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExtensionAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExtensionAssociationRequest, GetExtensionAssociationResponse>()
                            .withOperationName("GetExtensionAssociation")
                            .withMarshaller(new GetExtensionAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getExtensionAssociationRequest));
            CompletableFuture<GetExtensionAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a specific configuration version.
     * </p>
     *
     * @param getHostedConfigurationVersionRequest
     * @return A Java Future containing the result of the GetHostedConfigurationVersion operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.GetHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetHostedConfigurationVersionResponse> getHostedConfigurationVersion(
            GetHostedConfigurationVersionRequest getHostedConfigurationVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHostedConfigurationVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHostedConfigurationVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetHostedConfigurationVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetHostedConfigurationVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetHostedConfigurationVersionRequest, GetHostedConfigurationVersionResponse>()
                            .withOperationName("GetHostedConfigurationVersion")
                            .withMarshaller(new GetHostedConfigurationVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getHostedConfigurationVersionRequest));
            CompletableFuture<GetHostedConfigurationVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all applications in your Amazon Web Services account.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications")
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApplicationsRequest));
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return A Java Future containing the result of the ListConfigurationProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListConfigurationProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationProfilesResponse> listConfigurationProfiles(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationProfilesRequest, ListConfigurationProfilesResponse>()
                            .withOperationName("ListConfigurationProfiles")
                            .withMarshaller(new ListConfigurationProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationProfilesRequest));
            CompletableFuture<ListConfigurationProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists deployment strategies.
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return A Java Future containing the result of the ListDeploymentStrategies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeploymentStrategies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentStrategiesResponse> listDeploymentStrategies(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeploymentStrategiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentStrategiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeploymentStrategies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentStrategiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeploymentStrategiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentStrategiesRequest, ListDeploymentStrategiesResponse>()
                            .withOperationName("ListDeploymentStrategies")
                            .withMarshaller(new ListDeploymentStrategiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeploymentStrategiesRequest));
            CompletableFuture<ListDeploymentStrategiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the deployments for an environment in descending deployment number order.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A Java Future containing the result of the ListDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentsResponse> listDeployments(ListDeploymentsRequest listDeploymentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeploymentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                            .withOperationName("ListDeployments")
                            .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDeploymentsRequest));
            CompletableFuture<ListDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the environments for an application.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments")
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnvironmentsRequest));
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all AppConfig extension associations in the account. For more information about extensions and
     * associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param listExtensionAssociationsRequest
     * @return A Java Future containing the result of the ListExtensionAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListExtensionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListExtensionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListExtensionAssociationsResponse> listExtensionAssociations(
            ListExtensionAssociationsRequest listExtensionAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExtensionAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensionAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExtensionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExtensionAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExtensionAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExtensionAssociationsRequest, ListExtensionAssociationsResponse>()
                            .withOperationName("ListExtensionAssociations")
                            .withMarshaller(new ListExtensionAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExtensionAssociationsRequest));
            CompletableFuture<ListExtensionAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all custom and Amazon Web Services authored AppConfig extensions in the account. For more information about
     * extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param listExtensionsRequest
     * @return A Java Future containing the result of the ListExtensions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListExtensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExtensionsResponse> listExtensions(ListExtensionsRequest listExtensionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExtensionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExtensionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExtensionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExtensionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExtensionsRequest, ListExtensionsResponse>()
                            .withOperationName("ListExtensions")
                            .withMarshaller(new ListExtensionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listExtensionsRequest));
            CompletableFuture<ListExtensionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists configurations stored in the AppConfig hosted configuration store by version.
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return A Java Future containing the result of the ListHostedConfigurationVersions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListHostedConfigurationVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListHostedConfigurationVersionsResponse> listHostedConfigurationVersions(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHostedConfigurationVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listHostedConfigurationVersionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHostedConfigurationVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListHostedConfigurationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListHostedConfigurationVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListHostedConfigurationVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListHostedConfigurationVersionsRequest, ListHostedConfigurationVersionsResponse>()
                            .withOperationName("ListHostedConfigurationVersions")
                            .withMarshaller(new ListHostedConfigurationVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listHostedConfigurationVersionsRequest));
            CompletableFuture<ListHostedConfigurationVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of key-value tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a deployment.
     * </p>
     *
     * @param startDeploymentRequest
     * @return A Java Future containing the result of the StartDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeploymentResponse> startDeployment(StartDeploymentRequest startDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                            .withOperationName("StartDeployment")
                            .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startDeploymentRequest));
            CompletableFuture<StartDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops a deployment. This API action works only on deployments that have a status of <code>DEPLOYING</code>. This
     * action moves the deployment to a status of <code>ROLLED_BACK</code>.
     * </p>
     *
     * @param stopDeploymentRequest
     * @return A Java Future containing the result of the StopDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.StopDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StopDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopDeploymentResponse> stopDeployment(StopDeploymentRequest stopDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopDeploymentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopDeploymentRequest, StopDeploymentResponse>()
                            .withOperationName("StopDeployment")
                            .withMarshaller(new StopDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopDeploymentRequest));
            CompletableFuture<StopDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns metadata to an AppConfig resource. Tags help organize and categorize your AppConfig resources. Each tag
     * consists of a key and an optional value, both of which you define. You can specify a maximum of 50 tags for a
     * resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tag key and value from an AppConfig resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication")
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApplicationRequest));
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a configuration profile.
     * </p>
     *
     * @param updateConfigurationProfileRequest
     * @return A Java Future containing the result of the UpdateConfigurationProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationProfileResponse> updateConfigurationProfile(
            UpdateConfigurationProfileRequest updateConfigurationProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfigurationProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationProfileRequest, UpdateConfigurationProfileResponse>()
                            .withOperationName("UpdateConfigurationProfile")
                            .withMarshaller(new UpdateConfigurationProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationProfileRequest));
            CompletableFuture<UpdateConfigurationProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a deployment strategy.
     * </p>
     *
     * @param updateDeploymentStrategyRequest
     * @return A Java Future containing the result of the UpdateDeploymentStrategy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDeploymentStrategyResponse> updateDeploymentStrategy(
            UpdateDeploymentStrategyRequest updateDeploymentStrategyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDeploymentStrategyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeploymentStrategy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDeploymentStrategyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDeploymentStrategyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDeploymentStrategyRequest, UpdateDeploymentStrategyResponse>()
                            .withOperationName("UpdateDeploymentStrategy")
                            .withMarshaller(new UpdateDeploymentStrategyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDeploymentStrategyRequest));
            CompletableFuture<UpdateDeploymentStrategyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment")
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnvironmentRequest));
            CompletableFuture<UpdateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an AppConfig extension. For more information about extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param updateExtensionRequest
     * @return A Java Future containing the result of the UpdateExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateExtensionResponse> updateExtension(UpdateExtensionRequest updateExtensionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExtensionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateExtensionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateExtensionRequest, UpdateExtensionResponse>()
                            .withOperationName("UpdateExtension")
                            .withMarshaller(new UpdateExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateExtensionRequest));
            CompletableFuture<UpdateExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an association. For more information about extensions and associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param updateExtensionAssociationRequest
     * @return A Java Future containing the result of the UpdateExtensionAssociation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.UpdateExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateExtensionAssociationResponse> updateExtensionAssociation(
            UpdateExtensionAssociationRequest updateExtensionAssociationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExtensionAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExtensionAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateExtensionAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateExtensionAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateExtensionAssociationRequest, UpdateExtensionAssociationResponse>()
                            .withOperationName("UpdateExtensionAssociation")
                            .withMarshaller(new UpdateExtensionAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateExtensionAssociationRequest));
            CompletableFuture<UpdateExtensionAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uses the validators in a configuration profile to validate a configuration.
     * </p>
     *
     * @param validateConfigurationRequest
     * @return A Java Future containing the result of the ValidateConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>ResourceNotFoundException The requested resource could not be found.</li>
     *         <li>InternalServerException There was an internal failure in the AppConfig service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppConfigException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppConfigAsyncClient.ValidateConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ValidateConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidateConfigurationResponse> validateConfiguration(
            ValidateConfigurationRequest validateConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(validateConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validateConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidateConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidateConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidateConfigurationRequest, ValidateConfigurationResponse>()
                            .withOperationName("ValidateConfiguration")
                            .withMarshaller(new ValidateConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(validateConfigurationRequest));
            CompletableFuture<ValidateConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppConfigServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PayloadTooLargeException")
                                .exceptionBuilderSupplier(PayloadTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        AppConfigServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = AppConfigServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
