/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.appconfig.AppConfigClient;
import software.amazon.awssdk.services.appconfig.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.appconfig.model.AppConfigRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAppConfigWaiter implements AppConfigWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final AppConfigClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetEnvironmentResponse> environmentReadyForDeploymentWaiter;

    private final Waiter<GetDeploymentResponse> deploymentCompleteWaiter;

    private DefaultAppConfigWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AppConfigClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.environmentReadyForDeploymentWaiter = Waiter.builder(GetEnvironmentResponse.class)
                .acceptors(environmentReadyForDeploymentWaiterAcceptors())
                .overrideConfiguration(environmentReadyForDeploymentWaiterConfig(builder.overrideConfiguration)).build();
        this.deploymentCompleteWaiter = Waiter.builder(GetDeploymentResponse.class)
                .acceptors(deploymentCompleteWaiterAcceptors())
                .overrideConfiguration(deploymentCompleteWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetDeploymentResponse> waitUntilDeploymentComplete(GetDeploymentRequest getDeploymentRequest) {
        return deploymentCompleteWaiter.run(() -> client.getDeployment(applyWaitersUserAgent(getDeploymentRequest)));
    }

    @Override
    public WaiterResponse<GetDeploymentResponse> waitUntilDeploymentComplete(GetDeploymentRequest getDeploymentRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return deploymentCompleteWaiter.run(() -> client.getDeployment(applyWaitersUserAgent(getDeploymentRequest)),
                deploymentCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentReadyForDeployment(
            GetEnvironmentRequest getEnvironmentRequest) {
        return environmentReadyForDeploymentWaiter.run(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)));
    }

    @Override
    public WaiterResponse<GetEnvironmentResponse> waitUntilEnvironmentReadyForDeployment(
            GetEnvironmentRequest getEnvironmentRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentReadyForDeploymentWaiter.run(() -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)),
                environmentReadyForDeploymentWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetEnvironmentResponse>> environmentReadyForDeploymentWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ReadyForDeployment");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "RolledBack");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=RolledBack) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "Reverted");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=Reverted) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetDeploymentResponse>> deploymentCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super GetDeploymentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "ROLLED_BACK");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=ROLLED_BACK) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "REVERTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=REVERTED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration environmentReadyForDeploymentWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration deploymentCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AppConfigWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AppConfigRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AppConfigWaiter.Builder {
        private AppConfigClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public AppConfigWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AppConfigWaiter.Builder client(AppConfigClient client) {
            this.client = client;
            return this;
        }

        public AppConfigWaiter build() {
            return new DefaultAppConfigWaiter(this);
        }
    }
}
