/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appfabric.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains a summary of an app authorization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppAuthorizationSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<AppAuthorizationSummary.Builder, AppAuthorizationSummary> {
    private static final SdkField<String> APP_AUTHORIZATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appAuthorizationArn").getter(getter(AppAuthorizationSummary::appAuthorizationArn))
            .setter(setter(Builder::appAuthorizationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appAuthorizationArn").build())
            .build();

    private static final SdkField<String> APP_BUNDLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appBundleArn").getter(getter(AppAuthorizationSummary::appBundleArn))
            .setter(setter(Builder::appBundleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appBundleArn").build()).build();

    private static final SdkField<String> APP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("app")
            .getter(getter(AppAuthorizationSummary::app)).setter(setter(Builder::app))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("app").build()).build();

    private static final SdkField<Tenant> TENANT_FIELD = SdkField.<Tenant> builder(MarshallingType.SDK_POJO).memberName("tenant")
            .getter(getter(AppAuthorizationSummary::tenant)).setter(setter(Builder::tenant)).constructor(Tenant::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tenant").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AppAuthorizationSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(AppAuthorizationSummary::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_AUTHORIZATION_ARN_FIELD,
            APP_BUNDLE_ARN_FIELD, APP_FIELD, TENANT_FIELD, STATUS_FIELD, UPDATED_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String appAuthorizationArn;

    private final String appBundleArn;

    private final String app;

    private final Tenant tenant;

    private final String status;

    private final Instant updatedAt;

    private AppAuthorizationSummary(BuilderImpl builder) {
        this.appAuthorizationArn = builder.appAuthorizationArn;
        this.appBundleArn = builder.appBundleArn;
        this.app = builder.app;
        this.tenant = builder.tenant;
        this.status = builder.status;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the app authorization.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the app authorization.
     */
    public final String appAuthorizationArn() {
        return appAuthorizationArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the app bundle for the app authorization.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the app bundle for the app authorization.
     */
    public final String appBundleArn() {
        return appBundleArn;
    }

    /**
     * <p>
     * The name of the application.
     * </p>
     * 
     * @return The name of the application.
     */
    public final String app() {
        return app;
    }

    /**
     * <p>
     * Contains information about an application tenant, such as the application display name and identifier.
     * </p>
     * 
     * @return Contains information about an application tenant, such as the application display name and identifier.
     */
    public final Tenant tenant() {
        return tenant;
    }

    /**
     * <p>
     * The state of the app authorization.
     * </p>
     * <p>
     * The following states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but not
     * yet connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
     * connect to the application. If the app authorization is in this state, you should verify the configured
     * credentials and try to connect the app authorization again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization is
     * in this state, you should try to reconnect the app authorization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AppAuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the app authorization.</p>
     *         <p>
     *         The following states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created
     *         but not yet connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
     *         trying to connect to the application. If the app authorization is in this state, you should verify the
     *         configured credentials and try to connect the app authorization again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
     *         authorization is in this state, you should try to reconnect the app authorization.
     *         </p>
     *         </li>
     * @see AppAuthorizationStatus
     */
    public final AppAuthorizationStatus status() {
        return AppAuthorizationStatus.fromValue(status);
    }

    /**
     * <p>
     * The state of the app authorization.
     * </p>
     * <p>
     * The following states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but not
     * yet connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
     * connect to the application. If the app authorization is in this state, you should verify the configured
     * credentials and try to connect the app authorization again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization is
     * in this state, you should try to reconnect the app authorization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AppAuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the app authorization.</p>
     *         <p>
     *         The following states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created
     *         but not yet connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
     *         trying to connect to the application. If the app authorization is in this state, you should verify the
     *         configured credentials and try to connect the app authorization again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
     *         authorization is in this state, you should try to reconnect the app authorization.
     *         </p>
     *         </li>
     * @see AppAuthorizationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Timestamp for when the app authorization was last updated.
     * </p>
     * 
     * @return Timestamp for when the app authorization was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appAuthorizationArn());
        hashCode = 31 * hashCode + Objects.hashCode(appBundleArn());
        hashCode = 31 * hashCode + Objects.hashCode(app());
        hashCode = 31 * hashCode + Objects.hashCode(tenant());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppAuthorizationSummary)) {
            return false;
        }
        AppAuthorizationSummary other = (AppAuthorizationSummary) obj;
        return Objects.equals(appAuthorizationArn(), other.appAuthorizationArn())
                && Objects.equals(appBundleArn(), other.appBundleArn()) && Objects.equals(app(), other.app())
                && Objects.equals(tenant(), other.tenant()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppAuthorizationSummary").add("AppAuthorizationArn", appAuthorizationArn())
                .add("AppBundleArn", appBundleArn()).add("App", app()).add("Tenant", tenant()).add("Status", statusAsString())
                .add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appAuthorizationArn":
            return Optional.ofNullable(clazz.cast(appAuthorizationArn()));
        case "appBundleArn":
            return Optional.ofNullable(clazz.cast(appBundleArn()));
        case "app":
            return Optional.ofNullable(clazz.cast(app()));
        case "tenant":
            return Optional.ofNullable(clazz.cast(tenant()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("appAuthorizationArn", APP_AUTHORIZATION_ARN_FIELD);
        map.put("appBundleArn", APP_BUNDLE_ARN_FIELD);
        map.put("app", APP_FIELD);
        map.put("tenant", TENANT_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AppAuthorizationSummary, T> g) {
        return obj -> g.apply((AppAuthorizationSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppAuthorizationSummary> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the app authorization.
         * </p>
         * 
         * @param appAuthorizationArn
         *        The Amazon Resource Name (ARN) of the app authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appAuthorizationArn(String appAuthorizationArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the app bundle for the app authorization.
         * </p>
         * 
         * @param appBundleArn
         *        The Amazon Resource Name (ARN) of the app bundle for the app authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBundleArn(String appBundleArn);

        /**
         * <p>
         * The name of the application.
         * </p>
         * 
         * @param app
         *        The name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder app(String app);

        /**
         * <p>
         * Contains information about an application tenant, such as the application display name and identifier.
         * </p>
         * 
         * @param tenant
         *        Contains information about an application tenant, such as the application display name and identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tenant(Tenant tenant);

        /**
         * <p>
         * Contains information about an application tenant, such as the application display name and identifier.
         * </p>
         * This is a convenience method that creates an instance of the {@link Tenant.Builder} avoiding the need to
         * create one manually via {@link Tenant#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Tenant.Builder#build()} is called immediately and its result is
         * passed to {@link #tenant(Tenant)}.
         * 
         * @param tenant
         *        a consumer that will call methods on {@link Tenant.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tenant(Tenant)
         */
        default Builder tenant(Consumer<Tenant.Builder> tenant) {
            return tenant(Tenant.builder().applyMutation(tenant).build());
        }

        /**
         * <p>
         * The state of the app authorization.
         * </p>
         * <p>
         * The following states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but
         * not yet connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
         * connect to the application. If the app authorization is in this state, you should verify the configured
         * credentials and try to connect the app authorization again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization
         * is in this state, you should try to reconnect the app authorization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The state of the app authorization.</p>
         *        <p>
         *        The following states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is
         *        created but not yet connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Connected</code>: The app authorization is connected to the application, and is ready to be
         *        used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
         *        trying to connect to the application. If the app authorization is in this state, you should verify the
         *        configured credentials and try to connect the app authorization again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
         *        authorization is in this state, you should try to reconnect the app authorization.
         *        </p>
         *        </li>
         * @see AppAuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppAuthorizationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The state of the app authorization.
         * </p>
         * <p>
         * The following states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but
         * not yet connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
         * connect to the application. If the app authorization is in this state, you should verify the configured
         * credentials and try to connect the app authorization again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization
         * is in this state, you should try to reconnect the app authorization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The state of the app authorization.</p>
         *        <p>
         *        The following states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is
         *        created but not yet connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Connected</code>: The app authorization is connected to the application, and is ready to be
         *        used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
         *        trying to connect to the application. If the app authorization is in this state, you should verify the
         *        configured credentials and try to connect the app authorization again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
         *        authorization is in this state, you should try to reconnect the app authorization.
         *        </p>
         *        </li>
         * @see AppAuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppAuthorizationStatus
         */
        Builder status(AppAuthorizationStatus status);

        /**
         * <p>
         * Timestamp for when the app authorization was last updated.
         * </p>
         * 
         * @param updatedAt
         *        Timestamp for when the app authorization was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String appAuthorizationArn;

        private String appBundleArn;

        private String app;

        private Tenant tenant;

        private String status;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(AppAuthorizationSummary model) {
            appAuthorizationArn(model.appAuthorizationArn);
            appBundleArn(model.appBundleArn);
            app(model.app);
            tenant(model.tenant);
            status(model.status);
            updatedAt(model.updatedAt);
        }

        public final String getAppAuthorizationArn() {
            return appAuthorizationArn;
        }

        public final void setAppAuthorizationArn(String appAuthorizationArn) {
            this.appAuthorizationArn = appAuthorizationArn;
        }

        @Override
        public final Builder appAuthorizationArn(String appAuthorizationArn) {
            this.appAuthorizationArn = appAuthorizationArn;
            return this;
        }

        public final String getAppBundleArn() {
            return appBundleArn;
        }

        public final void setAppBundleArn(String appBundleArn) {
            this.appBundleArn = appBundleArn;
        }

        @Override
        public final Builder appBundleArn(String appBundleArn) {
            this.appBundleArn = appBundleArn;
            return this;
        }

        public final String getApp() {
            return app;
        }

        public final void setApp(String app) {
            this.app = app;
        }

        @Override
        public final Builder app(String app) {
            this.app = app;
            return this;
        }

        public final Tenant.Builder getTenant() {
            return tenant != null ? tenant.toBuilder() : null;
        }

        public final void setTenant(Tenant.BuilderImpl tenant) {
            this.tenant = tenant != null ? tenant.build() : null;
        }

        @Override
        public final Builder tenant(Tenant tenant) {
            this.tenant = tenant;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AppAuthorizationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public AppAuthorizationSummary build() {
            return new AppAuthorizationSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
