/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appflow;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appflow.model.AccessDeniedException;
import software.amazon.awssdk.services.appflow.model.AppflowException;
import software.amazon.awssdk.services.appflow.model.AppflowRequest;
import software.amazon.awssdk.services.appflow.model.CancelFlowExecutionsRequest;
import software.amazon.awssdk.services.appflow.model.CancelFlowExecutionsResponse;
import software.amazon.awssdk.services.appflow.model.ConflictException;
import software.amazon.awssdk.services.appflow.model.ConnectorAuthenticationException;
import software.amazon.awssdk.services.appflow.model.ConnectorServerException;
import software.amazon.awssdk.services.appflow.model.CreateConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.CreateConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.CreateFlowRequest;
import software.amazon.awssdk.services.appflow.model.CreateFlowResponse;
import software.amazon.awssdk.services.appflow.model.DeleteConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.DeleteConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.DeleteFlowRequest;
import software.amazon.awssdk.services.appflow.model.DeleteFlowResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorEntityRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorEntityResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorsRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorsResponse;
import software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsRequest;
import software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsResponse;
import software.amazon.awssdk.services.appflow.model.DescribeFlowRequest;
import software.amazon.awssdk.services.appflow.model.DescribeFlowResponse;
import software.amazon.awssdk.services.appflow.model.InternalServerException;
import software.amazon.awssdk.services.appflow.model.ListConnectorEntitiesRequest;
import software.amazon.awssdk.services.appflow.model.ListConnectorEntitiesResponse;
import software.amazon.awssdk.services.appflow.model.ListConnectorsRequest;
import software.amazon.awssdk.services.appflow.model.ListConnectorsResponse;
import software.amazon.awssdk.services.appflow.model.ListFlowsRequest;
import software.amazon.awssdk.services.appflow.model.ListFlowsResponse;
import software.amazon.awssdk.services.appflow.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appflow.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appflow.model.RegisterConnectorRequest;
import software.amazon.awssdk.services.appflow.model.RegisterConnectorResponse;
import software.amazon.awssdk.services.appflow.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appflow.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appflow.model.StartFlowRequest;
import software.amazon.awssdk.services.appflow.model.StartFlowResponse;
import software.amazon.awssdk.services.appflow.model.StopFlowRequest;
import software.amazon.awssdk.services.appflow.model.StopFlowResponse;
import software.amazon.awssdk.services.appflow.model.TagResourceRequest;
import software.amazon.awssdk.services.appflow.model.TagResourceResponse;
import software.amazon.awssdk.services.appflow.model.ThrottlingException;
import software.amazon.awssdk.services.appflow.model.UnregisterConnectorRequest;
import software.amazon.awssdk.services.appflow.model.UnregisterConnectorResponse;
import software.amazon.awssdk.services.appflow.model.UnsupportedOperationException;
import software.amazon.awssdk.services.appflow.model.UntagResourceRequest;
import software.amazon.awssdk.services.appflow.model.UntagResourceResponse;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorRegistrationRequest;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorRegistrationResponse;
import software.amazon.awssdk.services.appflow.model.UpdateFlowRequest;
import software.amazon.awssdk.services.appflow.model.UpdateFlowResponse;
import software.amazon.awssdk.services.appflow.model.ValidationException;
import software.amazon.awssdk.services.appflow.paginators.DescribeConnectorProfilesPublisher;
import software.amazon.awssdk.services.appflow.paginators.DescribeConnectorsPublisher;
import software.amazon.awssdk.services.appflow.paginators.DescribeFlowExecutionRecordsPublisher;
import software.amazon.awssdk.services.appflow.paginators.ListConnectorsPublisher;
import software.amazon.awssdk.services.appflow.paginators.ListFlowsPublisher;
import software.amazon.awssdk.services.appflow.transform.CancelFlowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.CreateConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DeleteConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorEntityRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorProfilesRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeFlowExecutionRecordsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListConnectorEntitiesRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.RegisterConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UnregisterConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateConnectorRegistrationRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateFlowRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppflowAsyncClient}.
 *
 * @see AppflowAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppflowAsyncClient implements AppflowAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppflowAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AppflowServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAppflowAsyncClient(AppflowServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels active runs for a flow.
     * </p>
     * <p>
     * You can cancel all of the active runs for a flow, or you can cancel specific runs by providing their IDs.
     * </p>
     * <p>
     * You can cancel a flow run only when the run is in progress. You can't cancel a run that has already completed or
     * failed. You also can't cancel a run that's scheduled to occur but hasn't started yet. To prevent a scheduled run,
     * you can deactivate the flow with the <code>StopFlow</code> action.
     * </p>
     * <p>
     * You cannot resume a run after you cancel it.
     * </p>
     * <p>
     * When you send your request, the status for each run becomes <code>CancelStarted</code>. When the cancellation
     * completes, the status becomes <code>Canceled</code>.
     * </p>
     * <note>
     * <p>
     * When you cancel a run, you still incur charges for any data that the run already processed before the
     * cancellation. If the run had already written some data to the flow destination, then that data remains in the
     * destination. If you configured the flow to use a batch API (such as the Salesforce Bulk API 2.0), then the run
     * will finish reading or writing its entire batch of data after the cancellation. For these operations, the data
     * processing charges for Amazon AppFlow apply. For the pricing information, see <a
     * href="http://aws.amazon.com/appflow/pricing/">Amazon AppFlow pricing</a>.
     * </p>
     * </note>
     *
     * @param cancelFlowExecutionsRequest
     * @return A Java Future containing the result of the CancelFlowExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>AccessDeniedException AppFlow/Requester has invalid or missing permissions.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ThrottlingException API calls have exceeded the maximum allowed API request rate per account and per
     *         Region.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.CancelFlowExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CancelFlowExecutions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelFlowExecutionsResponse> cancelFlowExecutions(
            CancelFlowExecutionsRequest cancelFlowExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelFlowExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelFlowExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelFlowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelFlowExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelFlowExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelFlowExecutionsRequest, CancelFlowExecutionsResponse>()
                            .withOperationName("CancelFlowExecutions")
                            .withMarshaller(new CancelFlowExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelFlowExecutionsRequest));
            CompletableFuture<CancelFlowExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new connector profile associated with your Amazon Web Services account. There is a soft quota of 100
     * connector profiles per Amazon Web Services account. If you need more connector profiles than this quota allows,
     * you can submit a request to the Amazon AppFlow team through the Amazon AppFlow support channel. In each connector
     * profile that you create, you can provide the credentials and properties for only one connector.
     * </p>
     *
     * @param createConnectorProfileRequest
     * @return A Java Future containing the result of the CreateConnectorProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.CreateConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CreateConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectorProfileResponse> createConnectorProfile(
            CreateConnectorProfileRequest createConnectorProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnectorProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectorProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConnectorProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectorProfileRequest, CreateConnectorProfileResponse>()
                            .withOperationName("CreateConnectorProfile")
                            .withMarshaller(new CreateConnectorProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConnectorProfileRequest));
            CompletableFuture<CreateConnectorProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables your application to create a new flow using Amazon AppFlow. You must create a connector profile before
     * calling this API. Please note that the Request Syntax below shows syntax for multiple destinations, however, you
     * can only transfer data to one item in this list at a time. Amazon AppFlow does not currently support flows to
     * multiple destinations at once.
     * </p>
     *
     * @param createFlowRequest
     * @return A Java Future containing the result of the CreateFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.CreateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateFlowResponse> createFlow(CreateFlowRequest createFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>().withOperationName("CreateFlow")
                            .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createFlowRequest));
            CompletableFuture<CreateFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables you to delete an existing connector profile.
     * </p>
     *
     * @param deleteConnectorProfileRequest
     * @return A Java Future containing the result of the DeleteConnectorProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DeleteConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DeleteConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectorProfileResponse> deleteConnectorProfile(
            DeleteConnectorProfileRequest deleteConnectorProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnectorProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectorProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConnectorProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectorProfileRequest, DeleteConnectorProfileResponse>()
                            .withOperationName("DeleteConnectorProfile")
                            .withMarshaller(new DeleteConnectorProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConnectorProfileRequest));
            CompletableFuture<DeleteConnectorProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables your application to delete an existing flow. Before deleting the flow, Amazon AppFlow validates the
     * request by checking the flow configuration and status. You can delete flows one at a time.
     * </p>
     *
     * @param deleteFlowRequest
     * @return A Java Future containing the result of the DeleteFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DeleteFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteFlowResponse> deleteFlow(DeleteFlowRequest deleteFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>().withOperationName("DeleteFlow")
                            .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteFlowRequest));
            CompletableFuture<DeleteFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the given custom connector registered in your Amazon Web Services account. This API can be used for
     * custom connectors that are registered in your account and also for Amazon authored connectors.
     * </p>
     *
     * @param describeConnectorRequest
     * @return A Java Future containing the result of the DescribeConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectorResponse> describeConnector(DescribeConnectorRequest describeConnectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorRequest, DescribeConnectorResponse>()
                            .withOperationName("DescribeConnector")
                            .withMarshaller(new DescribeConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConnectorRequest));
            CompletableFuture<DescribeConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides details regarding the entity used with the connector, with a description of the data model for each
     * field in that entity.
     * </p>
     *
     * @param describeConnectorEntityRequest
     * @return A Java Future containing the result of the DescribeConnectorEntity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnectorEntity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectorEntity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectorEntityResponse> describeConnectorEntity(
            DescribeConnectorEntityRequest describeConnectorEntityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorEntityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectorEntity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectorEntityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectorEntityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectorEntityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorEntityRequest, DescribeConnectorEntityResponse>()
                            .withOperationName("DescribeConnectorEntity")
                            .withMarshaller(new DescribeConnectorEntityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConnectorEntityRequest));
            CompletableFuture<DescribeConnectorEntityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>connector-profile</code> details matching the provided <code>connector-profile</code>
     * names and <code>connector-types</code>. Both input lists are optional, and you can use them to filter the result.
     * </p>
     * <p>
     * If no names or <code>connector-types</code> are provided, returns all connector profiles in a paginated form. If
     * there is no match, this operation returns an empty list.
     * </p>
     *
     * @param describeConnectorProfilesRequest
     * @return A Java Future containing the result of the DescribeConnectorProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnectorProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectorProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectorProfilesResponse> describeConnectorProfiles(
            DescribeConnectorProfilesRequest describeConnectorProfilesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectorProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectorProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectorProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectorProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorProfilesRequest, DescribeConnectorProfilesResponse>()
                            .withOperationName("DescribeConnectorProfiles")
                            .withMarshaller(new DescribeConnectorProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConnectorProfilesRequest));
            CompletableFuture<DescribeConnectorProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>connector-profile</code> details matching the provided <code>connector-profile</code>
     * names and <code>connector-types</code>. Both input lists are optional, and you can use them to filter the result.
     * </p>
     * <p>
     * If no names or <code>connector-types</code> are provided, returns all connector profiles in a paginated form. If
     * there is no match, this operation returns an empty list.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeConnectorProfiles(software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeConnectorProfilesPublisher publisher = client.describeConnectorProfilesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeConnectorProfilesPublisher publisher = client.describeConnectorProfilesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeConnectorProfiles(software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeConnectorProfilesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnectorProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectorProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeConnectorProfilesPublisher describeConnectorProfilesPaginator(
            DescribeConnectorProfilesRequest describeConnectorProfilesRequest) {
        return new DescribeConnectorProfilesPublisher(this, applyPaginatorUserAgent(describeConnectorProfilesRequest));
    }

    /**
     * <p>
     * Describes the connectors vended by Amazon AppFlow for specified connector types. If you don't specify a connector
     * type, this operation describes all connectors vended by Amazon AppFlow. If there are more connectors than can be
     * returned in one page, the response contains a <code>nextToken</code> object, which can be be passed in to the
     * next call to the <code>DescribeConnectors</code> API operation to retrieve the next page.
     * </p>
     *
     * @param describeConnectorsRequest
     * @return A Java Future containing the result of the DescribeConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectorsResponse> describeConnectors(DescribeConnectorsRequest describeConnectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorsRequest, DescribeConnectorsResponse>()
                            .withOperationName("DescribeConnectors")
                            .withMarshaller(new DescribeConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConnectorsRequest));
            CompletableFuture<DescribeConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the connectors vended by Amazon AppFlow for specified connector types. If you don't specify a connector
     * type, this operation describes all connectors vended by Amazon AppFlow. If there are more connectors than can be
     * returned in one page, the response contains a <code>nextToken</code> object, which can be be passed in to the
     * next call to the <code>DescribeConnectors</code> API operation to retrieve the next page.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeConnectors(software.amazon.awssdk.services.appflow.model.DescribeConnectorsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeConnectorsPublisher publisher = client.describeConnectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeConnectorsPublisher publisher = client.describeConnectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appflow.model.DescribeConnectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appflow.model.DescribeConnectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeConnectors(software.amazon.awssdk.services.appflow.model.DescribeConnectorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeConnectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeConnectorsPublisher describeConnectorsPaginator(DescribeConnectorsRequest describeConnectorsRequest) {
        return new DescribeConnectorsPublisher(this, applyPaginatorUserAgent(describeConnectorsRequest));
    }

    /**
     * <p>
     * Provides a description of the specified flow.
     * </p>
     *
     * @param describeFlowRequest
     * @return A Java Future containing the result of the DescribeFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFlowResponse> describeFlow(DescribeFlowRequest describeFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                            .withOperationName("DescribeFlow").withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFlowRequest));
            CompletableFuture<DescribeFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the execution history of the flow.
     * </p>
     *
     * @param describeFlowExecutionRecordsRequest
     * @return A Java Future containing the result of the DescribeFlowExecutionRecords operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeFlowExecutionRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeFlowExecutionRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFlowExecutionRecordsResponse> describeFlowExecutionRecords(
            DescribeFlowExecutionRecordsRequest describeFlowExecutionRecordsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowExecutionRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlowExecutionRecords");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFlowExecutionRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFlowExecutionRecordsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFlowExecutionRecordsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFlowExecutionRecordsRequest, DescribeFlowExecutionRecordsResponse>()
                            .withOperationName("DescribeFlowExecutionRecords")
                            .withMarshaller(new DescribeFlowExecutionRecordsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFlowExecutionRecordsRequest));
            CompletableFuture<DescribeFlowExecutionRecordsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Fetches the execution history of the flow.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeFlowExecutionRecords(software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeFlowExecutionRecordsPublisher publisher = client.describeFlowExecutionRecordsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.DescribeFlowExecutionRecordsPublisher publisher = client.describeFlowExecutionRecordsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeFlowExecutionRecords(software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeFlowExecutionRecordsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.DescribeFlowExecutionRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeFlowExecutionRecords"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeFlowExecutionRecordsPublisher describeFlowExecutionRecordsPaginator(
            DescribeFlowExecutionRecordsRequest describeFlowExecutionRecordsRequest) {
        return new DescribeFlowExecutionRecordsPublisher(this, applyPaginatorUserAgent(describeFlowExecutionRecordsRequest));
    }

    /**
     * <p>
     * Returns the list of available connector entities supported by Amazon AppFlow. For example, you can query
     * Salesforce for <i>Account</i> and <i>Opportunity</i> entities, or query ServiceNow for the <i>Incident</i>
     * entity.
     * </p>
     *
     * @param listConnectorEntitiesRequest
     * @return A Java Future containing the result of the ListConnectorEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListConnectorEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListConnectorEntities" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectorEntitiesResponse> listConnectorEntities(
            ListConnectorEntitiesRequest listConnectorEntitiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectorEntities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectorEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectorEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectorEntitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectorEntitiesRequest, ListConnectorEntitiesResponse>()
                            .withOperationName("ListConnectorEntities")
                            .withMarshaller(new ListConnectorEntitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConnectorEntitiesRequest));
            CompletableFuture<ListConnectorEntitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of all registered custom connectors in your Amazon Web Services account. This API lists only
     * custom connectors registered in this account, not the Amazon Web Services authored connectors.
     * </p>
     *
     * @param listConnectorsRequest
     * @return A Java Future containing the result of the ListConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListConnectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectorsResponse> listConnectors(ListConnectorsRequest listConnectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                            .withOperationName("ListConnectors")
                            .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConnectorsRequest));
            CompletableFuture<ListConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of all registered custom connectors in your Amazon Web Services account. This API lists only
     * custom connectors registered in this account, not the Amazon Web Services authored connectors.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listConnectors(software.amazon.awssdk.services.appflow.model.ListConnectorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.ListConnectorsPublisher publisher = client.listConnectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.ListConnectorsPublisher publisher = client.listConnectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appflow.model.ListConnectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appflow.model.ListConnectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnectors(software.amazon.awssdk.services.appflow.model.ListConnectorsRequest)} operation.</b>
     * </p>
     *
     * @param listConnectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListConnectors" target="_top">AWS API
     *      Documentation</a>
     */
    public ListConnectorsPublisher listConnectorsPaginator(ListConnectorsRequest listConnectorsRequest) {
        return new ListConnectorsPublisher(this, applyPaginatorUserAgent(listConnectorsRequest));
    }

    /**
     * <p>
     * Lists all of the flows associated with your account.
     * </p>
     *
     * @param listFlowsRequest
     * @return A Java Future containing the result of the ListFlows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListFlowsResponse> listFlows(ListFlowsRequest listFlowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFlows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFlowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFlowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>().withOperationName("ListFlows")
                            .withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFlowsRequest));
            CompletableFuture<ListFlowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the flows associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listFlows(software.amazon.awssdk.services.appflow.model.ListFlowsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.appflow.paginators.ListFlowsPublisher publisher = client.listFlowsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.appflow.model.ListFlowsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.appflow.model.ListFlowsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFlows(software.amazon.awssdk.services.appflow.model.ListFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listFlowsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    public ListFlowsPublisher listFlowsPaginator(ListFlowsRequest listFlowsRequest) {
        return new ListFlowsPublisher(this, applyPaginatorUserAgent(listFlowsRequest));
    }

    /**
     * <p>
     * Retrieves the tags that are associated with a specified flow.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a new custom connector with your Amazon Web Services account. Before you can register the connector,
     * you must deploy the associated AWS lambda function in your account.
     * </p>
     *
     * @param registerConnectorRequest
     * @return A Java Future containing the result of the RegisterConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>AccessDeniedException AppFlow/Requester has invalid or missing permissions.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ThrottlingException API calls have exceeded the maximum allowed API request rate per account and per
     *         Region.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.RegisterConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/RegisterConnector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterConnectorResponse> registerConnector(RegisterConnectorRequest registerConnectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterConnectorRequest, RegisterConnectorResponse>()
                            .withOperationName("RegisterConnector")
                            .withMarshaller(new RegisterConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerConnectorRequest));
            CompletableFuture<RegisterConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates an existing flow. For on-demand flows, this operation runs the flow immediately. For schedule and
     * event-triggered flows, this operation activates the flow.
     * </p>
     *
     * @param startFlowRequest
     * @return A Java Future containing the result of the StartFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.StartFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartFlowResponse> startFlow(StartFlowRequest startFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>().withOperationName("StartFlow")
                            .withMarshaller(new StartFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(startFlowRequest));
            CompletableFuture<StartFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates the existing flow. For on-demand flows, this operation returns an
     * <code>unsupportedOperationException</code> error message. For schedule and event-triggered flows, this operation
     * deactivates the flow.
     * </p>
     *
     * @param stopFlowRequest
     * @return A Java Future containing the result of the StopFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>UnsupportedOperationException The requested operation is not supported for the current flow.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.StopFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopFlowResponse> stopFlow(StopFlowRequest stopFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>().withOperationName("StopFlow")
                            .withMarshaller(new StopFlowRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopFlowRequest));
            CompletableFuture<StopFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies a tag to the specified flow.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unregisters the custom connector registered in your account that matches the connector label provided in the
     * request.
     * </p>
     *
     * @param unregisterConnectorRequest
     * @return A Java Future containing the result of the UnregisterConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.UnregisterConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UnregisterConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UnregisterConnectorResponse> unregisterConnector(
            UnregisterConnectorRequest unregisterConnectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unregisterConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnregisterConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnregisterConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnregisterConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnregisterConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnregisterConnectorRequest, UnregisterConnectorResponse>()
                            .withOperationName("UnregisterConnector")
                            .withMarshaller(new UnregisterConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unregisterConnectorRequest));
            CompletableFuture<UnregisterConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from the specified flow.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a given connector profile associated with your account.
     * </p>
     *
     * @param updateConnectorProfileRequest
     * @return A Java Future containing the result of the UpdateConnectorProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.UpdateConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConnectorProfileResponse> updateConnectorProfile(
            UpdateConnectorProfileRequest updateConnectorProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnectorProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConnectorProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConnectorProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectorProfileRequest, UpdateConnectorProfileResponse>()
                            .withOperationName("UpdateConnectorProfile")
                            .withMarshaller(new UpdateConnectorProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConnectorProfileRequest));
            CompletableFuture<UpdateConnectorProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a custom connector that you've previously registered. This operation updates the connector with one of
     * the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The latest version of the AWS Lambda function that's assigned to the connector
     * </p>
     * </li>
     * <li>
     * <p>
     * A new AWS Lambda function that you specify
     * </p>
     * </li>
     * </ul>
     *
     * @param updateConnectorRegistrationRequest
     * @return A Java Future containing the result of the UpdateConnectorRegistration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>AccessDeniedException AppFlow/Requester has invalid or missing permissions.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ThrottlingException API calls have exceeded the maximum allowed API request rate per account and per
     *         Region.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.UpdateConnectorRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateConnectorRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConnectorRegistrationResponse> updateConnectorRegistration(
            UpdateConnectorRegistrationRequest updateConnectorRegistrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnectorRegistration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConnectorRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConnectorRegistrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConnectorRegistrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectorRegistrationRequest, UpdateConnectorRegistrationResponse>()
                            .withOperationName("UpdateConnectorRegistration")
                            .withMarshaller(new UpdateConnectorRegistrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConnectorRegistrationRequest));
            CompletableFuture<UpdateConnectorRegistrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing flow.
     * </p>
     *
     * @param updateFlowRequest
     * @return A Java Future containing the result of the UpdateFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request has invalid or missing parameters.</li>
     *         <li>ResourceNotFoundException The resource specified in the request (such as the source or destination
     *         connector profile) is not found.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota (such as the number of flows)
     *         to be exceeded.</li>
     *         <li>ConflictException There was a conflict when processing the request (for example, a flow with the
     *         given name already exists within the account. Check for conflicting resource names and try again.</li>
     *         <li>ConnectorAuthenticationException An error occurred when authenticating with the connector endpoint.</li>
     *         <li>ConnectorServerException An error occurred when retrieving data from the connector endpoint.</li>
     *         <li>InternalServerException An internal service error occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppflowException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppflowAsyncClient.UpdateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFlowResponse> updateFlow(UpdateFlowRequest updateFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFlowRequest, UpdateFlowResponse>().withOperationName("UpdateFlow")
                            .withMarshaller(new UpdateFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateFlowRequest));
            CompletableFuture<UpdateFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppflowServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppflowException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConnectorAuthenticationException")
                                .exceptionBuilderSupplier(ConnectorAuthenticationException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConnectorServerException")
                                .exceptionBuilderSupplier(ConnectorServerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AppflowRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
