/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appflow;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appflow.internal.AppflowServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.appflow.model.AccessDeniedException;
import software.amazon.awssdk.services.appflow.model.AppflowException;
import software.amazon.awssdk.services.appflow.model.CancelFlowExecutionsRequest;
import software.amazon.awssdk.services.appflow.model.CancelFlowExecutionsResponse;
import software.amazon.awssdk.services.appflow.model.ConflictException;
import software.amazon.awssdk.services.appflow.model.ConnectorAuthenticationException;
import software.amazon.awssdk.services.appflow.model.ConnectorServerException;
import software.amazon.awssdk.services.appflow.model.CreateConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.CreateConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.CreateFlowRequest;
import software.amazon.awssdk.services.appflow.model.CreateFlowResponse;
import software.amazon.awssdk.services.appflow.model.DeleteConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.DeleteConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.DeleteFlowRequest;
import software.amazon.awssdk.services.appflow.model.DeleteFlowResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorEntityRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorEntityResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorProfilesResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorResponse;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorsRequest;
import software.amazon.awssdk.services.appflow.model.DescribeConnectorsResponse;
import software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsRequest;
import software.amazon.awssdk.services.appflow.model.DescribeFlowExecutionRecordsResponse;
import software.amazon.awssdk.services.appflow.model.DescribeFlowRequest;
import software.amazon.awssdk.services.appflow.model.DescribeFlowResponse;
import software.amazon.awssdk.services.appflow.model.InternalServerException;
import software.amazon.awssdk.services.appflow.model.ListConnectorEntitiesRequest;
import software.amazon.awssdk.services.appflow.model.ListConnectorEntitiesResponse;
import software.amazon.awssdk.services.appflow.model.ListConnectorsRequest;
import software.amazon.awssdk.services.appflow.model.ListConnectorsResponse;
import software.amazon.awssdk.services.appflow.model.ListFlowsRequest;
import software.amazon.awssdk.services.appflow.model.ListFlowsResponse;
import software.amazon.awssdk.services.appflow.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appflow.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appflow.model.RegisterConnectorRequest;
import software.amazon.awssdk.services.appflow.model.RegisterConnectorResponse;
import software.amazon.awssdk.services.appflow.model.ResetConnectorMetadataCacheRequest;
import software.amazon.awssdk.services.appflow.model.ResetConnectorMetadataCacheResponse;
import software.amazon.awssdk.services.appflow.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appflow.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appflow.model.StartFlowRequest;
import software.amazon.awssdk.services.appflow.model.StartFlowResponse;
import software.amazon.awssdk.services.appflow.model.StopFlowRequest;
import software.amazon.awssdk.services.appflow.model.StopFlowResponse;
import software.amazon.awssdk.services.appflow.model.TagResourceRequest;
import software.amazon.awssdk.services.appflow.model.TagResourceResponse;
import software.amazon.awssdk.services.appflow.model.ThrottlingException;
import software.amazon.awssdk.services.appflow.model.UnregisterConnectorRequest;
import software.amazon.awssdk.services.appflow.model.UnregisterConnectorResponse;
import software.amazon.awssdk.services.appflow.model.UnsupportedOperationException;
import software.amazon.awssdk.services.appflow.model.UntagResourceRequest;
import software.amazon.awssdk.services.appflow.model.UntagResourceResponse;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorProfileRequest;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorProfileResponse;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorRegistrationRequest;
import software.amazon.awssdk.services.appflow.model.UpdateConnectorRegistrationResponse;
import software.amazon.awssdk.services.appflow.model.UpdateFlowRequest;
import software.amazon.awssdk.services.appflow.model.UpdateFlowResponse;
import software.amazon.awssdk.services.appflow.model.ValidationException;
import software.amazon.awssdk.services.appflow.transform.CancelFlowExecutionsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.CreateConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.CreateFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DeleteConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DeleteFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorEntityRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorProfilesRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeConnectorsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeFlowExecutionRecordsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.DescribeFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListConnectorEntitiesRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListFlowsRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.RegisterConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.ResetConnectorMetadataCacheRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.StartFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.StopFlowRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UnregisterConnectorRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateConnectorProfileRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateConnectorRegistrationRequestMarshaller;
import software.amazon.awssdk.services.appflow.transform.UpdateFlowRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppflowClient}.
 *
 * @see AppflowClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppflowClient implements AppflowClient {
    private static final Logger log = Logger.loggerFor(DefaultAppflowClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppflowClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels active runs for a flow.
     * </p>
     * <p>
     * You can cancel all of the active runs for a flow, or you can cancel specific runs by providing their IDs.
     * </p>
     * <p>
     * You can cancel a flow run only when the run is in progress. You can't cancel a run that has already completed or
     * failed. You also can't cancel a run that's scheduled to occur but hasn't started yet. To prevent a scheduled run,
     * you can deactivate the flow with the <code>StopFlow</code> action.
     * </p>
     * <p>
     * You cannot resume a run after you cancel it.
     * </p>
     * <p>
     * When you send your request, the status for each run becomes <code>CancelStarted</code>. When the cancellation
     * completes, the status becomes <code>Canceled</code>.
     * </p>
     * <note>
     * <p>
     * When you cancel a run, you still incur charges for any data that the run already processed before the
     * cancellation. If the run had already written some data to the flow destination, then that data remains in the
     * destination. If you configured the flow to use a batch API (such as the Salesforce Bulk API 2.0), then the run
     * will finish reading or writing its entire batch of data after the cancellation. For these operations, the data
     * processing charges for Amazon AppFlow apply. For the pricing information, see <a
     * href="http://aws.amazon.com/appflow/pricing/">Amazon AppFlow pricing</a>.
     * </p>
     * </note>
     *
     * @param cancelFlowExecutionsRequest
     * @return Result of the CancelFlowExecutions operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws AccessDeniedException
     *         AppFlow/Requester has invalid or missing permissions.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ThrottlingException
     *         API calls have exceeded the maximum allowed API request rate per account and per Region.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.CancelFlowExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CancelFlowExecutions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelFlowExecutionsResponse cancelFlowExecutions(CancelFlowExecutionsRequest cancelFlowExecutionsRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelFlowExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelFlowExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelFlowExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelFlowExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelFlowExecutions");

            return clientHandler.execute(new ClientExecutionParams<CancelFlowExecutionsRequest, CancelFlowExecutionsResponse>()
                    .withOperationName("CancelFlowExecutions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelFlowExecutionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelFlowExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new connector profile associated with your Amazon Web Services account. There is a soft quota of 100
     * connector profiles per Amazon Web Services account. If you need more connector profiles than this quota allows,
     * you can submit a request to the Amazon AppFlow team through the Amazon AppFlow support channel. In each connector
     * profile that you create, you can provide the credentials and properties for only one connector.
     * </p>
     *
     * @param createConnectorProfileRequest
     * @return Result of the CreateConnectorProfile operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.CreateConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CreateConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConnectorProfileResponse createConnectorProfile(CreateConnectorProfileRequest createConnectorProfileRequest)
            throws ValidationException, ConflictException, ServiceQuotaExceededException, ConnectorAuthenticationException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateConnectorProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnectorProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateConnectorProfileRequest, CreateConnectorProfileResponse>()
                            .withOperationName("CreateConnectorProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createConnectorProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateConnectorProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables your application to create a new flow using Amazon AppFlow. You must create a connector profile before
     * calling this API. Please note that the Request Syntax below shows syntax for multiple destinations, however, you
     * can only transfer data to one item in this list at a time. Amazon AppFlow does not currently support flows to
     * multiple destinations at once.
     * </p>
     *
     * @param createFlowRequest
     * @return Result of the CreateFlow operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws AccessDeniedException
     *         AppFlow/Requester has invalid or missing permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.CreateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/CreateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFlowResponse createFlow(CreateFlowRequest createFlowRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, ConflictException,
            ConnectorAuthenticationException, ConnectorServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFlow");

            return clientHandler.execute(new ClientExecutionParams<CreateFlowRequest, CreateFlowResponse>()
                    .withOperationName("CreateFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables you to delete an existing connector profile.
     * </p>
     *
     * @param deleteConnectorProfileRequest
     * @return Result of the DeleteConnectorProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DeleteConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DeleteConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConnectorProfileResponse deleteConnectorProfile(DeleteConnectorProfileRequest deleteConnectorProfileRequest)
            throws ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteConnectorProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnectorProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectorProfileRequest, DeleteConnectorProfileResponse>()
                            .withOperationName("DeleteConnectorProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteConnectorProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteConnectorProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables your application to delete an existing flow. Before deleting the flow, Amazon AppFlow validates the
     * request by checking the flow configuration and status. You can delete flows one at a time.
     * </p>
     *
     * @param deleteFlowRequest
     * @return Result of the DeleteFlow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DeleteFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DeleteFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFlowResponse deleteFlow(DeleteFlowRequest deleteFlowRequest) throws ResourceNotFoundException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFlow");

            return clientHandler.execute(new ClientExecutionParams<DeleteFlowRequest, DeleteFlowResponse>()
                    .withOperationName("DeleteFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the given custom connector registered in your Amazon Web Services account. This API can be used for
     * custom connectors that are registered in your account and also for Amazon authored connectors.
     * </p>
     *
     * @param describeConnectorRequest
     * @return Result of the DescribeConnector operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeConnectorResponse describeConnector(DescribeConnectorRequest describeConnectorRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnector");

            return clientHandler.execute(new ClientExecutionParams<DescribeConnectorRequest, DescribeConnectorResponse>()
                    .withOperationName("DescribeConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides details regarding the entity used with the connector, with a description of the data model for each
     * field in that entity.
     * </p>
     *
     * @param describeConnectorEntityRequest
     * @return Result of the DescribeConnectorEntity operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeConnectorEntity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectorEntity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeConnectorEntityResponse describeConnectorEntity(DescribeConnectorEntityRequest describeConnectorEntityRequest)
            throws ValidationException, ResourceNotFoundException, ConnectorAuthenticationException, ConnectorServerException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorEntityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConnectorEntityResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorEntityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorEntityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectorEntity");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorEntityRequest, DescribeConnectorEntityResponse>()
                            .withOperationName("DescribeConnectorEntity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeConnectorEntityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeConnectorEntityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>connector-profile</code> details matching the provided <code>connector-profile</code>
     * names and <code>connector-types</code>. Both input lists are optional, and you can use them to filter the result.
     * </p>
     * <p>
     * If no names or <code>connector-types</code> are provided, returns all connector profiles in a paginated form. If
     * there is no match, this operation returns an empty list.
     * </p>
     *
     * @param describeConnectorProfilesRequest
     * @return Result of the DescribeConnectorProfiles operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeConnectorProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectorProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeConnectorProfilesResponse describeConnectorProfiles(
            DescribeConnectorProfilesRequest describeConnectorProfilesRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConnectorProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectorProfiles");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorProfilesRequest, DescribeConnectorProfilesResponse>()
                            .withOperationName("DescribeConnectorProfiles").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeConnectorProfilesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeConnectorProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the connectors vended by Amazon AppFlow for specified connector types. If you don't specify a connector
     * type, this operation describes all connectors vended by Amazon AppFlow. If there are more connectors than can be
     * returned in one page, the response contains a <code>nextToken</code> object, which can be be passed in to the
     * next call to the <code>DescribeConnectors</code> API operation to retrieve the next page.
     * </p>
     *
     * @param describeConnectorsRequest
     * @return Result of the DescribeConnectors operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeConnectorsResponse describeConnectors(DescribeConnectorsRequest describeConnectorsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConnectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectors");

            return clientHandler.execute(new ClientExecutionParams<DescribeConnectorsRequest, DescribeConnectorsResponse>()
                    .withOperationName("DescribeConnectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeConnectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeConnectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a description of the specified flow.
     * </p>
     *
     * @param describeFlowRequest
     * @return Result of the DescribeFlow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFlowResponse describeFlow(DescribeFlowRequest describeFlowRequest) throws ResourceNotFoundException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlow");

            return clientHandler.execute(new ClientExecutionParams<DescribeFlowRequest, DescribeFlowResponse>()
                    .withOperationName("DescribeFlow").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFlowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Fetches the execution history of the flow.
     * </p>
     *
     * @param describeFlowExecutionRecordsRequest
     * @return Result of the DescribeFlowExecutionRecords operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.DescribeFlowExecutionRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/DescribeFlowExecutionRecords"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeFlowExecutionRecordsResponse describeFlowExecutionRecords(
            DescribeFlowExecutionRecordsRequest describeFlowExecutionRecordsRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFlowExecutionRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFlowExecutionRecordsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFlowExecutionRecordsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFlowExecutionRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFlowExecutionRecords");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFlowExecutionRecordsRequest, DescribeFlowExecutionRecordsResponse>()
                            .withOperationName("DescribeFlowExecutionRecords").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeFlowExecutionRecordsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFlowExecutionRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of available connector entities supported by Amazon AppFlow. For example, you can query
     * Salesforce for <i>Account</i> and <i>Opportunity</i> entities, or query ServiceNow for the <i>Incident</i>
     * entity.
     * </p>
     *
     * @param listConnectorEntitiesRequest
     * @return Result of the ListConnectorEntities operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.ListConnectorEntities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListConnectorEntities" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListConnectorEntitiesResponse listConnectorEntities(ListConnectorEntitiesRequest listConnectorEntitiesRequest)
            throws ValidationException, ResourceNotFoundException, ConnectorAuthenticationException, ConnectorServerException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectorEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConnectorEntitiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorEntitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorEntitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectorEntities");

            return clientHandler.execute(new ClientExecutionParams<ListConnectorEntitiesRequest, ListConnectorEntitiesResponse>()
                    .withOperationName("ListConnectorEntities").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConnectorEntitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectorEntitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of all registered custom connectors in your Amazon Web Services account. This API lists only
     * custom connectors registered in this account, not the Amazon Web Services authored connectors.
     * </p>
     *
     * @param listConnectorsRequest
     * @return Result of the ListConnectors operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListConnectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConnectorsResponse listConnectors(ListConnectorsRequest listConnectorsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConnectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");

            return clientHandler.execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                    .withOperationName("ListConnectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConnectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the flows associated with your account.
     * </p>
     *
     * @param listFlowsRequest
     * @return Result of the ListFlows operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.ListFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFlowsResponse listFlows(ListFlowsRequest listFlowsRequest) throws ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFlowsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFlows");

            return clientHandler.execute(new ClientExecutionParams<ListFlowsRequest, ListFlowsResponse>()
                    .withOperationName("ListFlows").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listFlowsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFlowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the tags that are associated with a specified flow.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers a new custom connector with your Amazon Web Services account. Before you can register the connector,
     * you must deploy the associated AWS lambda function in your account.
     * </p>
     *
     * @param registerConnectorRequest
     * @return Result of the RegisterConnector operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws AccessDeniedException
     *         AppFlow/Requester has invalid or missing permissions.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ThrottlingException
     *         API calls have exceeded the maximum allowed API request rate per account and per Region.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.RegisterConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/RegisterConnector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RegisterConnectorResponse registerConnector(RegisterConnectorRequest registerConnectorRequest)
            throws ValidationException, ConflictException, AccessDeniedException, ResourceNotFoundException,
            ServiceQuotaExceededException, ThrottlingException, InternalServerException, ConnectorServerException,
            ConnectorAuthenticationException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterConnector");

            return clientHandler.execute(new ClientExecutionParams<RegisterConnectorRequest, RegisterConnectorResponse>()
                    .withOperationName("RegisterConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(registerConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resets metadata about your connector entities that Amazon AppFlow stored in its cache. Use this action when you
     * want Amazon AppFlow to return the latest information about the data that you have in a source application.
     * </p>
     * <p>
     * Amazon AppFlow returns metadata about your entities when you use the ListConnectorEntities or
     * DescribeConnectorEntities actions. Following these actions, Amazon AppFlow caches the metadata to reduce the
     * number of API requests that it must send to the source application. Amazon AppFlow automatically resets the cache
     * once every hour, but you can use this action when you want to get the latest metadata right away.
     * </p>
     *
     * @param resetConnectorMetadataCacheRequest
     * @return Result of the ResetConnectorMetadataCache operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.ResetConnectorMetadataCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/ResetConnectorMetadataCache"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResetConnectorMetadataCacheResponse resetConnectorMetadataCache(
            ResetConnectorMetadataCacheRequest resetConnectorMetadataCacheRequest) throws ValidationException,
            ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetConnectorMetadataCacheResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResetConnectorMetadataCacheResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetConnectorMetadataCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetConnectorMetadataCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetConnectorMetadataCache");

            return clientHandler
                    .execute(new ClientExecutionParams<ResetConnectorMetadataCacheRequest, ResetConnectorMetadataCacheResponse>()
                            .withOperationName("ResetConnectorMetadataCache").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(resetConnectorMetadataCacheRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ResetConnectorMetadataCacheRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Activates an existing flow. For on-demand flows, this operation runs the flow immediately. For schedule and
     * event-triggered flows, this operation activates the flow.
     * </p>
     *
     * @param startFlowRequest
     * @return Result of the StartFlow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.StartFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/StartFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartFlowResponse startFlow(StartFlowRequest startFlowRequest) throws ResourceNotFoundException,
            InternalServerException, ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartFlow");

            return clientHandler.execute(new ClientExecutionParams<StartFlowRequest, StartFlowResponse>()
                    .withOperationName("StartFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deactivates the existing flow. For on-demand flows, this operation returns an
     * <code>unsupportedOperationException</code> error message. For schedule and event-triggered flows, this operation
     * deactivates the flow.
     * </p>
     *
     * @param stopFlowRequest
     * @return Result of the StopFlow operation returned by the service.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for the current flow.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.StopFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/StopFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopFlowResponse stopFlow(StopFlowRequest stopFlowRequest) throws ConflictException, ResourceNotFoundException,
            UnsupportedOperationException, InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopFlow");

            return clientHandler.execute(new ClientExecutionParams<StopFlowRequest, StopFlowResponse>()
                    .withOperationName("StopFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies a tag to the specified flow.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unregisters the custom connector registered in your account that matches the connector label provided in the
     * request.
     * </p>
     *
     * @param unregisterConnectorRequest
     * @return Result of the UnregisterConnector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.UnregisterConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UnregisterConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UnregisterConnectorResponse unregisterConnector(UnregisterConnectorRequest unregisterConnectorRequest)
            throws ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnregisterConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UnregisterConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unregisterConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unregisterConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnregisterConnector");

            return clientHandler.execute(new ClientExecutionParams<UnregisterConnectorRequest, UnregisterConnectorResponse>()
                    .withOperationName("UnregisterConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(unregisterConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnregisterConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from the specified flow.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a given connector profile associated with your account.
     * </p>
     *
     * @param updateConnectorProfileRequest
     * @return Result of the UpdateConnectorProfile operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.UpdateConnectorProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateConnectorProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConnectorProfileResponse updateConnectorProfile(UpdateConnectorProfileRequest updateConnectorProfileRequest)
            throws ValidationException, ResourceNotFoundException, ConflictException, ConnectorAuthenticationException,
            InternalServerException, AwsServiceException, SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConnectorProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConnectorProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConnectorProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnectorProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectorProfileRequest, UpdateConnectorProfileResponse>()
                            .withOperationName("UpdateConnectorProfile").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateConnectorProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateConnectorProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a custom connector that you've previously registered. This operation updates the connector with one of
     * the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The latest version of the AWS Lambda function that's assigned to the connector
     * </p>
     * </li>
     * <li>
     * <p>
     * A new AWS Lambda function that you specify
     * </p>
     * </li>
     * </ul>
     *
     * @param updateConnectorRegistrationRequest
     * @return Result of the UpdateConnectorRegistration operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws AccessDeniedException
     *         AppFlow/Requester has invalid or missing permissions.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ThrottlingException
     *         API calls have exceeded the maximum allowed API request rate per account and per Region.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.UpdateConnectorRegistration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateConnectorRegistration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConnectorRegistrationResponse updateConnectorRegistration(
            UpdateConnectorRegistrationRequest updateConnectorRegistrationRequest) throws ValidationException, ConflictException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, ThrottlingException,
            InternalServerException, ConnectorServerException, ConnectorAuthenticationException, AwsServiceException,
            SdkClientException, AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConnectorRegistrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConnectorRegistrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConnectorRegistrationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorRegistrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnectorRegistration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectorRegistrationRequest, UpdateConnectorRegistrationResponse>()
                            .withOperationName("UpdateConnectorRegistration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateConnectorRegistrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateConnectorRegistrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing flow.
     * </p>
     *
     * @param updateFlowRequest
     * @return Result of the UpdateFlow operation returned by the service.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ResourceNotFoundException
     *         The resource specified in the request (such as the source or destination connector profile) is not found.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota (such as the number of flows) to be exceeded.
     * @throws ConflictException
     *         There was a conflict when processing the request (for example, a flow with the given name already exists
     *         within the account. Check for conflicting resource names and try again.
     * @throws ConnectorAuthenticationException
     *         An error occurred when authenticating with the connector endpoint.
     * @throws ConnectorServerException
     *         An error occurred when retrieving data from the connector endpoint.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws AccessDeniedException
     *         AppFlow/Requester has invalid or missing permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppflowException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppflowClient.UpdateFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appflow-2020-08-23/UpdateFlow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFlowResponse updateFlow(UpdateFlowRequest updateFlowRequest) throws ValidationException,
            ResourceNotFoundException, ServiceQuotaExceededException, ConflictException, ConnectorAuthenticationException,
            ConnectorServerException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            AppflowException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFlowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFlowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Appflow");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFlow");

            return clientHandler.execute(new ClientExecutionParams<UpdateFlowRequest, UpdateFlowResponse>()
                    .withOperationName("UpdateFlow").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateFlowRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFlowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AppflowServiceClientConfigurationBuilder serviceConfigBuilder = new AppflowServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppflowException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConnectorAuthenticationException")
                                .exceptionBuilderSupplier(ConnectorAuthenticationException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConnectorServerException")
                                .exceptionBuilderSupplier(ConnectorServerException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build());
    }

    @Override
    public final AppflowServiceClientConfiguration serviceClientConfiguration() {
        return new AppflowServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
