/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appintegrations;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appintegrations.model.AccessDeniedException;
import software.amazon.awssdk.services.appintegrations.model.AppIntegrationsException;
import software.amazon.awssdk.services.appintegrations.model.CreateDataIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.CreateDataIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.CreateEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.CreateEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.DeleteDataIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.DeleteDataIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.DeleteEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.DeleteEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.DuplicateResourceException;
import software.amazon.awssdk.services.appintegrations.model.GetDataIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.GetDataIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.GetEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.GetEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.InternalServiceErrorException;
import software.amazon.awssdk.services.appintegrations.model.InvalidRequestException;
import software.amazon.awssdk.services.appintegrations.model.ListDataIntegrationAssociationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListDataIntegrationAssociationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListDataIntegrationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListDataIntegrationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationAssociationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationAssociationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationsRequest;
import software.amazon.awssdk.services.appintegrations.model.ListEventIntegrationsResponse;
import software.amazon.awssdk.services.appintegrations.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appintegrations.model.ResourceQuotaExceededException;
import software.amazon.awssdk.services.appintegrations.model.TagResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.TagResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.ThrottlingException;
import software.amazon.awssdk.services.appintegrations.model.UntagResourceRequest;
import software.amazon.awssdk.services.appintegrations.model.UntagResourceResponse;
import software.amazon.awssdk.services.appintegrations.model.UpdateDataIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.UpdateDataIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.model.UpdateEventIntegrationRequest;
import software.amazon.awssdk.services.appintegrations.model.UpdateEventIntegrationResponse;
import software.amazon.awssdk.services.appintegrations.transform.CreateDataIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.CreateEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.DeleteDataIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.DeleteEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.GetDataIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.GetEventIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListDataIntegrationAssociationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListDataIntegrationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListEventIntegrationAssociationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListEventIntegrationsRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.UpdateDataIntegrationRequestMarshaller;
import software.amazon.awssdk.services.appintegrations.transform.UpdateEventIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppIntegrationsClient}.
 *
 * @see AppIntegrationsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppIntegrationsClient implements AppIntegrationsClient {
    private static final Logger log = Logger.loggerFor(DefaultAppIntegrationsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppIntegrationsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates and persists a DataIntegration resource.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <code>CreateDataIntegration</code> API.
     * </p>
     * </note>
     *
     * @param createDataIntegrationRequest
     * @return Result of the CreateDataIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceQuotaExceededException
     *         The allowed quota for the resource has been exceeded.
     * @throws DuplicateResourceException
     *         A resource with the specified name already exists.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.CreateDataIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/CreateDataIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataIntegrationResponse createDataIntegration(CreateDataIntegrationRequest createDataIntegrationRequest)
            throws InternalServiceErrorException, ResourceQuotaExceededException, DuplicateResourceException,
            ThrottlingException, InvalidRequestException, AccessDeniedException, AwsServiceException, SdkClientException,
            AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataIntegration");

            return clientHandler.execute(new ClientExecutionParams<CreateDataIntegrationRequest, CreateDataIntegrationResponse>()
                    .withOperationName("CreateDataIntegration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDataIntegrationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDataIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an EventIntegration, given a specified name, description, and a reference to an Amazon EventBridge bus in
     * your account and a partner event source that pushes events to that bus. No objects are created in the your
     * account, only metadata that is persisted on the EventIntegration control plane.
     * </p>
     *
     * @param createEventIntegrationRequest
     * @return Result of the CreateEventIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceQuotaExceededException
     *         The allowed quota for the resource has been exceeded.
     * @throws DuplicateResourceException
     *         A resource with the specified name already exists.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.CreateEventIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/CreateEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEventIntegrationResponse createEventIntegration(CreateEventIntegrationRequest createEventIntegrationRequest)
            throws InternalServiceErrorException, ResourceQuotaExceededException, DuplicateResourceException,
            ThrottlingException, InvalidRequestException, AccessDeniedException, AwsServiceException, SdkClientException,
            AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEventIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateEventIntegrationRequest, CreateEventIntegrationResponse>()
                            .withOperationName("CreateEventIntegration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createEventIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateEventIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the DataIntegration. Only DataIntegrations that don't have any DataIntegrationAssociations can be
     * deleted. Deleting a DataIntegration also deletes the underlying Amazon AppFlow flow and service linked role.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> API.
     * </p>
     * </note>
     *
     * @param deleteDataIntegrationRequest
     * @return Result of the DeleteDataIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.DeleteDataIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/DeleteDataIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataIntegrationResponse deleteDataIntegration(DeleteDataIntegrationRequest deleteDataIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataIntegration");

            return clientHandler.execute(new ClientExecutionParams<DeleteDataIntegrationRequest, DeleteDataIntegrationResponse>()
                    .withOperationName("DeleteDataIntegration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataIntegrationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDataIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified existing event integration. If the event integration is associated with clients, the
     * request is rejected.
     * </p>
     *
     * @param deleteEventIntegrationRequest
     * @return Result of the DeleteEventIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.DeleteEventIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/DeleteEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEventIntegrationResponse deleteEventIntegration(DeleteEventIntegrationRequest deleteEventIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEventIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteEventIntegrationRequest, DeleteEventIntegrationResponse>()
                            .withOperationName("DeleteEventIntegration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteEventIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteEventIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the DataIntegration.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> API.
     * </p>
     * </note>
     *
     * @param getDataIntegrationRequest
     * @return Result of the GetDataIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.GetDataIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/GetDataIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataIntegrationResponse getDataIntegration(GetDataIntegrationRequest getDataIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataIntegration");

            return clientHandler.execute(new ClientExecutionParams<GetDataIntegrationRequest, GetDataIntegrationResponse>()
                    .withOperationName("GetDataIntegration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDataIntegrationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDataIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the event integration.
     * </p>
     *
     * @param getEventIntegrationRequest
     * @return Result of the GetEventIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.GetEventIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/GetEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEventIntegrationResponse getEventIntegration(GetEventIntegrationRequest getEventIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEventIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventIntegration");

            return clientHandler.execute(new ClientExecutionParams<GetEventIntegrationRequest, GetEventIntegrationResponse>()
                    .withOperationName("GetEventIntegration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEventIntegrationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEventIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a paginated list of DataIntegration associations in the account.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> API.
     * </p>
     * </note>
     *
     * @param listDataIntegrationAssociationsRequest
     * @return Result of the ListDataIntegrationAssociations operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.ListDataIntegrationAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListDataIntegrationAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataIntegrationAssociationsResponse listDataIntegrationAssociations(
            ListDataIntegrationAssociationsRequest listDataIntegrationAssociationsRequest) throws InternalServiceErrorException,
            ThrottlingException, ResourceNotFoundException, InvalidRequestException, AccessDeniedException, AwsServiceException,
            SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataIntegrationAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataIntegrationAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listDataIntegrationAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataIntegrationAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataIntegrationAssociationsRequest, ListDataIntegrationAssociationsResponse>()
                            .withOperationName("ListDataIntegrationAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDataIntegrationAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataIntegrationAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a paginated list of DataIntegrations in the account.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> API.
     * </p>
     * </note>
     *
     * @param listDataIntegrationsRequest
     * @return Result of the ListDataIntegrations operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.ListDataIntegrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListDataIntegrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataIntegrationsResponse listDataIntegrations(ListDataIntegrationsRequest listDataIntegrationsRequest)
            throws InternalServiceErrorException, ThrottlingException, InvalidRequestException, AccessDeniedException,
            AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataIntegrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataIntegrationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataIntegrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataIntegrations");

            return clientHandler.execute(new ClientExecutionParams<ListDataIntegrationsRequest, ListDataIntegrationsResponse>()
                    .withOperationName("ListDataIntegrations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataIntegrationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataIntegrationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a paginated list of event integration associations in the account.
     * </p>
     *
     * @param listEventIntegrationAssociationsRequest
     * @return Result of the ListEventIntegrationAssociations operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.ListEventIntegrationAssociations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListEventIntegrationAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventIntegrationAssociationsResponse listEventIntegrationAssociations(
            ListEventIntegrationAssociationsRequest listEventIntegrationAssociationsRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventIntegrationAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEventIntegrationAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listEventIntegrationAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventIntegrationAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListEventIntegrationAssociationsRequest, ListEventIntegrationAssociationsResponse>()
                            .withOperationName("ListEventIntegrationAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listEventIntegrationAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListEventIntegrationAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a paginated list of event integrations in the account.
     * </p>
     *
     * @param listEventIntegrationsRequest
     * @return Result of the ListEventIntegrations operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.ListEventIntegrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListEventIntegrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventIntegrationsResponse listEventIntegrations(ListEventIntegrationsRequest listEventIntegrationsRequest)
            throws InternalServiceErrorException, ThrottlingException, InvalidRequestException, AccessDeniedException,
            AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventIntegrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEventIntegrationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventIntegrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventIntegrations");

            return clientHandler.execute(new ClientExecutionParams<ListEventIntegrationsRequest, ListEventIntegrationsResponse>()
                    .withOperationName("ListEventIntegrations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEventIntegrationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEventIntegrationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidRequestException, InternalServiceErrorException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidRequestException,
            InternalServiceErrorException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the description of a DataIntegration.
     * </p>
     * <note>
     * <p>
     * You cannot create a DataIntegration association for a DataIntegration that has been previously associated. Use a
     * different DataIntegration, or recreate the DataIntegration using the <a
     * href="https://docs.aws.amazon.com/appintegrations/latest/APIReference/API_CreateDataIntegration.html"
     * >CreateDataIntegration</a> API.
     * </p>
     * </note>
     *
     * @param updateDataIntegrationRequest
     * @return Result of the UpdateDataIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.UpdateDataIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/UpdateDataIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataIntegrationResponse updateDataIntegration(UpdateDataIntegrationRequest updateDataIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataIntegration");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataIntegrationRequest, UpdateDataIntegrationResponse>()
                    .withOperationName("UpdateDataIntegration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataIntegrationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the description of an event integration.
     * </p>
     *
     * @param updateEventIntegrationRequest
     * @return Result of the UpdateEventIntegration operation returned by the service.
     * @throws InternalServiceErrorException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppIntegrationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppIntegrationsClient.UpdateEventIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appintegrations-2020-07-29/UpdateEventIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEventIntegrationResponse updateEventIntegration(UpdateEventIntegrationRequest updateEventIntegrationRequest)
            throws InternalServiceErrorException, ThrottlingException, ResourceNotFoundException, InvalidRequestException,
            AccessDeniedException, AwsServiceException, SdkClientException, AppIntegrationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEventIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEventIntegrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppIntegrations");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventIntegration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateEventIntegrationRequest, UpdateEventIntegrationResponse>()
                            .withOperationName("UpdateEventIntegration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateEventIntegrationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateEventIntegrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppIntegrationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateResourceException")
                                .exceptionBuilderSupplier(DuplicateResourceException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceQuotaExceededException")
                                .exceptionBuilderSupplier(ResourceQuotaExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
