/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a step scaling policy configuration to use with Application Auto Scaling.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
 * >Step scaling policies</a> in the <i>Application Auto Scaling User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StepScalingPolicyConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<StepScalingPolicyConfiguration.Builder, StepScalingPolicyConfiguration> {
    private static final SdkField<String> ADJUSTMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdjustmentType").getter(getter(StepScalingPolicyConfiguration::adjustmentTypeAsString))
            .setter(setter(Builder::adjustmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdjustmentType").build()).build();

    private static final SdkField<List<StepAdjustment>> STEP_ADJUSTMENTS_FIELD = SdkField
            .<List<StepAdjustment>> builder(MarshallingType.LIST)
            .memberName("StepAdjustments")
            .getter(getter(StepScalingPolicyConfiguration::stepAdjustments))
            .setter(setter(Builder::stepAdjustments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepAdjustments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StepAdjustment> builder(MarshallingType.SDK_POJO)
                                            .constructor(StepAdjustment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MIN_ADJUSTMENT_MAGNITUDE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinAdjustmentMagnitude").getter(getter(StepScalingPolicyConfiguration::minAdjustmentMagnitude))
            .setter(setter(Builder::minAdjustmentMagnitude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinAdjustmentMagnitude").build())
            .build();

    private static final SdkField<Integer> COOLDOWN_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Cooldown").getter(getter(StepScalingPolicyConfiguration::cooldown)).setter(setter(Builder::cooldown))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cooldown").build()).build();

    private static final SdkField<String> METRIC_AGGREGATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricAggregationType").getter(getter(StepScalingPolicyConfiguration::metricAggregationTypeAsString))
            .setter(setter(Builder::metricAggregationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricAggregationType").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADJUSTMENT_TYPE_FIELD,
            STEP_ADJUSTMENTS_FIELD, MIN_ADJUSTMENT_MAGNITUDE_FIELD, COOLDOWN_FIELD, METRIC_AGGREGATION_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String adjustmentType;

    private final List<StepAdjustment> stepAdjustments;

    private final Integer minAdjustmentMagnitude;

    private final Integer cooldown;

    private final String metricAggregationType;

    private StepScalingPolicyConfiguration(BuilderImpl builder) {
        this.adjustmentType = builder.adjustmentType;
        this.stepAdjustments = builder.stepAdjustments;
        this.minAdjustmentMagnitude = builder.minAdjustmentMagnitude;
        this.cooldown = builder.cooldown;
        this.metricAggregationType = builder.metricAggregationType;
    }

    /**
     * <p>
     * Specifies how the <code>ScalingAdjustment</code> value in a <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
     * >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values are
     * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     * </p>
     * <p>
     * <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #adjustmentType}
     * will return {@link AdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #adjustmentTypeAsString}.
     * </p>
     * 
     * @return Specifies how the <code>ScalingAdjustment</code> value in a <a
     *         href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
     *         >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values
     *         are <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     *         </p>
     *         <p>
     *         <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
     * @see AdjustmentType
     */
    public final AdjustmentType adjustmentType() {
        return AdjustmentType.fromValue(adjustmentType);
    }

    /**
     * <p>
     * Specifies how the <code>ScalingAdjustment</code> value in a <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
     * >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values are
     * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     * </p>
     * <p>
     * <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #adjustmentType}
     * will return {@link AdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #adjustmentTypeAsString}.
     * </p>
     * 
     * @return Specifies how the <code>ScalingAdjustment</code> value in a <a
     *         href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
     *         >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values
     *         are <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
     *         </p>
     *         <p>
     *         <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
     * @see AdjustmentType
     */
    public final String adjustmentTypeAsString() {
        return adjustmentType;
    }

    /**
     * For responses, this returns true if the service returned a value for the StepAdjustments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStepAdjustments() {
        return stepAdjustments != null && !(stepAdjustments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of adjustments that enable you to scale based on the size of the alarm breach.
     * </p>
     * <p>
     * At least one step adjustment is required if you are adding a new step scaling policy configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStepAdjustments} method.
     * </p>
     * 
     * @return A set of adjustments that enable you to scale based on the size of the alarm breach.</p>
     *         <p>
     *         At least one step adjustment is required if you are adding a new step scaling policy configuration.
     */
    public final List<StepAdjustment> stepAdjustments() {
        return stepAdjustments;
    }

    /**
     * <p>
     * The minimum value to scale by when the adjustment type is <code>PercentChangeInCapacity</code>. For example,
     * suppose that you create a step scaling policy to scale out an Amazon ECS service by 25 percent and you specify a
     * <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the scaling policy is performed, 25
     * percent of 4 is 1. However, because you specified a <code>MinAdjustmentMagnitude</code> of 2, Application Auto
     * Scaling scales out the service by 2 tasks.
     * </p>
     * 
     * @return The minimum value to scale by when the adjustment type is <code>PercentChangeInCapacity</code>. For
     *         example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25 percent
     *         and you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the scaling
     *         policy is performed, 25 percent of 4 is 1. However, because you specified a
     *         <code>MinAdjustmentMagnitude</code> of 2, Application Auto Scaling scales out the service by 2 tasks.
     */
    public final Integer minAdjustmentMagnitude() {
        return minAdjustmentMagnitude;
    }

    /**
     * <p>
     * The amount of time, in seconds, to wait for a previous scaling activity to take effect. If not specified, the
     * default value is 300. For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/step-scaling-policy-overview.html#step-scaling-cooldown"
     * >Cooldown period</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * 
     * @return The amount of time, in seconds, to wait for a previous scaling activity to take effect. If not specified,
     *         the default value is 300. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/step-scaling-policy-overview.html#step-scaling-cooldown"
     *         >Cooldown period</a> in the <i>Application Auto Scaling User Guide</i>.
     */
    public final Integer cooldown() {
        return cooldown;
    }

    /**
     * <p>
     * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>, and
     * <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #metricAggregationType} will return {@link MetricAggregationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #metricAggregationTypeAsString}.
     * </p>
     * 
     * @return The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
     *         <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated as
     *         <code>Average</code>.
     * @see MetricAggregationType
     */
    public final MetricAggregationType metricAggregationType() {
        return MetricAggregationType.fromValue(metricAggregationType);
    }

    /**
     * <p>
     * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>, and
     * <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #metricAggregationType} will return {@link MetricAggregationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #metricAggregationTypeAsString}.
     * </p>
     * 
     * @return The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
     *         <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated as
     *         <code>Average</code>.
     * @see MetricAggregationType
     */
    public final String metricAggregationTypeAsString() {
        return metricAggregationType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(adjustmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStepAdjustments() ? stepAdjustments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(minAdjustmentMagnitude());
        hashCode = 31 * hashCode + Objects.hashCode(cooldown());
        hashCode = 31 * hashCode + Objects.hashCode(metricAggregationTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StepScalingPolicyConfiguration)) {
            return false;
        }
        StepScalingPolicyConfiguration other = (StepScalingPolicyConfiguration) obj;
        return Objects.equals(adjustmentTypeAsString(), other.adjustmentTypeAsString())
                && hasStepAdjustments() == other.hasStepAdjustments()
                && Objects.equals(stepAdjustments(), other.stepAdjustments())
                && Objects.equals(minAdjustmentMagnitude(), other.minAdjustmentMagnitude())
                && Objects.equals(cooldown(), other.cooldown())
                && Objects.equals(metricAggregationTypeAsString(), other.metricAggregationTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StepScalingPolicyConfiguration").add("AdjustmentType", adjustmentTypeAsString())
                .add("StepAdjustments", hasStepAdjustments() ? stepAdjustments() : null)
                .add("MinAdjustmentMagnitude", minAdjustmentMagnitude()).add("Cooldown", cooldown())
                .add("MetricAggregationType", metricAggregationTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdjustmentType":
            return Optional.ofNullable(clazz.cast(adjustmentTypeAsString()));
        case "StepAdjustments":
            return Optional.ofNullable(clazz.cast(stepAdjustments()));
        case "MinAdjustmentMagnitude":
            return Optional.ofNullable(clazz.cast(minAdjustmentMagnitude()));
        case "Cooldown":
            return Optional.ofNullable(clazz.cast(cooldown()));
        case "MetricAggregationType":
            return Optional.ofNullable(clazz.cast(metricAggregationTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StepScalingPolicyConfiguration, T> g) {
        return obj -> g.apply((StepScalingPolicyConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StepScalingPolicyConfiguration> {
        /**
         * <p>
         * Specifies how the <code>ScalingAdjustment</code> value in a <a
         * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
         * >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values are
         * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
         * </p>
         * <p>
         * <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
         * </p>
         * 
         * @param adjustmentType
         *        Specifies how the <code>ScalingAdjustment</code> value in a <a
         *        href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
         *        >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values
         *        are <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and
         *        <code>PercentChangeInCapacity</code>. </p>
         *        <p>
         *        <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
         * @see AdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdjustmentType
         */
        Builder adjustmentType(String adjustmentType);

        /**
         * <p>
         * Specifies how the <code>ScalingAdjustment</code> value in a <a
         * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
         * >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values are
         * <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and <code>PercentChangeInCapacity</code>.
         * </p>
         * <p>
         * <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
         * </p>
         * 
         * @param adjustmentType
         *        Specifies how the <code>ScalingAdjustment</code> value in a <a
         *        href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_StepAdjustment.html"
         *        >StepAdjustment</a> is interpreted (for example, an absolute number or a percentage). The valid values
         *        are <code>ChangeInCapacity</code>, <code>ExactCapacity</code>, and
         *        <code>PercentChangeInCapacity</code>. </p>
         *        <p>
         *        <code>AdjustmentType</code> is required if you are adding a new step scaling policy configuration.
         * @see AdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdjustmentType
         */
        Builder adjustmentType(AdjustmentType adjustmentType);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * <p>
         * At least one step adjustment is required if you are adding a new step scaling policy configuration.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.</p>
         *        <p>
         *        At least one step adjustment is required if you are adding a new step scaling policy configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * <p>
         * At least one step adjustment is required if you are adding a new step scaling policy configuration.
         * </p>
         * 
         * @param stepAdjustments
         *        A set of adjustments that enable you to scale based on the size of the alarm breach.</p>
         *        <p>
         *        At least one step adjustment is required if you are adding a new step scaling policy configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepAdjustments(StepAdjustment... stepAdjustments);

        /**
         * <p>
         * A set of adjustments that enable you to scale based on the size of the alarm breach.
         * </p>
         * <p>
         * At least one step adjustment is required if you are adding a new step scaling policy configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.StepAdjustment.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.StepAdjustment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.StepAdjustment.Builder#build()} is called
         * immediately and its result is passed to {@link #stepAdjustments(List<StepAdjustment>)}.
         * 
         * @param stepAdjustments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationautoscaling.model.StepAdjustment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepAdjustments(java.util.Collection<StepAdjustment>)
         */
        Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments);

        /**
         * <p>
         * The minimum value to scale by when the adjustment type is <code>PercentChangeInCapacity</code>. For example,
         * suppose that you create a step scaling policy to scale out an Amazon ECS service by 25 percent and you
         * specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the scaling policy is
         * performed, 25 percent of 4 is 1. However, because you specified a <code>MinAdjustmentMagnitude</code> of 2,
         * Application Auto Scaling scales out the service by 2 tasks.
         * </p>
         * 
         * @param minAdjustmentMagnitude
         *        The minimum value to scale by when the adjustment type is <code>PercentChangeInCapacity</code>. For
         *        example, suppose that you create a step scaling policy to scale out an Amazon ECS service by 25
         *        percent and you specify a <code>MinAdjustmentMagnitude</code> of 2. If the service has 4 tasks and the
         *        scaling policy is performed, 25 percent of 4 is 1. However, because you specified a
         *        <code>MinAdjustmentMagnitude</code> of 2, Application Auto Scaling scales out the service by 2 tasks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude);

        /**
         * <p>
         * The amount of time, in seconds, to wait for a previous scaling activity to take effect. If not specified, the
         * default value is 300. For more information, see <a href=
         * "https://docs.aws.amazon.com/autoscaling/application/userguide/step-scaling-policy-overview.html#step-scaling-cooldown"
         * >Cooldown period</a> in the <i>Application Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param cooldown
         *        The amount of time, in seconds, to wait for a previous scaling activity to take effect. If not
         *        specified, the default value is 300. For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/application/userguide/step-scaling-policy-overview.html#step-scaling-cooldown"
         *        >Cooldown period</a> in the <i>Application Auto Scaling User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cooldown(Integer cooldown);

        /**
         * <p>
         * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>,
         * and <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
         * </p>
         * 
         * @param metricAggregationType
         *        The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
         *        <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated
         *        as <code>Average</code>.
         * @see MetricAggregationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricAggregationType
         */
        Builder metricAggregationType(String metricAggregationType);

        /**
         * <p>
         * The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>, <code>Maximum</code>,
         * and <code>Average</code>. If the aggregation type is null, the value is treated as <code>Average</code>.
         * </p>
         * 
         * @param metricAggregationType
         *        The aggregation type for the CloudWatch metrics. Valid values are <code>Minimum</code>,
         *        <code>Maximum</code>, and <code>Average</code>. If the aggregation type is null, the value is treated
         *        as <code>Average</code>.
         * @see MetricAggregationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricAggregationType
         */
        Builder metricAggregationType(MetricAggregationType metricAggregationType);
    }

    static final class BuilderImpl implements Builder {
        private String adjustmentType;

        private List<StepAdjustment> stepAdjustments = DefaultSdkAutoConstructList.getInstance();

        private Integer minAdjustmentMagnitude;

        private Integer cooldown;

        private String metricAggregationType;

        private BuilderImpl() {
        }

        private BuilderImpl(StepScalingPolicyConfiguration model) {
            adjustmentType(model.adjustmentType);
            stepAdjustments(model.stepAdjustments);
            minAdjustmentMagnitude(model.minAdjustmentMagnitude);
            cooldown(model.cooldown);
            metricAggregationType(model.metricAggregationType);
        }

        public final String getAdjustmentType() {
            return adjustmentType;
        }

        public final void setAdjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
        }

        @Override
        public final Builder adjustmentType(String adjustmentType) {
            this.adjustmentType = adjustmentType;
            return this;
        }

        @Override
        public final Builder adjustmentType(AdjustmentType adjustmentType) {
            this.adjustmentType(adjustmentType == null ? null : adjustmentType.toString());
            return this;
        }

        public final List<StepAdjustment.Builder> getStepAdjustments() {
            List<StepAdjustment.Builder> result = StepAdjustmentsCopier.copyToBuilder(this.stepAdjustments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStepAdjustments(Collection<StepAdjustment.BuilderImpl> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copyFromBuilder(stepAdjustments);
        }

        @Override
        public final Builder stepAdjustments(Collection<StepAdjustment> stepAdjustments) {
            this.stepAdjustments = StepAdjustmentsCopier.copy(stepAdjustments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(StepAdjustment... stepAdjustments) {
            stepAdjustments(Arrays.asList(stepAdjustments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepAdjustments(Consumer<StepAdjustment.Builder>... stepAdjustments) {
            stepAdjustments(Stream.of(stepAdjustments).map(c -> StepAdjustment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getMinAdjustmentMagnitude() {
            return minAdjustmentMagnitude;
        }

        public final void setMinAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
        }

        @Override
        public final Builder minAdjustmentMagnitude(Integer minAdjustmentMagnitude) {
            this.minAdjustmentMagnitude = minAdjustmentMagnitude;
            return this;
        }

        public final Integer getCooldown() {
            return cooldown;
        }

        public final void setCooldown(Integer cooldown) {
            this.cooldown = cooldown;
        }

        @Override
        public final Builder cooldown(Integer cooldown) {
            this.cooldown = cooldown;
            return this;
        }

        public final String getMetricAggregationType() {
            return metricAggregationType;
        }

        public final void setMetricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
        }

        @Override
        public final Builder metricAggregationType(String metricAggregationType) {
            this.metricAggregationType = metricAggregationType;
            return this;
        }

        @Override
        public final Builder metricAggregationType(MetricAggregationType metricAggregationType) {
            this.metricAggregationType(metricAggregationType == null ? null : metricAggregationType.toString());
            return this;
        }

        @Override
        public StepScalingPolicyConfiguration build() {
            return new StepScalingPolicyConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
