/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.applicationautoscaling.model.ApplicationAutoScalingException;
import software.amazon.awssdk.services.applicationautoscaling.model.ConcurrentUpdateException;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.FailedResourceAccessException;
import software.amazon.awssdk.services.applicationautoscaling.model.InternalServiceException;
import software.amazon.awssdk.services.applicationautoscaling.model.InvalidNextTokenException;
import software.amazon.awssdk.services.applicationautoscaling.model.LimitExceededException;
import software.amazon.awssdk.services.applicationautoscaling.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.ObjectNotFoundException;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.ResourceNotFoundException;
import software.amazon.awssdk.services.applicationautoscaling.model.TagResourceRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.TagResourceResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.TooManyTagsException;
import software.amazon.awssdk.services.applicationautoscaling.model.UntagResourceRequest;
import software.amazon.awssdk.services.applicationautoscaling.model.UntagResourceResponse;
import software.amazon.awssdk.services.applicationautoscaling.model.ValidationException;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable;
import software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable;

/**
 * Service client for accessing Application Auto Scaling. This can be created using the static {@link #builder()}
 * method.
 *
 * <p>
 * With Application Auto Scaling, you can configure automatic scaling for the following resources:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Amazon AppStream 2.0 fleets
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon Aurora Replicas
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon Comprehend document classification and entity recognizer endpoints
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon DynamoDB tables and global secondary indexes throughput capacity
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon ECS services
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon ElastiCache for Redis clusters (replication groups)
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon EMR clusters
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon Keyspaces (for Apache Cassandra) tables
 * </p>
 * </li>
 * <li>
 * <p>
 * Lambda function provisioned concurrency
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon Managed Streaming for Apache Kafka broker storage
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon Neptune clusters
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon SageMaker endpoint variants
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon SageMaker inference components
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon SageMaker serverless endpoint provisioned concurrency
 * </p>
 * </li>
 * <li>
 * <p>
 * Spot Fleets (Amazon EC2)
 * </p>
 * </li>
 * <li>
 * <p>
 * Pool of WorkSpaces
 * </p>
 * </li>
 * <li>
 * <p>
 * Custom resources provided by your own applications or services
 * </p>
 * </li>
 * </ul>
 * <p>
 * To learn more about Application Auto Scaling, see the <a
 * href="https://docs.aws.amazon.com/autoscaling/application/userguide/what-is-application-auto-scaling.html"
 * >Application Auto Scaling User Guide</a>.
 * </p>
 * <p>
 * <b>API Summary</b>
 * </p>
 * <p>
 * The Application Auto Scaling service API includes three key sets of actions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Register and manage scalable targets - Register Amazon Web Services or custom resources as scalable targets (a
 * resource that Application Auto Scaling can scale), set minimum and maximum capacity limits, and retrieve information
 * on existing scalable targets.
 * </p>
 * </li>
 * <li>
 * <p>
 * Configure and manage automatic scaling - Define scaling policies to dynamically scale your resources in response to
 * CloudWatch alarms, schedule one-time or recurring scaling actions, and retrieve your recent scaling activity history.
 * </p>
 * </li>
 * <li>
 * <p>
 * Suspend and resume scaling - Temporarily suspend and later resume automatic scaling by calling the <a
 * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_RegisterScalableTarget.html"
 * >RegisterScalableTarget</a> API action for any Application Auto Scaling scalable target. You can suspend and resume
 * (individually or in combination) scale-out activities that are triggered by a scaling policy, scale-in activities
 * that are triggered by a scaling policy, and scheduled scaling.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ApplicationAutoScalingClient extends AwsClient {
    String SERVICE_NAME = "application-autoscaling";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "application-autoscaling";

    /**
     * <p>
     * Deletes the specified scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deleting a step scaling policy deletes the underlying alarm action, but does not delete the CloudWatch alarm
     * associated with the scaling policy, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/create-step-scaling-policy-cli.html#delete-step-scaling-policy"
     * >Delete a step scaling policy</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/create-target-tracking-policy-cli.html#delete-target-tracking-policy"
     * >Delete a target tracking scaling policy</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteScalingPolicyRequest
     * @return Result of the DeleteScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteScalingPolicyResponse deleteScalingPolicy(DeleteScalingPolicyRequest deleteScalingPolicyRequest)
            throws ValidationException, ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Deleting a step scaling policy deletes the underlying alarm action, but does not delete the CloudWatch alarm
     * associated with the scaling policy, even if it no longer has an associated action.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/create-step-scaling-policy-cli.html#delete-step-scaling-policy"
     * >Delete a step scaling policy</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/create-target-tracking-policy-cli.html#delete-target-tracking-policy"
     * >Delete a target tracking scaling policy</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteScalingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteScalingPolicyRequest#builder()}
     * </p>
     *
     * @param deleteScalingPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DeleteScalingPolicyRequest.Builder} to
     *        create a request.
     * @return Result of the DeleteScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteScalingPolicyResponse deleteScalingPolicy(
            Consumer<DeleteScalingPolicyRequest.Builder> deleteScalingPolicyRequest) throws ValidationException,
            ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return deleteScalingPolicy(DeleteScalingPolicyRequest.builder().applyMutation(deleteScalingPolicyRequest).build());
    }

    /**
     * <p>
     * Deletes the specified scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/scheduled-scaling-additional-cli-commands.html#delete-scheduled-action"
     * >Delete a scheduled action</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param deleteScheduledActionRequest
     * @return Result of the DeleteScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteScheduledActionResponse deleteScheduledAction(DeleteScheduledActionRequest deleteScheduledActionRequest)
            throws ValidationException, ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/scheduled-scaling-additional-cli-commands.html#delete-scheduled-action"
     * >Delete a scheduled action</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteScheduledActionRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteScheduledActionRequest#builder()}
     * </p>
     *
     * @param deleteScheduledActionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DeleteScheduledActionRequest.Builder}
     *        to create a request.
     * @return Result of the DeleteScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeleteScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeleteScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteScheduledActionResponse deleteScheduledAction(
            Consumer<DeleteScheduledActionRequest.Builder> deleteScheduledActionRequest) throws ValidationException,
            ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return deleteScheduledAction(DeleteScheduledActionRequest.builder().applyMutation(deleteScheduledActionRequest).build());
    }

    /**
     * <p>
     * Deregisters an Application Auto Scaling scalable target when you have finished using it. To see which resources
     * have been registered, use <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>.
     * </p>
     * <note>
     * <p>
     * Deregistering a scalable target deletes the scaling policies and the scheduled actions that are associated with
     * it.
     * </p>
     * </note>
     *
     * @param deregisterScalableTargetRequest
     * @return Result of the DeregisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeregisterScalableTarget
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeregisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default DeregisterScalableTargetResponse deregisterScalableTarget(
            DeregisterScalableTargetRequest deregisterScalableTargetRequest) throws ValidationException, ObjectNotFoundException,
            ConcurrentUpdateException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deregisters an Application Auto Scaling scalable target when you have finished using it. To see which resources
     * have been registered, use <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>.
     * </p>
     * <note>
     * <p>
     * Deregistering a scalable target deletes the scaling policies and the scheduled actions that are associated with
     * it.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeregisterScalableTargetRequest.Builder} avoiding
     * the need to create one manually via {@link DeregisterScalableTargetRequest#builder()}
     * </p>
     *
     * @param deregisterScalableTargetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DeregisterScalableTargetRequest.Builder}
     *        to create a request.
     * @return Result of the DeregisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DeregisterScalableTarget
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DeregisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default DeregisterScalableTargetResponse deregisterScalableTarget(
            Consumer<DeregisterScalableTargetRequest.Builder> deregisterScalableTargetRequest) throws ValidationException,
            ObjectNotFoundException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return deregisterScalableTarget(DeregisterScalableTargetRequest.builder().applyMutation(deregisterScalableTargetRequest)
                .build());
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return Result of the DescribeScalableTargets operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalableTargetsResponse describeScalableTargets(DescribeScalableTargetsRequest describeScalableTargetsRequest)
            throws ValidationException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about the scalable targets in the specified namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceIds</code> and <code>ScalableDimension</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalableTargetsRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalableTargetsRequest#builder()}
     * </p>
     *
     * @param describeScalableTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeScalableTargets operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalableTargetsResponse describeScalableTargets(
            Consumer<DescribeScalableTargetsRequest.Builder> describeScalableTargetsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScalableTargets(DescribeScalableTargetsRequest.builder().applyMutation(describeScalableTargetsRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client
     *             .describeScalableTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalableTargetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalableTargetsIterable describeScalableTargetsPaginator(
            DescribeScalableTargetsRequest describeScalableTargetsRequest) throws ValidationException, InvalidNextTokenException,
            ConcurrentUpdateException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return new DescribeScalableTargetsIterable(this, describeScalableTargetsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client
     *             .describeScalableTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalableTargetsIterable responses = client.describeScalableTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalableTargets(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalableTargetsRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalableTargetsRequest#builder()}
     * </p>
     *
     * @param describeScalableTargetsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalableTargetsRequest.Builder}
     *        to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalableTargets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalableTargets"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalableTargetsIterable describeScalableTargetsPaginator(
            Consumer<DescribeScalableTargetsRequest.Builder> describeScalableTargetsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScalableTargetsPaginator(DescribeScalableTargetsRequest.builder()
                .applyMutation(describeScalableTargetsRequest).build());
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * For information about viewing scaling activities using the Amazon Web Services CLI, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scaling-activities.html"
     * >Scaling activities for Application Auto Scaling</a>.
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return Result of the DescribeScalingActivities operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingActivitiesResponse describeScalingActivities(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Provides descriptive information about the scaling activities in the specified namespace from the previous six
     * weeks.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code> and <code>ScalableDimension</code>.
     * </p>
     * <p>
     * For information about viewing scaling activities using the Amazon Web Services CLI, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scaling-activities.html"
     * >Scaling activities for Application Auto Scaling</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalingActivitiesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalingActivitiesRequest#builder()}
     * </p>
     *
     * @param describeScalingActivitiesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeScalingActivities operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingActivitiesResponse describeScalingActivities(
            Consumer<DescribeScalingActivitiesRequest.Builder> describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScalingActivities(DescribeScalingActivitiesRequest.builder()
                .applyMutation(describeScalingActivitiesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client
     *             .describeScalingActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingActivitiesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingActivitiesIterable describeScalingActivitiesPaginator(
            DescribeScalingActivitiesRequest describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return new DescribeScalingActivitiesIterable(this, describeScalingActivitiesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client
     *             .describeScalingActivitiesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingActivitiesIterable responses = client.describeScalingActivitiesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingActivities(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalingActivitiesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalingActivitiesRequest#builder()}
     * </p>
     *
     * @param describeScalingActivitiesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingActivitiesRequest.Builder}
     *        to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingActivities
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingActivities"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingActivitiesIterable describeScalingActivitiesPaginator(
            Consumer<DescribeScalingActivitiesRequest.Builder> describeScalingActivitiesRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScalingActivitiesPaginator(DescribeScalingActivitiesRequest.builder()
                .applyMutation(describeScalingActivitiesRequest).build());
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target tracking scaling policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step scaling policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return Result of the DescribeScalingPolicies operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingPoliciesResponse describeScalingPolicies(DescribeScalingPoliciesRequest describeScalingPoliciesRequest)
            throws ValidationException, FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scaling policies for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>PolicyNames</code>.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target tracking scaling policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step scaling policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalingPoliciesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalingPoliciesRequest#builder()}
     * </p>
     *
     * @param describeScalingPoliciesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeScalingPolicies operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingPoliciesResponse describeScalingPolicies(
            Consumer<DescribeScalingPoliciesRequest.Builder> describeScalingPoliciesRequest) throws ValidationException,
            FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return describeScalingPolicies(DescribeScalingPoliciesRequest.builder().applyMutation(describeScalingPoliciesRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client
     *             .describeScalingPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScalingPoliciesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingPoliciesIterable describeScalingPoliciesPaginator(
            DescribeScalingPoliciesRequest describeScalingPoliciesRequest) throws ValidationException,
            FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return new DescribeScalingPoliciesIterable(this, describeScalingPoliciesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client
     *             .describeScalingPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScalingPoliciesIterable responses = client.describeScalingPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScalingPolicies(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScalingPoliciesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScalingPoliciesRequest#builder()}
     * </p>
     *
     * @param describeScalingPoliciesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScalingPoliciesRequest.Builder}
     *        to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScalingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScalingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScalingPoliciesIterable describeScalingPoliciesPaginator(
            Consumer<DescribeScalingPoliciesRequest.Builder> describeScalingPoliciesRequest) throws ValidationException,
            FailedResourceAccessException, InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return describeScalingPoliciesPaginator(DescribeScalingPoliciesRequest.builder()
                .applyMutation(describeScalingPoliciesRequest).build());
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html"
     * >Scheduled scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return Result of the DescribeScheduledActions operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScheduledActionsResponse describeScheduledActions(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the Application Auto Scaling scheduled actions for the specified service namespace.
     * </p>
     * <p>
     * You can filter the results using the <code>ResourceId</code>, <code>ScalableDimension</code>, and
     * <code>ScheduledActionNames</code> parameters.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html"
     * >Scheduled scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScheduledActionsRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScheduledActionsRequest#builder()}
     * </p>
     *
     * @param describeScheduledActionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest.Builder}
     *        to create a request.
     * @return Result of the DescribeScheduledActions operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScheduledActionsResponse describeScheduledActions(
            Consumer<DescribeScheduledActionsRequest.Builder> describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScheduledActions(DescribeScheduledActionsRequest.builder().applyMutation(describeScheduledActionsRequest)
                .build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client
     *             .describeScheduledActionsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeScheduledActionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScheduledActionsIterable describeScheduledActionsPaginator(
            DescribeScheduledActionsRequest describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return new DescribeScheduledActionsIterable(this, describeScheduledActionsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client
     *             .describeScheduledActionsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationautoscaling.paginators.DescribeScheduledActionsIterable responses = client.describeScheduledActionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeScheduledActions(software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeScheduledActionsRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeScheduledActionsRequest#builder()}
     * </p>
     *
     * @param describeScheduledActionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.DescribeScheduledActionsRequest.Builder}
     *        to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws InvalidNextTokenException
     *         The next token supplied was invalid.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.DescribeScheduledActions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/DescribeScheduledActions"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeScheduledActionsIterable describeScheduledActionsPaginator(
            Consumer<DescribeScheduledActionsRequest.Builder> describeScheduledActionsRequest) throws ValidationException,
            InvalidNextTokenException, ConcurrentUpdateException, InternalServiceException, AwsServiceException,
            SdkClientException, ApplicationAutoScalingException {
        return describeScheduledActionsPaginator(DescribeScheduledActionsRequest.builder()
                .applyMutation(describeScheduledActionsRequest).build());
    }

    /**
     * <p>
     * Returns all the tags on the specified Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For general information about tags, including the format and syntax, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging your Amazon Web Services
     * resources</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns all the tags on the specified Application Auto Scaling scalable target.
     * </p>
     * <p>
     * For general information about tags, including the format and syntax, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging your Amazon Web Services
     * resources</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.ListTagsForResourceRequest.Builder} to
     *        create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Creates or updates a scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scaling policy
     * applies to the scalable target identified by those three attributes. You cannot create a scaling policy until you
     * have registered the resource as a scalable target.
     * </p>
     * <p>
     * Multiple scaling policies can be in force at the same time for the same scalable target. You can have one or more
     * target tracking scaling policies, one or more step scaling policies, or both. However, there is a chance that
     * multiple policies could conflict, instructing the scalable target to scale out or in at the same time.
     * Application Auto Scaling gives precedence to the policy that provides the largest capacity for both scale out and
     * scale in. For example, if one policy increases capacity by 3, another policy increases capacity by 200 percent,
     * and the current capacity is 10, Application Auto Scaling uses the policy with the highest calculated capacity
     * (200% of 10 = 20) and scales out to 30.
     * </p>
     * <p>
     * We recommend caution, however, when using target tracking scaling policies with step scaling policies because
     * conflicts between these policies can cause undesirable behavior. For example, if the step scaling policy
     * initiates a scale-in activity before the target tracking policy is ready to scale in, the scale-in activity will
     * not be blocked. After the scale-in activity completes, the target tracking policy could instruct the scalable
     * target to scale out again.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target tracking scaling policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step scaling policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * If a scalable target is deregistered, the scalable target is no longer available to use scaling policies. Any
     * scaling policies that were specified for the scalable target are deleted.
     * </p>
     * </note>
     *
     * @param putScalingPolicyRequest
     * @return Result of the PutScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default PutScalingPolicyResponse putScalingPolicy(PutScalingPolicyRequest putScalingPolicyRequest)
            throws ValidationException, LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException,
            FailedResourceAccessException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates or updates a scaling policy for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scaling policy
     * applies to the scalable target identified by those three attributes. You cannot create a scaling policy until you
     * have registered the resource as a scalable target.
     * </p>
     * <p>
     * Multiple scaling policies can be in force at the same time for the same scalable target. You can have one or more
     * target tracking scaling policies, one or more step scaling policies, or both. However, there is a chance that
     * multiple policies could conflict, instructing the scalable target to scale out or in at the same time.
     * Application Auto Scaling gives precedence to the policy that provides the largest capacity for both scale out and
     * scale in. For example, if one policy increases capacity by 3, another policy increases capacity by 200 percent,
     * and the current capacity is 10, Application Auto Scaling uses the policy with the highest calculated capacity
     * (200% of 10 = 20) and scales out to 30.
     * </p>
     * <p>
     * We recommend caution, however, when using target tracking scaling policies with step scaling policies because
     * conflicts between these policies can cause undesirable behavior. For example, if the step scaling policy
     * initiates a scale-in activity before the target tracking policy is ready to scale in, the scale-in activity will
     * not be blocked. After the scale-in activity completes, the target tracking policy could instruct the scalable
     * target to scale out again.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target tracking scaling policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step scaling policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * If a scalable target is deregistered, the scalable target is no longer available to use scaling policies. Any
     * scaling policies that were specified for the scalable target are deleted.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutScalingPolicyRequest.Builder} avoiding the need
     * to create one manually via {@link PutScalingPolicyRequest#builder()}
     * </p>
     *
     * @param putScalingPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.PutScalingPolicyRequest.Builder} to
     *        create a request.
     * @return Result of the PutScalingPolicy operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws FailedResourceAccessException
     *         Failed access to resources caused an exception. This exception is thrown when Application Auto Scaling is
     *         unable to retrieve the alarms associated with a scaling policy due to a client error, for example, if the
     *         role ARN specified for a scalable target does not have permission to call the CloudWatch <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarms.html"
     *         >DescribeAlarms</a> on your behalf.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScalingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScalingPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default PutScalingPolicyResponse putScalingPolicy(Consumer<PutScalingPolicyRequest.Builder> putScalingPolicyRequest)
            throws ValidationException, LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException,
            FailedResourceAccessException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return putScalingPolicy(PutScalingPolicyRequest.builder().applyMutation(putScalingPolicyRequest).build());
    }

    /**
     * <p>
     * Creates or updates a scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scheduled
     * action applies to the scalable target identified by those three attributes. You cannot create a scheduled action
     * until you have registered the resource as a scalable target.
     * </p>
     * <p>
     * When you specify start and end times with a recurring schedule using a cron expression or rates, they form the
     * boundaries for when the recurring action starts and stops.
     * </p>
     * <p>
     * To update a scheduled action, specify the parameters that you want to change. If you don't specify start and end
     * times, the old values are deleted.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html"
     * >Scheduled scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * If a scalable target is deregistered, the scalable target is no longer available to run scheduled actions. Any
     * scheduled actions that were specified for the scalable target are deleted.
     * </p>
     * </note>
     *
     * @param putScheduledActionRequest
     * @return Result of the PutScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    default PutScheduledActionResponse putScheduledAction(PutScheduledActionRequest putScheduledActionRequest)
            throws ValidationException, LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates or updates a scheduled action for an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each scalable target is identified by a service namespace, resource ID, and scalable dimension. A scheduled
     * action applies to the scalable target identified by those three attributes. You cannot create a scheduled action
     * until you have registered the resource as a scalable target.
     * </p>
     * <p>
     * When you specify start and end times with a recurring schedule using a cron expression or rates, they form the
     * boundaries for when the recurring action starts and stops.
     * </p>
     * <p>
     * To update a scheduled action, specify the parameters that you want to change. If you don't specify start and end
     * times, the old values are deleted.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-scheduled-scaling.html"
     * >Scheduled scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <note>
     * <p>
     * If a scalable target is deregistered, the scalable target is no longer available to run scheduled actions. Any
     * scheduled actions that were specified for the scalable target are deleted.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutScheduledActionRequest.Builder} avoiding the
     * need to create one manually via {@link PutScheduledActionRequest#builder()}
     * </p>
     *
     * @param putScheduledActionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.PutScheduledActionRequest.Builder} to
     *        create a request.
     * @return Result of the PutScheduledAction operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ObjectNotFoundException
     *         The specified object could not be found. For any operation that depends on the existence of a scalable
     *         target, this exception is thrown if the scalable target with the specified service namespace, resource
     *         ID, and scalable dimension does not exist. For any operation that deletes or deregisters a resource, this
     *         exception is thrown if the resource cannot be found.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.PutScheduledAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/PutScheduledAction"
     *      target="_top">AWS API Documentation</a>
     */
    default PutScheduledActionResponse putScheduledAction(Consumer<PutScheduledActionRequest.Builder> putScheduledActionRequest)
            throws ValidationException, LimitExceededException, ObjectNotFoundException, ConcurrentUpdateException,
            InternalServiceException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        return putScheduledAction(PutScheduledActionRequest.builder().applyMutation(putScheduledActionRequest).build());
    }

    /**
     * <p>
     * Registers or updates a scalable target, which is the resource that you want to scale.
     * </p>
     * <p>
     * Scalable targets are uniquely identified by the combination of resource ID, scalable dimension, and namespace,
     * which represents some capacity dimension of the underlying service.
     * </p>
     * <p>
     * When you register a new scalable target, you must specify values for the minimum and maximum capacity. If the
     * specified resource is not active in the target service, this operation does not change the resource's current
     * capacity. Otherwise, it changes the resource's current capacity to a value that is inside of this range.
     * </p>
     * <p>
     * If you add a scaling policy, current capacity is adjustable within the specified range when scaling starts.
     * Application Auto Scaling scaling policies will not scale capacity to values that are outside of the minimum and
     * maximum range.
     * </p>
     * <p>
     * After you register a scalable target, you do not need to register it again to use other Application Auto Scaling
     * operations. To see which resources have been registered, use <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>. You can also view the scaling policies for a service namespace by using <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>. If you no longer need a scalable target, you can deregister it by using <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DeregisterScalableTarget.html"
     * >DeregisterScalableTarget</a>.
     * </p>
     * <p>
     * To update a scalable target, specify the parameters that you want to change. Include the parameters that identify
     * the scalable target: resource ID, scalable dimension, and namespace. Any parameters that you don't specify are
     * not changed by this update request.
     * </p>
     * <note>
     * <p>
     * If you call the <code>RegisterScalableTarget</code> API operation to create a scalable target, there might be a
     * brief delay until the operation achieves <a href="https://en.wikipedia.org/wiki/Eventual_consistency">eventual
     * consistency</a>. You might become aware of this brief delay if you get unexpected errors when performing
     * sequential operations. The typical strategy is to retry the request, and some Amazon Web Services SDKs include
     * automatic backoff and retry logic.
     * </p>
     * <p>
     * If you call the <code>RegisterScalableTarget</code> API operation to update an existing scalable target,
     * Application Auto Scaling retrieves the current capacity of the resource. If it's below the minimum capacity or
     * above the maximum capacity, Application Auto Scaling adjusts the capacity of the scalable target to place it
     * within these bounds, even if you don't include the <code>MinCapacity</code> or <code>MaxCapacity</code> request
     * parameters.
     * </p>
     * </note>
     *
     * @param registerScalableTargetRequest
     * @return Result of the RegisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.RegisterScalableTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/RegisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default RegisterScalableTargetResponse registerScalableTarget(RegisterScalableTargetRequest registerScalableTargetRequest)
            throws ValidationException, LimitExceededException, ConcurrentUpdateException, InternalServiceException,
            AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Registers or updates a scalable target, which is the resource that you want to scale.
     * </p>
     * <p>
     * Scalable targets are uniquely identified by the combination of resource ID, scalable dimension, and namespace,
     * which represents some capacity dimension of the underlying service.
     * </p>
     * <p>
     * When you register a new scalable target, you must specify values for the minimum and maximum capacity. If the
     * specified resource is not active in the target service, this operation does not change the resource's current
     * capacity. Otherwise, it changes the resource's current capacity to a value that is inside of this range.
     * </p>
     * <p>
     * If you add a scaling policy, current capacity is adjustable within the specified range when scaling starts.
     * Application Auto Scaling scaling policies will not scale capacity to values that are outside of the minimum and
     * maximum range.
     * </p>
     * <p>
     * After you register a scalable target, you do not need to register it again to use other Application Auto Scaling
     * operations. To see which resources have been registered, use <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>. You can also view the scaling policies for a service namespace by using <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalableTargets.html"
     * >DescribeScalableTargets</a>. If you no longer need a scalable target, you can deregister it by using <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DeregisterScalableTarget.html"
     * >DeregisterScalableTarget</a>.
     * </p>
     * <p>
     * To update a scalable target, specify the parameters that you want to change. Include the parameters that identify
     * the scalable target: resource ID, scalable dimension, and namespace. Any parameters that you don't specify are
     * not changed by this update request.
     * </p>
     * <note>
     * <p>
     * If you call the <code>RegisterScalableTarget</code> API operation to create a scalable target, there might be a
     * brief delay until the operation achieves <a href="https://en.wikipedia.org/wiki/Eventual_consistency">eventual
     * consistency</a>. You might become aware of this brief delay if you get unexpected errors when performing
     * sequential operations. The typical strategy is to retry the request, and some Amazon Web Services SDKs include
     * automatic backoff and retry logic.
     * </p>
     * <p>
     * If you call the <code>RegisterScalableTarget</code> API operation to update an existing scalable target,
     * Application Auto Scaling retrieves the current capacity of the resource. If it's below the minimum capacity or
     * above the maximum capacity, Application Auto Scaling adjusts the capacity of the scalable target to place it
     * within these bounds, even if you don't include the <code>MinCapacity</code> or <code>MaxCapacity</code> request
     * parameters.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterScalableTargetRequest.Builder} avoiding the
     * need to create one manually via {@link RegisterScalableTargetRequest#builder()}
     * </p>
     *
     * @param registerScalableTargetRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.RegisterScalableTargetRequest.Builder}
     *        to create a request.
     * @return Result of the RegisterScalableTarget operation returned by the service.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws LimitExceededException
     *         A per-account resource limit is exceeded. For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-limits.html"
     *         >Application Auto Scaling service quotas</a>.
     * @throws ConcurrentUpdateException
     *         Concurrent updates caused an exception, for example, if you request an update to an Application Auto
     *         Scaling resource that already has a pending update.
     * @throws InternalServiceException
     *         The service encountered an internal error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.RegisterScalableTarget
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/RegisterScalableTarget"
     *      target="_top">AWS API Documentation</a>
     */
    default RegisterScalableTargetResponse registerScalableTarget(
            Consumer<RegisterScalableTargetRequest.Builder> registerScalableTargetRequest) throws ValidationException,
            LimitExceededException, ConcurrentUpdateException, InternalServiceException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return registerScalableTarget(RegisterScalableTargetRequest.builder().applyMutation(registerScalableTargetRequest)
                .build());
    }

    /**
     * <p>
     * Adds or edits tags on an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each tag consists of a tag key and a tag value, which are both case-sensitive strings. To add a tag, specify a
     * new tag key and a tag value. To edit a tag, specify an existing tag key and a new tag value.
     * </p>
     * <p>
     * You can use this operation to tag an Application Auto Scaling scalable target, but you cannot tag a scaling
     * policy or scheduled action.
     * </p>
     * <p>
     * You can also add tags to an Application Auto Scaling scalable target while creating it (
     * <code>RegisterScalableTarget</code>).
     * </p>
     * <p>
     * For general information about tags, including the format and syntax, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging your Amazon Web Services
     * resources</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * Use tags to control access to a scalable target. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/resource-tagging-support.html">Tagging
     * support for Application Auto Scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws TooManyTagsException
     *         The request contains too many tags. Try the request again with fewer tags.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            TooManyTagsException, ValidationException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or edits tags on an Application Auto Scaling scalable target.
     * </p>
     * <p>
     * Each tag consists of a tag key and a tag value, which are both case-sensitive strings. To add a tag, specify a
     * new tag key and a tag value. To edit a tag, specify an existing tag key and a new tag value.
     * </p>
     * <p>
     * You can use this operation to tag an Application Auto Scaling scalable target, but you cannot tag a scaling
     * policy or scheduled action.
     * </p>
     * <p>
     * You can also add tags to an Application Auto Scaling scalable target while creating it (
     * <code>RegisterScalableTarget</code>).
     * </p>
     * <p>
     * For general information about tags, including the format and syntax, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging your Amazon Web Services
     * resources</a> in the <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * Use tags to control access to a scalable target. For more information, see <a
     * href="https://docs.aws.amazon.com/autoscaling/application/userguide/resource-tagging-support.html">Tagging
     * support for Application Auto Scaling</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.TagResourceRequest.Builder} to create
     *        a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws TooManyTagsException
     *         The request contains too many tags. Try the request again with fewer tags.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, TooManyTagsException, ValidationException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Deletes tags from an Application Auto Scaling scalable target. To delete a tag, specify the tag key and the
     * Application Auto Scaling scalable target.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, ApplicationAutoScalingException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes tags from an Application Auto Scaling scalable target. To delete a tag, specify the tag key and the
     * Application Auto Scaling scalable target.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.applicationautoscaling.model.UntagResourceRequest.Builder} to
     *        create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource doesn't exist.
     * @throws ValidationException
     *         An exception was thrown for a validation issue. Review the available parameters for the API request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationAutoScalingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationAutoScalingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-autoscaling-2016-02-06/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            ApplicationAutoScalingException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link ApplicationAutoScalingClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ApplicationAutoScalingClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ApplicationAutoScalingClient}.
     */
    static ApplicationAutoScalingClientBuilder builder() {
        return new DefaultApplicationAutoScalingClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ApplicationAutoScalingServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
