/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationcostprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the Amazon Simple Storage Service (Amazon S3) location where usage data is read from.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceS3Location implements SdkPojo, Serializable,
        ToCopyableBuilder<SourceS3Location.Builder, SourceS3Location> {
    private static final SdkField<String> BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("bucket")
            .getter(getter(SourceS3Location::bucket)).setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucket").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("key")
            .getter(getter(SourceS3Location::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(SourceS3Location::regionAsString)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, KEY_FIELD,
            REGION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String key;

    private final String region;

    private SourceS3Location(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.key = builder.key;
        this.region = builder.region;
    }

    /**
     * <p>
     * Name of the bucket.
     * </p>
     * 
     * @return Name of the bucket.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Key of the object.
     * </p>
     * 
     * @return Key of the object.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Region of the bucket. Only required for Regions that are disabled by default. For more infomration about Regions
     * that are disabled by default, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a Region</a>
     * in the <i>AWS General Reference guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link S3BucketRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return Region of the bucket. Only required for Regions that are disabled by default. For more infomration about
     *         Regions that are disabled by default, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
     *         Region</a> in the <i>AWS General Reference guide</i>.
     * @see S3BucketRegion
     */
    public final S3BucketRegion region() {
        return S3BucketRegion.fromValue(region);
    }

    /**
     * <p>
     * Region of the bucket. Only required for Regions that are disabled by default. For more infomration about Regions
     * that are disabled by default, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a Region</a>
     * in the <i>AWS General Reference guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link S3BucketRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return Region of the bucket. Only required for Regions that are disabled by default. For more infomration about
     *         Regions that are disabled by default, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
     *         Region</a> in the <i>AWS General Reference guide</i>.
     * @see S3BucketRegion
     */
    public final String regionAsString() {
        return region;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(regionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceS3Location)) {
            return false;
        }
        SourceS3Location other = (SourceS3Location) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(key(), other.key())
                && Objects.equals(regionAsString(), other.regionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SourceS3Location").add("Bucket", bucket()).add("Key", key()).add("Region", regionAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "key":
            return Optional.ofNullable(clazz.cast(key()));
        case "region":
            return Optional.ofNullable(clazz.cast(regionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SourceS3Location, T> g) {
        return obj -> g.apply((SourceS3Location) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SourceS3Location> {
        /**
         * <p>
         * Name of the bucket.
         * </p>
         * 
         * @param bucket
         *        Name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Key of the object.
         * </p>
         * 
         * @param key
         *        Key of the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Region of the bucket. Only required for Regions that are disabled by default. For more infomration about
         * Regions that are disabled by default, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
         * Region</a> in the <i>AWS General Reference guide</i>.
         * </p>
         * 
         * @param region
         *        Region of the bucket. Only required for Regions that are disabled by default. For more infomration
         *        about Regions that are disabled by default, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
         *        Region</a> in the <i>AWS General Reference guide</i>.
         * @see S3BucketRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3BucketRegion
         */
        Builder region(String region);

        /**
         * <p>
         * Region of the bucket. Only required for Regions that are disabled by default. For more infomration about
         * Regions that are disabled by default, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
         * Region</a> in the <i>AWS General Reference guide</i>.
         * </p>
         * 
         * @param region
         *        Region of the bucket. Only required for Regions that are disabled by default. For more infomration
         *        about Regions that are disabled by default, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/rande-manage.html#rande-manage-enable"> Enabling a
         *        Region</a> in the <i>AWS General Reference guide</i>.
         * @see S3BucketRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3BucketRegion
         */
        Builder region(S3BucketRegion region);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String key;

        private String region;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceS3Location model) {
            bucket(model.bucket);
            key(model.key);
            region(model.region);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        @Override
        public final Builder region(S3BucketRegion region) {
            this.region(region == null ? null : region.toString());
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public SourceS3Location build() {
            return new SourceS3Location(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
