/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationcostprofiler;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationcostprofiler.model.AccessDeniedException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ApplicationCostProfilerException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ApplicationCostProfilerRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.DeleteReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.DeleteReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.GetReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.GetReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ImportApplicationUsageRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.ImportApplicationUsageResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.InternalServerException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.PutReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.PutReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ThrottlingException;
import software.amazon.awssdk.services.applicationcostprofiler.model.UpdateReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.UpdateReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ValidationException;
import software.amazon.awssdk.services.applicationcostprofiler.paginators.ListReportDefinitionsPublisher;
import software.amazon.awssdk.services.applicationcostprofiler.transform.DeleteReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.GetReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.ImportApplicationUsageRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.ListReportDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.PutReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.UpdateReportDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ApplicationCostProfilerAsyncClient}.
 *
 * @see ApplicationCostProfilerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationCostProfilerAsyncClient implements ApplicationCostProfilerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultApplicationCostProfilerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationCostProfilerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified report definition in AWS Application Cost Profiler. This stops the report from being
     * generated.
     * </p>
     *
     * @param deleteReportDefinitionRequest
     * @return A Java Future containing the result of the DeleteReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.DeleteReportDefinition
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/DeleteReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReportDefinitionResponse> deleteReportDefinition(
            DeleteReportDefinitionRequest deleteReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReportDefinitionRequest, DeleteReportDefinitionResponse>()
                            .withOperationName("DeleteReportDefinition")
                            .withMarshaller(new DeleteReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteReportDefinitionRequest));
            CompletableFuture<DeleteReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the definition of a report already configured in AWS Application Cost Profiler.
     * </p>
     *
     * @param getReportDefinitionRequest
     * @return A Java Future containing the result of the GetReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.GetReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/GetReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReportDefinitionResponse> getReportDefinition(
            GetReportDefinitionRequest getReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReportDefinitionRequest, GetReportDefinitionResponse>()
                            .withOperationName("GetReportDefinition")
                            .withMarshaller(new GetReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getReportDefinitionRequest));
            CompletableFuture<GetReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Ingests application usage data from Amazon Simple Storage Service (Amazon S3).
     * </p>
     * <p>
     * The data must already exist in the S3 location. As part of the action, AWS Application Cost Profiler copies the
     * object from your S3 bucket to an S3 bucket owned by Amazon for processing asynchronously.
     * </p>
     *
     * @param importApplicationUsageRequest
     * @return A Java Future containing the result of the ImportApplicationUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.ImportApplicationUsage
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/ImportApplicationUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportApplicationUsageResponse> importApplicationUsage(
            ImportApplicationUsageRequest importApplicationUsageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importApplicationUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportApplicationUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportApplicationUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportApplicationUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportApplicationUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportApplicationUsageRequest, ImportApplicationUsageResponse>()
                            .withOperationName("ImportApplicationUsage")
                            .withMarshaller(new ImportApplicationUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importApplicationUsageRequest));
            CompletableFuture<ImportApplicationUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all reports and their configurations for your AWS account.
     * </p>
     * <p>
     * The maximum number of reports is one.
     * </p>
     *
     * @param listReportDefinitionsRequest
     * @return A Java Future containing the result of the ListReportDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.ListReportDefinitions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/ListReportDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReportDefinitionsResponse> listReportDefinitions(
            ListReportDefinitionsRequest listReportDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReportDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReportDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReportDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReportDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReportDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReportDefinitionsRequest, ListReportDefinitionsResponse>()
                            .withOperationName("ListReportDefinitions")
                            .withMarshaller(new ListReportDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listReportDefinitionsRequest));
            CompletableFuture<ListReportDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all reports and their configurations for your AWS account.
     * </p>
     * <p>
     * The maximum number of reports is one.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listReportDefinitions(software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationcostprofiler.paginators.ListReportDefinitionsPublisher publisher = client.listReportDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationcostprofiler.paginators.ListReportDefinitionsPublisher publisher = client.listReportDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReportDefinitions(software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReportDefinitionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.ListReportDefinitions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/ListReportDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListReportDefinitionsPublisher listReportDefinitionsPaginator(ListReportDefinitionsRequest listReportDefinitionsRequest) {
        return new ListReportDefinitionsPublisher(this, applyPaginatorUserAgent(listReportDefinitionsRequest));
    }

    /**
     * <p>
     * Creates the report definition for a report in Application Cost Profiler.
     * </p>
     *
     * @param putReportDefinitionRequest
     * @return A Java Future containing the result of the PutReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds one or more of the service quotas.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.PutReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/PutReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutReportDefinitionResponse> putReportDefinition(
            PutReportDefinitionRequest putReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutReportDefinitionRequest, PutReportDefinitionResponse>()
                            .withOperationName("PutReportDefinition")
                            .withMarshaller(new PutReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putReportDefinitionRequest));
            CompletableFuture<PutReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates existing report in AWS Application Cost Profiler.
     * </p>
     *
     * @param updateReportDefinitionRequest
     * @return A Java Future containing the result of the UpdateReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The calls to AWS Application Cost Profiler API are throttled. The request was
     *         denied.</li>
     *         <li>ValidationException The input fails to satisfy the constraints for the API.</li>
     *         <li>AccessDeniedException You do not have permission to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationCostProfilerException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationCostProfilerAsyncClient.UpdateReportDefinition
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/UpdateReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateReportDefinitionResponse> updateReportDefinition(
            UpdateReportDefinitionRequest updateReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateReportDefinitionRequest, UpdateReportDefinitionResponse>()
                            .withOperationName("UpdateReportDefinition")
                            .withMarshaller(new UpdateReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateReportDefinitionRequest));
            CompletableFuture<UpdateReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationCostProfilerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ApplicationCostProfilerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
