/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationcostprofiler;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationcostprofiler.internal.ApplicationCostProfilerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.applicationcostprofiler.model.AccessDeniedException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ApplicationCostProfilerException;
import software.amazon.awssdk.services.applicationcostprofiler.model.DeleteReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.DeleteReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.GetReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.GetReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ImportApplicationUsageRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.ImportApplicationUsageResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.InternalServerException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.ListReportDefinitionsResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.PutReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.PutReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.applicationcostprofiler.model.ThrottlingException;
import software.amazon.awssdk.services.applicationcostprofiler.model.UpdateReportDefinitionRequest;
import software.amazon.awssdk.services.applicationcostprofiler.model.UpdateReportDefinitionResponse;
import software.amazon.awssdk.services.applicationcostprofiler.model.ValidationException;
import software.amazon.awssdk.services.applicationcostprofiler.transform.DeleteReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.GetReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.ImportApplicationUsageRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.ListReportDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.PutReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.applicationcostprofiler.transform.UpdateReportDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ApplicationCostProfilerClient}.
 *
 * @see ApplicationCostProfilerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationCostProfilerClient implements ApplicationCostProfilerClient {
    private static final Logger log = Logger.loggerFor(DefaultApplicationCostProfilerClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ApplicationCostProfilerServiceClientConfiguration serviceClientConfiguration;

    protected DefaultApplicationCostProfilerClient(ApplicationCostProfilerServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified report definition in AWS Application Cost Profiler. This stops the report from being
     * generated.
     * </p>
     *
     * @param deleteReportDefinitionRequest
     * @return Result of the DeleteReportDefinition operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.DeleteReportDefinition
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/DeleteReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReportDefinitionResponse deleteReportDefinition(DeleteReportDefinitionRequest deleteReportDefinitionRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteReportDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReportDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteReportDefinitionRequest, DeleteReportDefinitionResponse>()
                            .withOperationName("DeleteReportDefinition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deleteReportDefinitionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteReportDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the definition of a report already configured in AWS Application Cost Profiler.
     * </p>
     *
     * @param getReportDefinitionRequest
     * @return Result of the GetReportDefinition operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.GetReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/GetReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReportDefinitionResponse getReportDefinition(GetReportDefinitionRequest getReportDefinitionRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getReportDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReportDefinition");

            return clientHandler.execute(new ClientExecutionParams<GetReportDefinitionRequest, GetReportDefinitionResponse>()
                    .withOperationName("GetReportDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getReportDefinitionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetReportDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Ingests application usage data from Amazon Simple Storage Service (Amazon S3).
     * </p>
     * <p>
     * The data must already exist in the S3 location. As part of the action, AWS Application Cost Profiler copies the
     * object from your S3 bucket to an S3 bucket owned by Amazon for processing asynchronously.
     * </p>
     *
     * @param importApplicationUsageRequest
     * @return Result of the ImportApplicationUsage operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.ImportApplicationUsage
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/ImportApplicationUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportApplicationUsageResponse importApplicationUsage(ImportApplicationUsageRequest importApplicationUsageRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportApplicationUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportApplicationUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importApplicationUsageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importApplicationUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportApplicationUsage");

            return clientHandler
                    .execute(new ClientExecutionParams<ImportApplicationUsageRequest, ImportApplicationUsageResponse>()
                            .withOperationName("ImportApplicationUsage").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(importApplicationUsageRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ImportApplicationUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a list of all reports and their configurations for your AWS account.
     * </p>
     * <p>
     * The maximum number of reports is one.
     * </p>
     *
     * @param listReportDefinitionsRequest
     * @return Result of the ListReportDefinitions operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.ListReportDefinitions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/ListReportDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReportDefinitionsResponse listReportDefinitions(ListReportDefinitionsRequest listReportDefinitionsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReportDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReportDefinitionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReportDefinitionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReportDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReportDefinitions");

            return clientHandler.execute(new ClientExecutionParams<ListReportDefinitionsRequest, ListReportDefinitionsResponse>()
                    .withOperationName("ListReportDefinitions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listReportDefinitionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReportDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates the report definition for a report in Application Cost Profiler.
     * </p>
     *
     * @param putReportDefinitionRequest
     * @return Result of the PutReportDefinition operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws ServiceQuotaExceededException
     *         Your request exceeds one or more of the service quotas.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.PutReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/PutReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutReportDefinitionResponse putReportDefinition(PutReportDefinitionRequest putReportDefinitionRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putReportDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutReportDefinition");

            return clientHandler.execute(new ClientExecutionParams<PutReportDefinitionRequest, PutReportDefinitionResponse>()
                    .withOperationName("PutReportDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putReportDefinitionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutReportDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates existing report in AWS Application Cost Profiler.
     * </p>
     *
     * @param updateReportDefinitionRequest
     * @return Result of the UpdateReportDefinition operation returned by the service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The calls to AWS Application Cost Profiler API are throttled. The request was denied.
     * @throws ValidationException
     *         The input fails to satisfy the constraints for the API.
     * @throws AccessDeniedException
     *         You do not have permission to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationCostProfilerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationCostProfilerClient.UpdateReportDefinition
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/AWSApplicationCostProfiler-2020-09-10/UpdateReportDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReportDefinitionResponse updateReportDefinition(UpdateReportDefinitionRequest updateReportDefinitionRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ApplicationCostProfilerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateReportDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateReportDefinitionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ApplicationCostProfiler");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReportDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateReportDefinitionRequest, UpdateReportDefinitionResponse>()
                            .withOperationName("UpdateReportDefinition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(updateReportDefinitionRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateReportDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        ApplicationCostProfilerServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = ApplicationCostProfilerServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationCostProfilerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    @Override
    public final ApplicationCostProfilerServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
