/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationdiscovery;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.applicationdiscovery.model.ApplicationDiscoveryException;
import software.amazon.awssdk.services.applicationdiscovery.model.ApplicationDiscoveryRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.AssociateConfigurationItemsToApplicationRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.AssociateConfigurationItemsToApplicationResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.AuthorizationErrorException;
import software.amazon.awssdk.services.applicationdiscovery.model.ConflictErrorException;
import software.amazon.awssdk.services.applicationdiscovery.model.CreateApplicationRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.CreateApplicationResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.CreateTagsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.CreateTagsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DeleteApplicationsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DeleteApplicationsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DeleteTagsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DeleteTagsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeAgentsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeAgentsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeConfigurationsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeConfigurationsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeExportTasksRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeExportTasksResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeTagsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeTagsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.DisassociateConfigurationItemsFromApplicationRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DisassociateConfigurationItemsFromApplicationResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.GetDiscoverySummaryRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.GetDiscoverySummaryResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.InvalidParameterException;
import software.amazon.awssdk.services.applicationdiscovery.model.InvalidParameterValueException;
import software.amazon.awssdk.services.applicationdiscovery.model.ListConfigurationsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.ListConfigurationsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.ListServerNeighborsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.ListServerNeighborsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.OperationNotPermittedException;
import software.amazon.awssdk.services.applicationdiscovery.model.ResourceInUseException;
import software.amazon.awssdk.services.applicationdiscovery.model.ResourceNotFoundException;
import software.amazon.awssdk.services.applicationdiscovery.model.ServerInternalErrorException;
import software.amazon.awssdk.services.applicationdiscovery.model.StartContinuousExportRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.StartContinuousExportResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.StartDataCollectionByAgentIdsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.StartDataCollectionByAgentIdsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.StartExportTaskRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.StartExportTaskResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.StopContinuousExportRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.StopContinuousExportResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.StopDataCollectionByAgentIdsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.StopDataCollectionByAgentIdsResponse;
import software.amazon.awssdk.services.applicationdiscovery.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable;
import software.amazon.awssdk.services.applicationdiscovery.transform.AssociateConfigurationItemsToApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DeleteApplicationsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DescribeAgentsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DescribeConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DescribeContinuousExportsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DescribeExportTasksRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.DisassociateConfigurationItemsFromApplicationRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.GetDiscoverySummaryRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.ListConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.ListServerNeighborsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.StartContinuousExportRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.StartDataCollectionByAgentIdsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.StartExportTaskRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.StopContinuousExportRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.StopDataCollectionByAgentIdsRequestMarshaller;
import software.amazon.awssdk.services.applicationdiscovery.transform.UpdateApplicationRequestMarshaller;

/**
 * Internal implementation of {@link ApplicationDiscoveryClient}.
 *
 * @see ApplicationDiscoveryClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationDiscoveryClient implements ApplicationDiscoveryClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationDiscoveryClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates one or more configuration items with an application.
     * </p>
     *
     * @param associateConfigurationItemsToApplicationRequest
     * @return Result of the AssociateConfigurationItemsToApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.AssociateConfigurationItemsToApplication
     */
    @Override
    public AssociateConfigurationItemsToApplicationResponse associateConfigurationItemsToApplication(
            AssociateConfigurationItemsToApplicationRequest associateConfigurationItemsToApplicationRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateConfigurationItemsToApplicationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, AssociateConfigurationItemsToApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<AssociateConfigurationItemsToApplicationRequest, AssociateConfigurationItemsToApplicationResponse>()
                        .withOperationName("AssociateConfigurationItemsToApplication").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(associateConfigurationItemsToApplicationRequest)
                        .withMarshaller(new AssociateConfigurationItemsToApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates an application with the given name and description.
     * </p>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.CreateApplication
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                .withOperationName("CreateApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationRequest)
                .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates one or more tags for configuration items. Tags are metadata that help you categorize IT assets. This API
     * accepts a list of multiple configuration items.
     * </p>
     *
     * @param createTagsRequest
     * @return Result of the CreateTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.CreateTags
     */
    @Override
    public CreateTagsResponse createTags(CreateTagsRequest createTagsRequest) throws AuthorizationErrorException,
            ResourceNotFoundException, InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException,
            AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>()
                .withOperationName("CreateTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createTagsRequest)
                .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a list of applications and their associations with configuration items.
     * </p>
     *
     * @param deleteApplicationsRequest
     * @return Result of the DeleteApplications operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DeleteApplications
     */
    @Override
    public DeleteApplicationsResponse deleteApplications(DeleteApplicationsRequest deleteApplicationsRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteApplicationsRequest, DeleteApplicationsResponse>()
                .withOperationName("DeleteApplications").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteApplicationsRequest)
                .withMarshaller(new DeleteApplicationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes the association between configuration items and one or more tags. This API accepts a list of multiple
     * configuration items.
     * </p>
     *
     * @param deleteTagsRequest
     * @return Result of the DeleteTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DeleteTags
     */
    @Override
    public DeleteTagsResponse deleteTags(DeleteTagsRequest deleteTagsRequest) throws AuthorizationErrorException,
            ResourceNotFoundException, InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException,
            AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>()
                .withOperationName("DeleteTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteTagsRequest)
                .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists agents or connectors as specified by ID or other filters. All agents/connectors associated with your user
     * account can be listed if you call <code>DescribeAgents</code> as is without passing any parameters.
     * </p>
     *
     * @param describeAgentsRequest
     * @return Result of the DescribeAgents operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeAgents
     */
    @Override
    public DescribeAgentsResponse describeAgents(DescribeAgentsRequest describeAgentsRequest) throws AuthorizationErrorException,
            InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException, AwsServiceException,
            SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAgentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeAgentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeAgentsRequest, DescribeAgentsResponse>()
                .withOperationName("DescribeAgents").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAgentsRequest)
                .withMarshaller(new DescribeAgentsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves attributes for a list of configuration item IDs.
     * </p>
     * <note>
     * <p>
     * All of the supplied IDs must be for the same asset type from one of the follwoing:
     * </p>
     * <ul>
     * <li>
     * <p>
     * server
     * </p>
     * </li>
     * <li>
     * <p>
     * application
     * </p>
     * </li>
     * <li>
     * <p>
     * process
     * </p>
     * </li>
     * <li>
     * <p>
     * connection
     * </p>
     * </li>
     * </ul>
     * <p>
     * Output fields are specific to the asset type specified. For example, the output for a <i>server</i> configuration
     * item includes a list of attributes about the server, such as host name, operating system, number of network
     * cards, etc.
     * </p>
     * <p>
     * For a complete list of outputs for each asset type, see <a href=
     * "http://docs.aws.amazon.com/application-discovery/latest/APIReference/discovery-api-queries.html#DescribeConfigurations"
     * >Using the DescribeConfigurations Action</a>.
     * </p>
     * </note>
     *
     * @param describeConfigurationsRequest
     * @return Result of the DescribeConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeConfigurations
     */
    @Override
    public DescribeConfigurationsResponse describeConfigurations(DescribeConfigurationsRequest describeConfigurationsRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeConfigurationsRequest, DescribeConfigurationsResponse>()
                .withOperationName("DescribeConfigurations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeConfigurationsRequest)
                .withMarshaller(new DescribeConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists exports as specified by ID. All continuous exports associated with your user account can be listed if you
     * call <code>DescribeContinuousExports</code> as is without passing any parameters.
     * </p>
     *
     * @param describeContinuousExportsRequest
     * @return Result of the DescribeContinuousExports operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeContinuousExports
     */
    @Override
    public DescribeContinuousExportsResponse describeContinuousExports(
            DescribeContinuousExportsRequest describeContinuousExportsRequest) throws AuthorizationErrorException,
            InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException,
            OperationNotPermittedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContinuousExportsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeContinuousExportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeContinuousExportsRequest, DescribeContinuousExportsResponse>()
                        .withOperationName("DescribeContinuousExports").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeContinuousExportsRequest)
                        .withMarshaller(new DescribeContinuousExportsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists exports as specified by ID. All continuous exports associated with your user account can be listed if you
     * call <code>DescribeContinuousExports</code> as is without passing any parameters.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeContinuousExports(software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client.describeContinuousExportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client
     *             .describeContinuousExportsPaginator(request);
     *     for (software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client.describeContinuousExportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeContinuousExports(software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeContinuousExportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeContinuousExports
     */
    @Override
    public DescribeContinuousExportsIterable describeContinuousExportsPaginator(
            DescribeContinuousExportsRequest describeContinuousExportsRequest) throws AuthorizationErrorException,
            InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException,
            OperationNotPermittedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            ApplicationDiscoveryException {
        return new DescribeContinuousExportsIterable(this, applyPaginatorUserAgent(describeContinuousExportsRequest));
    }

    /**
     * <p>
     * Retrieve status of one or more export tasks. You can retrieve the status of up to 100 export tasks.
     * </p>
     *
     * @param describeExportTasksRequest
     * @return Result of the DescribeExportTasks operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeExportTasks
     */
    @Override
    public DescribeExportTasksResponse describeExportTasks(DescribeExportTasksRequest describeExportTasksRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeExportTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeExportTasksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeExportTasksRequest, DescribeExportTasksResponse>()
                .withOperationName("DescribeExportTasks").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeExportTasksRequest)
                .withMarshaller(new DescribeExportTasksRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a list of configuration items that have tags as specified by the key-value pairs, name and value,
     * passed to the optional parameter <code>filters</code>.
     * </p>
     * <p>
     * There are three valid tag filter names:
     * </p>
     * <ul>
     * <li>
     * <p>
     * tagKey
     * </p>
     * </li>
     * <li>
     * <p>
     * tagValue
     * </p>
     * </li>
     * <li>
     * <p>
     * configurationId
     * </p>
     * </li>
     * </ul>
     * <p>
     * Also, all configuration items associated with your user account that have tags can be listed if you call
     * <code>DescribeTags</code> as is without passing any parameters.
     * </p>
     *
     * @param describeTagsRequest
     * @return Result of the DescribeTags operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DescribeTags
     */
    @Override
    public DescribeTagsResponse describeTags(DescribeTagsRequest describeTagsRequest) throws AuthorizationErrorException,
            ResourceNotFoundException, InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException,
            AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                .withOperationName("DescribeTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeTagsRequest)
                .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disassociates one or more configuration items from an application.
     * </p>
     *
     * @param disassociateConfigurationItemsFromApplicationRequest
     * @return Result of the DisassociateConfigurationItemsFromApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.DisassociateConfigurationItemsFromApplication
     */
    @Override
    public DisassociateConfigurationItemsFromApplicationResponse disassociateConfigurationItemsFromApplication(
            DisassociateConfigurationItemsFromApplicationRequest disassociateConfigurationItemsFromApplicationRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateConfigurationItemsFromApplicationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateConfigurationItemsFromApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DisassociateConfigurationItemsFromApplicationRequest, DisassociateConfigurationItemsFromApplicationResponse>()
                        .withOperationName("DisassociateConfigurationItemsFromApplication").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler)
                        .withInput(disassociateConfigurationItemsFromApplicationRequest)
                        .withMarshaller(new DisassociateConfigurationItemsFromApplicationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a short summary of discovered assets.
     * </p>
     * <p>
     * This API operation takes no request parameters and is called as is at the command prompt as shown in the example.
     * </p>
     *
     * @param getDiscoverySummaryRequest
     * @return Result of the GetDiscoverySummary operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.GetDiscoverySummary
     */
    @Override
    public GetDiscoverySummaryResponse getDiscoverySummary(GetDiscoverySummaryRequest getDiscoverySummaryRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDiscoverySummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDiscoverySummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDiscoverySummaryRequest, GetDiscoverySummaryResponse>()
                .withOperationName("GetDiscoverySummary").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDiscoverySummaryRequest)
                .withMarshaller(new GetDiscoverySummaryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a list of configuration items as specified by the value passed to the required paramater
     * <code>configurationType</code>. Optional filtering may be applied to refine search results.
     * </p>
     *
     * @param listConfigurationsRequest
     * @return Result of the ListConfigurations operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.ListConfigurations
     */
    @Override
    public ListConfigurationsResponse listConfigurations(ListConfigurationsRequest listConfigurationsRequest)
            throws AuthorizationErrorException, ResourceNotFoundException, InvalidParameterException,
            InvalidParameterValueException, ServerInternalErrorException, AwsServiceException, SdkClientException,
            ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListConfigurationsRequest, ListConfigurationsResponse>()
                .withOperationName("ListConfigurations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listConfigurationsRequest)
                .withMarshaller(new ListConfigurationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a list of servers that are one network hop away from a specified server.
     * </p>
     *
     * @param listServerNeighborsRequest
     * @return Result of the ListServerNeighbors operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.ListServerNeighbors
     */
    @Override
    public ListServerNeighborsResponse listServerNeighbors(ListServerNeighborsRequest listServerNeighborsRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServerNeighborsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServerNeighborsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListServerNeighborsRequest, ListServerNeighborsResponse>()
                .withOperationName("ListServerNeighbors").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listServerNeighborsRequest)
                .withMarshaller(new ListServerNeighborsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Start the continuous flow of agent's discovered data into Amazon Athena.
     * </p>
     *
     * @param startContinuousExportRequest
     * @return Result of the StartContinuousExport operation returned by the service.
     * @throws ConflictErrorException
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceInUseException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.StartContinuousExport
     */
    @Override
    public StartContinuousExportResponse startContinuousExport(StartContinuousExportRequest startContinuousExportRequest)
            throws ConflictErrorException, AuthorizationErrorException, InvalidParameterException,
            InvalidParameterValueException, ServerInternalErrorException, OperationNotPermittedException, ResourceInUseException,
            AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartContinuousExportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartContinuousExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartContinuousExportRequest, StartContinuousExportResponse>()
                .withOperationName("StartContinuousExport").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startContinuousExportRequest)
                .withMarshaller(new StartContinuousExportRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Instructs the specified agents or connectors to start collecting data.
     * </p>
     *
     * @param startDataCollectionByAgentIdsRequest
     * @return Result of the StartDataCollectionByAgentIds operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.StartDataCollectionByAgentIds
     */
    @Override
    public StartDataCollectionByAgentIdsResponse startDataCollectionByAgentIds(
            StartDataCollectionByAgentIdsRequest startDataCollectionByAgentIdsRequest) throws AuthorizationErrorException,
            InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException, AwsServiceException,
            SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDataCollectionByAgentIdsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDataCollectionByAgentIdsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StartDataCollectionByAgentIdsRequest, StartDataCollectionByAgentIdsResponse>()
                        .withOperationName("StartDataCollectionByAgentIds").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(startDataCollectionByAgentIdsRequest)
                        .withMarshaller(new StartDataCollectionByAgentIdsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Begins the export of discovered data to an S3 bucket.
     * </p>
     * <p>
     * If you specify <code>agentIds</code> in a filter, the task exports up to 72 hours of detailed data collected by
     * the identified Application Discovery Agent, including network, process, and performance details. A time range for
     * exported agent data may be set by using <code>startTime</code> and <code>endTime</code>. Export of detailed agent
     * data is limited to five concurrently running exports.
     * </p>
     * <p>
     * If you do not include an <code>agentIds</code> filter, summary data is exported that includes both AWS Agentless
     * Discovery Connector data and summary data from AWS Discovery Agents. Export of summary data is limited to two
     * exports per day.
     * </p>
     *
     * @param startExportTaskRequest
     * @return Result of the StartExportTask operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.StartExportTask
     */
    @Override
    public StartExportTaskResponse startExportTask(StartExportTaskRequest startExportTaskRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, OperationNotPermittedException, AwsServiceException, SdkClientException,
            ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartExportTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartExportTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StartExportTaskRequest, StartExportTaskResponse>()
                .withOperationName("StartExportTask").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(startExportTaskRequest)
                .withMarshaller(new StartExportTaskRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Stop the continuous flow of agent's discovered data into Amazon Athena.
     * </p>
     *
     * @param stopContinuousExportRequest
     * @return Result of the StopContinuousExport operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws OperationNotPermittedException
     *         This operation is not permitted.
     * @throws ResourceNotFoundException
     *         The specified configuration ID was not located. Verify the configuration ID and try again.
     * @throws ResourceInUseException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.StopContinuousExport
     */
    @Override
    public StopContinuousExportResponse stopContinuousExport(StopContinuousExportRequest stopContinuousExportRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, OperationNotPermittedException, ResourceNotFoundException, ResourceInUseException,
            AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopContinuousExportResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopContinuousExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopContinuousExportRequest, StopContinuousExportResponse>()
                .withOperationName("StopContinuousExport").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopContinuousExportRequest)
                .withMarshaller(new StopContinuousExportRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Instructs the specified agents or connectors to stop collecting data.
     * </p>
     *
     * @param stopDataCollectionByAgentIdsRequest
     * @return Result of the StopDataCollectionByAgentIds operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.StopDataCollectionByAgentIds
     */
    @Override
    public StopDataCollectionByAgentIdsResponse stopDataCollectionByAgentIds(
            StopDataCollectionByAgentIdsRequest stopDataCollectionByAgentIdsRequest) throws AuthorizationErrorException,
            InvalidParameterException, InvalidParameterValueException, ServerInternalErrorException, AwsServiceException,
            SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopDataCollectionByAgentIdsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopDataCollectionByAgentIdsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StopDataCollectionByAgentIdsRequest, StopDataCollectionByAgentIdsResponse>()
                        .withOperationName("StopDataCollectionByAgentIds").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(stopDataCollectionByAgentIdsRequest)
                        .withMarshaller(new StopDataCollectionByAgentIdsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates metadata about an application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws AuthorizationErrorException
     *         The AWS user account does not have permission to perform the action. Check the IAM policy associated with
     *         this account.
     * @throws InvalidParameterException
     *         One or more parameters are not valid. Verify the parameters and try again.
     * @throws InvalidParameterValueException
     *         The value of one or more parameters are either invalid or out of range. Verify the parameter values and
     *         try again.
     * @throws ServerInternalErrorException
     *         The server experienced an internal error. Try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationDiscoveryException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationDiscoveryClient.UpdateApplication
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws AuthorizationErrorException, InvalidParameterException, InvalidParameterValueException,
            ServerInternalErrorException, AwsServiceException, SdkClientException, ApplicationDiscoveryException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                .withOperationName("UpdateApplication").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationRequest)
                .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationDiscoveryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalErrorException")
                                .exceptionBuilderSupplier(ServerInternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictErrorException")
                                .exceptionBuilderSupplier(ConflictErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationErrorException")
                                .exceptionBuilderSupplier(AuthorizationErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ApplicationDiscoveryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
