/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationdiscovery.paginators;

import java.util.Collections;
import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.applicationdiscovery.ApplicationDiscoveryClient;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest;
import software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.applicationdiscovery.ApplicationDiscoveryClient#describeContinuousExportsPaginator(software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsResponse} that can be used
 * to iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client.describeContinuousExportsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client
 *             .describeContinuousExportsPaginator(request);
 *     for (software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.applicationdiscovery.paginators.DescribeContinuousExportsIterable responses = client.describeContinuousExportsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeContinuousExports(software.amazon.awssdk.services.applicationdiscovery.model.DescribeContinuousExportsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeContinuousExportsIterable implements SdkIterable<DescribeContinuousExportsResponse> {
    private final ApplicationDiscoveryClient client;

    private final DescribeContinuousExportsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public DescribeContinuousExportsIterable(ApplicationDiscoveryClient client, DescribeContinuousExportsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new DescribeContinuousExportsResponseFetcher();
    }

    @Override
    public Iterator<DescribeContinuousExportsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeContinuousExportsIterable} that can be used to retrieve
     * the consecutive pages that follows the input page.
     * </p>
     */
    private final DescribeContinuousExportsIterable resume(DescribeContinuousExportsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeContinuousExportsIterable(client, firstRequest.toBuilder()
                    .nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new DescribeContinuousExportsIterable(client, firstRequest) {
            @Override
            public Iterator<DescribeContinuousExportsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class DescribeContinuousExportsResponseFetcher implements SyncPageFetcher<DescribeContinuousExportsResponse> {
        @Override
        public boolean hasNextPage(DescribeContinuousExportsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public DescribeContinuousExportsResponse nextPage(DescribeContinuousExportsResponse previousPage) {
            if (previousPage == null) {
                return client.describeContinuousExports(firstRequest);
            }
            return client.describeContinuousExports(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
