/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.applicationsignals.internal.ApplicationSignalsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.applicationsignals.model.AccessDeniedException;
import software.amazon.awssdk.services.applicationsignals.model.ApplicationSignalsException;
import software.amazon.awssdk.services.applicationsignals.model.BatchGetServiceLevelObjectiveBudgetReportRequest;
import software.amazon.awssdk.services.applicationsignals.model.BatchGetServiceLevelObjectiveBudgetReportResponse;
import software.amazon.awssdk.services.applicationsignals.model.ConflictException;
import software.amazon.awssdk.services.applicationsignals.model.CreateServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.CreateServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.DeleteServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.DeleteServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceRequest;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependenciesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependenciesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependentsRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependentsResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceLevelObjectivesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceLevelObjectivesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceOperationsRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceOperationsResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServicesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServicesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ResourceNotFoundException;
import software.amazon.awssdk.services.applicationsignals.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.applicationsignals.model.StartDiscoveryRequest;
import software.amazon.awssdk.services.applicationsignals.model.StartDiscoveryResponse;
import software.amazon.awssdk.services.applicationsignals.model.TagResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.TagResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ThrottlingException;
import software.amazon.awssdk.services.applicationsignals.model.UntagResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.UntagResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.UpdateServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.UpdateServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.ValidationException;
import software.amazon.awssdk.services.applicationsignals.transform.BatchGetServiceLevelObjectiveBudgetReportRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.CreateServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.DeleteServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.GetServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceDependenciesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceDependentsRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceLevelObjectivesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceOperationsRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.StartDiscoveryRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.UpdateServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ApplicationSignalsClient}.
 *
 * @see ApplicationSignalsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationSignalsClient implements ApplicationSignalsClient {
    private static final Logger log = Logger.loggerFor(DefaultApplicationSignalsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationSignalsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Use this operation to retrieve one or more <i>service level objective (SLO) budget reports</i>.
     * </p>
     * <p>
     * An <i>error budget</i> is the amount of time or requests in an unhealthy state that your service can accumulate
     * during an interval before your overall SLO budget health is breached and the SLO is considered to be unmet. For
     * example, an SLO with a threshold of 99.95% and a monthly interval translates to an error budget of 21.9 minutes
     * of downtime in a 30-day month.
     * </p>
     * <p>
     * Budget reports include a health indicator, the attainment value, and remaining budget.
     * </p>
     * <p>
     * For more information about SLO error budgets, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-ServiceLevelObjectives.html#CloudWatch-ServiceLevelObjectives-concepts"
     * > SLO concepts</a>.
     * </p>
     *
     * @param batchGetServiceLevelObjectiveBudgetReportRequest
     * @return Result of the BatchGetServiceLevelObjectiveBudgetReport operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.BatchGetServiceLevelObjectiveBudgetReport
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/BatchGetServiceLevelObjectiveBudgetReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchGetServiceLevelObjectiveBudgetReportResponse batchGetServiceLevelObjectiveBudgetReport(
            BatchGetServiceLevelObjectiveBudgetReportRequest batchGetServiceLevelObjectiveBudgetReportRequest)
            throws ValidationException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetServiceLevelObjectiveBudgetReportResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, BatchGetServiceLevelObjectiveBudgetReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchGetServiceLevelObjectiveBudgetReportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetServiceLevelObjectiveBudgetReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetServiceLevelObjectiveBudgetReport");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchGetServiceLevelObjectiveBudgetReportRequest, BatchGetServiceLevelObjectiveBudgetReportResponse>()
                            .withOperationName("BatchGetServiceLevelObjectiveBudgetReport")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(batchGetServiceLevelObjectiveBudgetReportRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchGetServiceLevelObjectiveBudgetReportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a service level objective (SLO), which can help you ensure that your critical business operations are
     * meeting customer expectations. Use SLOs to set and track specific target levels for the reliability and
     * availability of your applications and services. SLOs use service level indicators (SLIs) to calculate whether the
     * application is performing at the level that you want.
     * </p>
     * <p>
     * Create an SLO to set a target for a service or operation’s availability or latency. CloudWatch measures this
     * target frequently you can find whether it has been breached.
     * </p>
     * <p>
     * The target performance quality that is defined for an SLO is the <i>attainment goal</i>.
     * </p>
     * <p>
     * You can set SLO targets for your applications that are discovered by Application Signals, using critical metrics
     * such as latency and availability. You can also set SLOs against any CloudWatch metric or math expression that
     * produces a time series.
     * </p>
     * <p>
     * When you create an SLO, you specify whether it is a <i>period-based SLO</i> or a <i>request-based SLO</i>. Each
     * type of SLO has a different way of evaluating your application's performance against its attainment goal.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <i>period-based SLO</i> uses defined <i>periods</i> of time within a specified total time interval. For each
     * period of time, Application Signals determines whether the application met its goal. The attainment rate is
     * calculated as the <code>number of good periods/number of total periods</code>.
     * </p>
     * <p>
     * For example, for a period-based SLO, meeting an attainment goal of 99.9% means that within your interval, your
     * application must meet its performance goal during at least 99.9% of the time periods.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <i>request-based SLO</i> doesn't use pre-defined periods of time. Instead, the SLO measures
     * <code>number of good requests/number of total requests</code> during the interval. At any time, you can find the
     * ratio of good requests to total requests for the interval up to the time stamp that you specify, and measure that
     * ratio against the goal set in your SLO.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you have created an SLO, you can retrieve error budget reports for it. An <i>error budget</i> is the amount
     * of time or amount of requests that your application can be non-compliant with the SLO's goal, and still have your
     * application meet the goal.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a period-based SLO, the error budget starts at a number defined by the highest number of periods that can
     * fail to meet the threshold, while still meeting the overall goal. The <i>remaining error budget</i> decreases
     * with every failed period that is recorded. The error budget within one interval can never increase.
     * </p>
     * <p>
     * For example, an SLO with a threshold that 99.95% of requests must be completed under 2000ms every month
     * translates to an error budget of 21.9 minutes of downtime per month.
     * </p>
     * </li>
     * <li>
     * <p>
     * For a request-based SLO, the remaining error budget is dynamic and can increase or decrease, depending on the
     * ratio of good requests to total requests.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about SLOs, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-ServiceLevelObjectives.html">
     * Service level objectives (SLOs)</a>.
     * </p>
     * <p>
     * When you perform a <code>CreateServiceLevelObjective</code> operation, Application Signals creates the
     * <i>AWSServiceRoleForCloudWatchApplicationSignals</i> service-linked role, if it doesn't already exist in your
     * account. This service- linked role has the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>xray:GetServiceGraph</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:StartQuery</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:GetQueryResults</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:GetMetricData</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:ListMetrics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>autoscaling:DescribeAutoScalingGroups</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createServiceLevelObjectiveRequest
     * @return Result of the CreateServiceLevelObjective operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws ConflictException
     *         This operation attempted to create a resource that already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.CreateServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/CreateServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateServiceLevelObjectiveResponse createServiceLevelObjective(
            CreateServiceLevelObjectiveRequest createServiceLevelObjectiveRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, ServiceQuotaExceededException, ConflictException, AwsServiceException,
            SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateServiceLevelObjectiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateServiceLevelObjective");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateServiceLevelObjectiveRequest, CreateServiceLevelObjectiveResponse>()
                            .withOperationName("CreateServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createServiceLevelObjectiveRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateServiceLevelObjectiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified service level objective.
     * </p>
     *
     * @param deleteServiceLevelObjectiveRequest
     * @return Result of the DeleteServiceLevelObjective operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.DeleteServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/DeleteServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteServiceLevelObjectiveResponse deleteServiceLevelObjective(
            DeleteServiceLevelObjectiveRequest deleteServiceLevelObjectiveRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteServiceLevelObjectiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServiceLevelObjective");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceLevelObjectiveRequest, DeleteServiceLevelObjectiveResponse>()
                            .withOperationName("DeleteServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteServiceLevelObjectiveRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteServiceLevelObjectiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a service discovered by Application Signals.
     * </p>
     *
     * @param getServiceRequest
     * @return Result of the GetService operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.GetService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/GetService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceResponse getService(GetServiceRequest getServiceRequest) throws ValidationException, ThrottlingException,
            AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetServiceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetService");

            return clientHandler.execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>()
                    .withOperationName("GetService").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getServiceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetServiceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about one SLO created in the account.
     * </p>
     *
     * @param getServiceLevelObjectiveRequest
     * @return Result of the GetServiceLevelObjective operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.GetServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/GetServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetServiceLevelObjectiveResponse getServiceLevelObjective(
            GetServiceLevelObjectiveRequest getServiceLevelObjectiveRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetServiceLevelObjectiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceLevelObjective");

            return clientHandler
                    .execute(new ClientExecutionParams<GetServiceLevelObjectiveRequest, GetServiceLevelObjectiveResponse>()
                            .withOperationName("GetServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getServiceLevelObjectiveRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetServiceLevelObjectiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of service dependencies of the service that you specify. A dependency is an infrastructure
     * component that an operation of this service connects with. Dependencies can include Amazon Web Services services,
     * Amazon Web Services resources, and third-party services.
     * </p>
     *
     * @param listServiceDependenciesRequest
     * @return Result of the ListServiceDependencies operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListServiceDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceDependenciesResponse listServiceDependencies(ListServiceDependenciesRequest listServiceDependenciesRequest)
            throws ValidationException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServiceDependenciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceDependenciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceDependenciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceDependencies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListServiceDependenciesRequest, ListServiceDependenciesResponse>()
                            .withOperationName("ListServiceDependencies").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listServiceDependenciesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListServiceDependenciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the list of dependents that invoked the specified service during the provided time range. Dependents
     * include other services, CloudWatch Synthetics canaries, and clients that are instrumented with CloudWatch RUM app
     * monitors.
     * </p>
     *
     * @param listServiceDependentsRequest
     * @return Result of the ListServiceDependents operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListServiceDependents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceDependents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceDependentsResponse listServiceDependents(ListServiceDependentsRequest listServiceDependentsRequest)
            throws ValidationException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceDependentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServiceDependentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceDependentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceDependentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceDependents");

            return clientHandler.execute(new ClientExecutionParams<ListServiceDependentsRequest, ListServiceDependentsResponse>()
                    .withOperationName("ListServiceDependents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServiceDependentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServiceDependentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of SLOs created in this account.
     * </p>
     *
     * @param listServiceLevelObjectivesRequest
     * @return Result of the ListServiceLevelObjectives operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListServiceLevelObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceLevelObjectives"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceLevelObjectivesResponse listServiceLevelObjectives(
            ListServiceLevelObjectivesRequest listServiceLevelObjectivesRequest) throws ValidationException, ThrottlingException,
            AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceLevelObjectivesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServiceLevelObjectivesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceLevelObjectivesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceLevelObjectivesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceLevelObjectives");

            return clientHandler
                    .execute(new ClientExecutionParams<ListServiceLevelObjectivesRequest, ListServiceLevelObjectivesResponse>()
                            .withOperationName("ListServiceLevelObjectives").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listServiceLevelObjectivesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListServiceLevelObjectivesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the <i>operations</i> of this service that have been discovered by Application Signals. Only
     * the operations that were invoked during the specified time range are returned.
     * </p>
     *
     * @param listServiceOperationsRequest
     * @return Result of the ListServiceOperations operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListServiceOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServiceOperationsResponse listServiceOperations(ListServiceOperationsRequest listServiceOperationsRequest)
            throws ValidationException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServiceOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListServiceOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceOperations");

            return clientHandler.execute(new ClientExecutionParams<ListServiceOperationsRequest, ListServiceOperationsResponse>()
                    .withOperationName("ListServiceOperations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServiceOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServiceOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of services that have been discovered by Application Signals. A service represents a minimum
     * logical and transactional unit that completes a business function. Services are discovered through Application
     * Signals instrumentation.
     * </p>
     *
     * @param listServicesRequest
     * @return Result of the ListServices operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListServicesResponse listServices(ListServicesRequest listServicesRequest) throws ValidationException,
            ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListServicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");

            return clientHandler.execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                    .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listServicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListServicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch resource. Tags can be assigned to service level objectives.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables this Amazon Web Services account to be able to use CloudWatch Application Signals by creating the
     * <i>AWSServiceRoleForCloudWatchApplicationSignals</i> service-linked role. This service- linked role has the
     * following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>xray:GetServiceGraph</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:StartQuery</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:GetQueryResults</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:GetMetricData</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:ListMetrics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>autoscaling:DescribeAutoScalingGroups</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * After completing this step, you still need to instrument your Java and Python applications to send data to
     * Application Signals. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Signals-Enable.html">
     * Enabling Application Signals</a>.
     * </p>
     *
     * @param startDiscoveryRequest
     * @return Result of the StartDiscovery operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws AccessDeniedException
     *         You don't have sufficient permissions to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.StartDiscovery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/StartDiscovery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDiscoveryResponse startDiscovery(StartDiscoveryRequest startDiscoveryRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDiscoveryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartDiscoveryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDiscoveryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDiscoveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDiscovery");

            return clientHandler.execute(new ClientExecutionParams<StartDiscoveryRequest, StartDiscoveryResponse>()
                    .withOperationName("StartDiscovery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startDiscoveryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDiscoveryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch resource, such as a service level
     * objective.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with an alarm that already has tags. If you specify a new tag key
     * for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key that
     * is already associated with the alarm, the new tag value that you specify replaces the previous value for that
     * tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a CloudWatch resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws ServiceQuotaExceededException
     *         This request exceeds a service quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing service level objective (SLO). If you omit parameters, the previous values of those
     * parameters are retained.
     * </p>
     * <p>
     * You cannot change from a period-based SLO to a request-based SLO, or change from a request-based SLO to a
     * period-based SLO.
     * </p>
     *
     * @param updateServiceLevelObjectiveRequest
     * @return Result of the UpdateServiceLevelObjective operation returned by the service.
     * @throws ValidationException
     *         The resource is not valid.
     * @throws ResourceNotFoundException
     *         Resource not found.
     * @throws ThrottlingException
     *         The request was throttled because of quota limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ApplicationSignalsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ApplicationSignalsClient.UpdateServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/UpdateServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateServiceLevelObjectiveResponse updateServiceLevelObjective(
            UpdateServiceLevelObjectiveRequest updateServiceLevelObjectiveRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ApplicationSignalsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateServiceLevelObjectiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceLevelObjective");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceLevelObjectiveRequest, UpdateServiceLevelObjectiveResponse>()
                            .withOperationName("UpdateServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateServiceLevelObjectiveRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateServiceLevelObjectiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ApplicationSignalsServiceClientConfigurationBuilder serviceConfigBuilder = new ApplicationSignalsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationSignalsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    @Override
    public final ApplicationSignalsServiceClientConfiguration serviceClientConfiguration() {
        return new ApplicationSignalsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
