/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about one CloudWatch metric associated with this entity discovered by Application
 * Signals.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricReference implements SdkPojo, Serializable, ToCopyableBuilder<MetricReference.Builder, MetricReference> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(MetricReference::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> METRIC_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricType").getter(getter(MetricReference::metricType)).setter(setter(Builder::metricType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricType").build()).build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(MetricReference::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(MetricReference::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            METRIC_TYPE_FIELD, DIMENSIONS_FIELD, METRIC_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final String metricType;

    private final List<Dimension> dimensions;

    private final String metricName;

    private MetricReference(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.metricType = builder.metricType;
        this.dimensions = builder.dimensions;
        this.metricName = builder.metricName;
    }

    /**
     * <p>
     * The namespace of the metric. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
     * >CloudWatchNamespaces</a>.
     * </p>
     * 
     * @return The namespace of the metric. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
     *         >CloudWatchNamespaces</a>.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * Used to display the appropriate statistics in the CloudWatch console.
     * </p>
     * 
     * @return Used to display the appropriate statistics in the CloudWatch console.
     */
    public final String metricType() {
        return metricType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of one or more dimensions that further define the metric. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
     * >CloudWatchDimensions</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return An array of one or more dimensions that further define the metric. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
     *         >CloudWatchDimensions</a>.
     */
    public final List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public final String metricName() {
        return metricName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(metricType());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricReference)) {
            return false;
        }
        MetricReference other = (MetricReference) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(metricType(), other.metricType())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(metricName(), other.metricName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricReference").add("Namespace", namespace()).add("MetricType", metricType())
                .add("Dimensions", hasDimensions() ? dimensions() : null).add("MetricName", metricName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "MetricType":
            return Optional.ofNullable(clazz.cast(metricType()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricReference, T> g) {
        return obj -> g.apply((MetricReference) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricReference> {
        /**
         * <p>
         * The namespace of the metric. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
         * >CloudWatchNamespaces</a>.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Namespace"
         *        >CloudWatchNamespaces</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * Used to display the appropriate statistics in the CloudWatch console.
         * </p>
         * 
         * @param metricType
         *        Used to display the appropriate statistics in the CloudWatch console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricType(String metricType);

        /**
         * <p>
         * An array of one or more dimensions that further define the metric. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
         * >CloudWatchDimensions</a>.
         * </p>
         * 
         * @param dimensions
         *        An array of one or more dimensions that further define the metric. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
         *        >CloudWatchDimensions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * An array of one or more dimensions that further define the metric. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
         * >CloudWatchDimensions</a>.
         * </p>
         * 
         * @param dimensions
         *        An array of one or more dimensions that further define the metric. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
         *        >CloudWatchDimensions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * An array of one or more dimensions that further define the metric. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension"
         * >CloudWatchDimensions</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationsignals.model.Dimension.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.applicationsignals.model.Dimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationsignals.model.Dimension.Builder#build()} is called
         * immediately and its result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationsignals.model.Dimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private String metricType;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String metricName;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricReference model) {
            namespace(model.namespace);
            metricType(model.metricType);
            dimensions(model.dimensions);
            metricName(model.metricName);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getMetricType() {
            return metricType;
        }

        public final void setMetricType(String metricType) {
            this.metricType = metricType;
        }

        @Override
        public final Builder metricType(String metricType) {
            this.metricType = metricType;
            return this;
        }

        public final List<Dimension.Builder> getDimensions() {
            List<Dimension.Builder> result = DimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public MetricReference build() {
            return new MetricReference(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
