/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This structure contains information about a service dependent that was discovered by Application Signals. A dependent
 * is an entity that invoked the specified service during the provided time range. Dependents include other services,
 * CloudWatch Synthetics canaries, and clients that are instrumented with CloudWatch RUM app monitors.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceDependent implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceDependent.Builder, ServiceDependent> {
    private static final SdkField<String> OPERATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperationName").getter(getter(ServiceDependent::operationName)).setter(setter(Builder::operationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationName").build()).build();

    private static final SdkField<Map<String, String>> DEPENDENT_KEY_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DependentKeyAttributes")
            .getter(getter(ServiceDependent::dependentKeyAttributes))
            .setter(setter(Builder::dependentKeyAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependentKeyAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> DEPENDENT_OPERATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DependentOperationName").getter(getter(ServiceDependent::dependentOperationName))
            .setter(setter(Builder::dependentOperationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependentOperationName").build())
            .build();

    private static final SdkField<List<MetricReference>> METRIC_REFERENCES_FIELD = SdkField
            .<List<MetricReference>> builder(MarshallingType.LIST)
            .memberName("MetricReferences")
            .getter(getter(ServiceDependent::metricReferences))
            .setter(setter(Builder::metricReferences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricReferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricReference> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricReference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPERATION_NAME_FIELD,
            DEPENDENT_KEY_ATTRIBUTES_FIELD, DEPENDENT_OPERATION_NAME_FIELD, METRIC_REFERENCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String operationName;

    private final Map<String, String> dependentKeyAttributes;

    private final String dependentOperationName;

    private final List<MetricReference> metricReferences;

    private ServiceDependent(BuilderImpl builder) {
        this.operationName = builder.operationName;
        this.dependentKeyAttributes = builder.dependentKeyAttributes;
        this.dependentOperationName = builder.dependentOperationName;
        this.metricReferences = builder.metricReferences;
    }

    /**
     * <p>
     * If the invoked entity is an operation on an entity, the name of that dependent operation is displayed here.
     * </p>
     * 
     * @return If the invoked entity is an operation on an entity, the name of that dependent operation is displayed
     *         here.
     */
    public final String operationName() {
        return operationName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependentKeyAttributes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDependentKeyAttributes() {
        return dependentKeyAttributes != null && !(dependentKeyAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * This is a string-to-string map. It can include the following fields.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Type</code> designates the type of object this is.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of the
     * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Name</code> specifies the name of the object. This is used only if the value of the <code>Type</code> field
     * is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value of the
     * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependentKeyAttributes} method.
     * </p>
     * 
     * @return This is a string-to-string map. It can include the following fields.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Type</code> designates the type of object this is.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of
     *         the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Name</code> specifies the name of the object. This is used only if the value of the
     *         <code>Type</code> field is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>
     *         .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value
     *         of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
     *         </p>
     *         </li>
     */
    public final Map<String, String> dependentKeyAttributes() {
        return dependentKeyAttributes;
    }

    /**
     * <p>
     * If the dependent invoker was a service that invoked it from an operation, the name of that dependent operation is
     * displayed here.
     * </p>
     * 
     * @return If the dependent invoker was a service that invoked it from an operation, the name of that dependent
     *         operation is displayed here.
     */
    public final String dependentOperationName() {
        return dependentOperationName;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricReferences property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricReferences() {
        return metricReferences != null && !(metricReferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of structures that each contain information about one metric associated with this service dependent that
     * was discovered by Application Signals.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricReferences} method.
     * </p>
     * 
     * @return An array of structures that each contain information about one metric associated with this service
     *         dependent that was discovered by Application Signals.
     */
    public final List<MetricReference> metricReferences() {
        return metricReferences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(operationName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDependentKeyAttributes() ? dependentKeyAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dependentOperationName());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricReferences() ? metricReferences() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceDependent)) {
            return false;
        }
        ServiceDependent other = (ServiceDependent) obj;
        return Objects.equals(operationName(), other.operationName())
                && hasDependentKeyAttributes() == other.hasDependentKeyAttributes()
                && Objects.equals(dependentKeyAttributes(), other.dependentKeyAttributes())
                && Objects.equals(dependentOperationName(), other.dependentOperationName())
                && hasMetricReferences() == other.hasMetricReferences()
                && Objects.equals(metricReferences(), other.metricReferences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceDependent").add("OperationName", operationName())
                .add("DependentKeyAttributes", hasDependentKeyAttributes() ? dependentKeyAttributes() : null)
                .add("DependentOperationName", dependentOperationName())
                .add("MetricReferences", hasMetricReferences() ? metricReferences() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OperationName":
            return Optional.ofNullable(clazz.cast(operationName()));
        case "DependentKeyAttributes":
            return Optional.ofNullable(clazz.cast(dependentKeyAttributes()));
        case "DependentOperationName":
            return Optional.ofNullable(clazz.cast(dependentOperationName()));
        case "MetricReferences":
            return Optional.ofNullable(clazz.cast(metricReferences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ServiceDependent, T> g) {
        return obj -> g.apply((ServiceDependent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceDependent> {
        /**
         * <p>
         * If the invoked entity is an operation on an entity, the name of that dependent operation is displayed here.
         * </p>
         * 
         * @param operationName
         *        If the invoked entity is an operation on an entity, the name of that dependent operation is displayed
         *        here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationName(String operationName);

        /**
         * <p>
         * This is a string-to-string map. It can include the following fields.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Type</code> designates the type of object this is.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ResourceType</code> specifies the type of the resource. This field is used only when the value of the
         * <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Name</code> specifies the name of the object. This is used only if the value of the <code>Type</code>
         * field is <code>Service</code>, <code>RemoteService</code>, or <code>AWS::Service</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Identifier</code> identifies the resource objects of this resource. This is used only if the value of
         * the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dependentKeyAttributes
         *        This is a string-to-string map. It can include the following fields.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Type</code> designates the type of object this is.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ResourceType</code> specifies the type of the resource. This field is used only when the value
         *        of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Name</code> specifies the name of the object. This is used only if the value of the
         *        <code>Type</code> field is <code>Service</code>, <code>RemoteService</code>, or
         *        <code>AWS::Service</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Identifier</code> identifies the resource objects of this resource. This is used only if the
         *        value of the <code>Type</code> field is <code>Resource</code> or <code>AWS::Resource</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Environment</code> specifies the location where this object is hosted, or what it belongs to.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependentKeyAttributes(Map<String, String> dependentKeyAttributes);

        /**
         * <p>
         * If the dependent invoker was a service that invoked it from an operation, the name of that dependent
         * operation is displayed here.
         * </p>
         * 
         * @param dependentOperationName
         *        If the dependent invoker was a service that invoked it from an operation, the name of that dependent
         *        operation is displayed here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependentOperationName(String dependentOperationName);

        /**
         * <p>
         * An array of structures that each contain information about one metric associated with this service dependent
         * that was discovered by Application Signals.
         * </p>
         * 
         * @param metricReferences
         *        An array of structures that each contain information about one metric associated with this service
         *        dependent that was discovered by Application Signals.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricReferences(Collection<MetricReference> metricReferences);

        /**
         * <p>
         * An array of structures that each contain information about one metric associated with this service dependent
         * that was discovered by Application Signals.
         * </p>
         * 
         * @param metricReferences
         *        An array of structures that each contain information about one metric associated with this service
         *        dependent that was discovered by Application Signals.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricReferences(MetricReference... metricReferences);

        /**
         * <p>
         * An array of structures that each contain information about one metric associated with this service dependent
         * that was discovered by Application Signals.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationsignals.model.MetricReference.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.applicationsignals.model.MetricReference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationsignals.model.MetricReference.Builder#build()} is called
         * immediately and its result is passed to {@link #metricReferences(List<MetricReference>)}.
         * 
         * @param metricReferences
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationsignals.model.MetricReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricReferences(java.util.Collection<MetricReference>)
         */
        Builder metricReferences(Consumer<MetricReference.Builder>... metricReferences);
    }

    static final class BuilderImpl implements Builder {
        private String operationName;

        private Map<String, String> dependentKeyAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String dependentOperationName;

        private List<MetricReference> metricReferences = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceDependent model) {
            operationName(model.operationName);
            dependentKeyAttributes(model.dependentKeyAttributes);
            dependentOperationName(model.dependentOperationName);
            metricReferences(model.metricReferences);
        }

        public final String getOperationName() {
            return operationName;
        }

        public final void setOperationName(String operationName) {
            this.operationName = operationName;
        }

        @Override
        public final Builder operationName(String operationName) {
            this.operationName = operationName;
            return this;
        }

        public final Map<String, String> getDependentKeyAttributes() {
            if (dependentKeyAttributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return dependentKeyAttributes;
        }

        public final void setDependentKeyAttributes(Map<String, String> dependentKeyAttributes) {
            this.dependentKeyAttributes = AttributesCopier.copy(dependentKeyAttributes);
        }

        @Override
        public final Builder dependentKeyAttributes(Map<String, String> dependentKeyAttributes) {
            this.dependentKeyAttributes = AttributesCopier.copy(dependentKeyAttributes);
            return this;
        }

        public final String getDependentOperationName() {
            return dependentOperationName;
        }

        public final void setDependentOperationName(String dependentOperationName) {
            this.dependentOperationName = dependentOperationName;
        }

        @Override
        public final Builder dependentOperationName(String dependentOperationName) {
            this.dependentOperationName = dependentOperationName;
            return this;
        }

        public final List<MetricReference.Builder> getMetricReferences() {
            List<MetricReference.Builder> result = MetricReferencesCopier.copyToBuilder(this.metricReferences);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricReferences(Collection<MetricReference.BuilderImpl> metricReferences) {
            this.metricReferences = MetricReferencesCopier.copyFromBuilder(metricReferences);
        }

        @Override
        public final Builder metricReferences(Collection<MetricReference> metricReferences) {
            this.metricReferences = MetricReferencesCopier.copy(metricReferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricReferences(MetricReference... metricReferences) {
            metricReferences(Arrays.asList(metricReferences));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricReferences(Consumer<MetricReference.Builder>... metricReferences) {
            metricReferences(Stream.of(metricReferences).map(c -> MetricReference.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ServiceDependent build() {
            return new ServiceDependent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
