/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.applicationsignals.internal.ApplicationSignalsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.applicationsignals.model.AccessDeniedException;
import software.amazon.awssdk.services.applicationsignals.model.ApplicationSignalsException;
import software.amazon.awssdk.services.applicationsignals.model.BatchGetServiceLevelObjectiveBudgetReportRequest;
import software.amazon.awssdk.services.applicationsignals.model.BatchGetServiceLevelObjectiveBudgetReportResponse;
import software.amazon.awssdk.services.applicationsignals.model.ConflictException;
import software.amazon.awssdk.services.applicationsignals.model.CreateServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.CreateServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.DeleteServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.DeleteServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceRequest;
import software.amazon.awssdk.services.applicationsignals.model.GetServiceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependenciesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependenciesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependentsRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceDependentsResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceLevelObjectivesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceLevelObjectivesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceOperationsRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServiceOperationsResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListServicesRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListServicesResponse;
import software.amazon.awssdk.services.applicationsignals.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ResourceNotFoundException;
import software.amazon.awssdk.services.applicationsignals.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.applicationsignals.model.StartDiscoveryRequest;
import software.amazon.awssdk.services.applicationsignals.model.StartDiscoveryResponse;
import software.amazon.awssdk.services.applicationsignals.model.TagResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.TagResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.ThrottlingException;
import software.amazon.awssdk.services.applicationsignals.model.UntagResourceRequest;
import software.amazon.awssdk.services.applicationsignals.model.UntagResourceResponse;
import software.amazon.awssdk.services.applicationsignals.model.UpdateServiceLevelObjectiveRequest;
import software.amazon.awssdk.services.applicationsignals.model.UpdateServiceLevelObjectiveResponse;
import software.amazon.awssdk.services.applicationsignals.model.ValidationException;
import software.amazon.awssdk.services.applicationsignals.transform.BatchGetServiceLevelObjectiveBudgetReportRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.CreateServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.DeleteServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.GetServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.GetServiceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceDependenciesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceDependentsRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceLevelObjectivesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServiceOperationsRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.StartDiscoveryRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.applicationsignals.transform.UpdateServiceLevelObjectiveRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ApplicationSignalsAsyncClient}.
 *
 * @see ApplicationSignalsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultApplicationSignalsAsyncClient implements ApplicationSignalsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultApplicationSignalsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultApplicationSignalsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Use this operation to retrieve one or more <i>service level objective (SLO) budget reports</i>.
     * </p>
     * <p>
     * An <i>error budget</i> is the amount of time or requests in an unhealthy state that your service can accumulate
     * during an interval before your overall SLO budget health is breached and the SLO is considered to be unmet. For
     * example, an SLO with a threshold of 99.95% and a monthly interval translates to an error budget of 21.9 minutes
     * of downtime in a 30-day month.
     * </p>
     * <p>
     * Budget reports include a health indicator, the attainment value, and remaining budget.
     * </p>
     * <p>
     * For more information about SLO error budgets, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-ServiceLevelObjectives.html#CloudWatch-ServiceLevelObjectives-concepts"
     * > SLO concepts</a>.
     * </p>
     *
     * @param batchGetServiceLevelObjectiveBudgetReportRequest
     * @return A Java Future containing the result of the BatchGetServiceLevelObjectiveBudgetReport operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.BatchGetServiceLevelObjectiveBudgetReport
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/BatchGetServiceLevelObjectiveBudgetReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetServiceLevelObjectiveBudgetReportResponse> batchGetServiceLevelObjectiveBudgetReport(
            BatchGetServiceLevelObjectiveBudgetReportRequest batchGetServiceLevelObjectiveBudgetReportRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                batchGetServiceLevelObjectiveBudgetReportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetServiceLevelObjectiveBudgetReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetServiceLevelObjectiveBudgetReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetServiceLevelObjectiveBudgetReportResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchGetServiceLevelObjectiveBudgetReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetServiceLevelObjectiveBudgetReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetServiceLevelObjectiveBudgetReportRequest, BatchGetServiceLevelObjectiveBudgetReportResponse>()
                            .withOperationName("BatchGetServiceLevelObjectiveBudgetReport")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetServiceLevelObjectiveBudgetReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetServiceLevelObjectiveBudgetReportRequest));
            CompletableFuture<BatchGetServiceLevelObjectiveBudgetReportResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service level objective (SLO), which can help you ensure that your critical business operations are
     * meeting customer expectations. Use SLOs to set and track specific target levels for the reliability and
     * availability of your applications and services. SLOs use service level indicators (SLIs) to calculate whether the
     * application is performing at the level that you want.
     * </p>
     * <p>
     * Create an SLO to set a target for a service or operation’s availability or latency. CloudWatch measures this
     * target frequently you can find whether it has been breached.
     * </p>
     * <p>
     * The target performance quality that is defined for an SLO is the <i>attainment goal</i>.
     * </p>
     * <p>
     * You can set SLO targets for your applications that are discovered by Application Signals, using critical metrics
     * such as latency and availability. You can also set SLOs against any CloudWatch metric or math expression that
     * produces a time series.
     * </p>
     * <p>
     * When you create an SLO, you specify whether it is a <i>period-based SLO</i> or a <i>request-based SLO</i>. Each
     * type of SLO has a different way of evaluating your application's performance against its attainment goal.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A <i>period-based SLO</i> uses defined <i>periods</i> of time within a specified total time interval. For each
     * period of time, Application Signals determines whether the application met its goal. The attainment rate is
     * calculated as the <code>number of good periods/number of total periods</code>.
     * </p>
     * <p>
     * For example, for a period-based SLO, meeting an attainment goal of 99.9% means that within your interval, your
     * application must meet its performance goal during at least 99.9% of the time periods.
     * </p>
     * </li>
     * <li>
     * <p>
     * A <i>request-based SLO</i> doesn't use pre-defined periods of time. Instead, the SLO measures
     * <code>number of good requests/number of total requests</code> during the interval. At any time, you can find the
     * ratio of good requests to total requests for the interval up to the time stamp that you specify, and measure that
     * ratio against the goal set in your SLO.
     * </p>
     * </li>
     * </ul>
     * <p>
     * After you have created an SLO, you can retrieve error budget reports for it. An <i>error budget</i> is the amount
     * of time or amount of requests that your application can be non-compliant with the SLO's goal, and still have your
     * application meet the goal.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a period-based SLO, the error budget starts at a number defined by the highest number of periods that can
     * fail to meet the threshold, while still meeting the overall goal. The <i>remaining error budget</i> decreases
     * with every failed period that is recorded. The error budget within one interval can never increase.
     * </p>
     * <p>
     * For example, an SLO with a threshold that 99.95% of requests must be completed under 2000ms every month
     * translates to an error budget of 21.9 minutes of downtime per month.
     * </p>
     * </li>
     * <li>
     * <p>
     * For a request-based SLO, the remaining error budget is dynamic and can increase or decrease, depending on the
     * ratio of good requests to total requests.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about SLOs, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-ServiceLevelObjectives.html">
     * Service level objectives (SLOs)</a>.
     * </p>
     * <p>
     * When you perform a <code>CreateServiceLevelObjective</code> operation, Application Signals creates the
     * <i>AWSServiceRoleForCloudWatchApplicationSignals</i> service-linked role, if it doesn't already exist in your
     * account. This service- linked role has the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>xray:GetServiceGraph</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:StartQuery</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:GetQueryResults</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:GetMetricData</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:ListMetrics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>autoscaling:DescribeAutoScalingGroups</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param createServiceLevelObjectiveRequest
     * @return A Java Future containing the result of the CreateServiceLevelObjective operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>ConflictException This operation attempted to create a resource that already exists.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.CreateServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/CreateServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServiceLevelObjectiveResponse> createServiceLevelObjective(
            CreateServiceLevelObjectiveRequest createServiceLevelObjectiveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateServiceLevelObjective");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateServiceLevelObjectiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServiceLevelObjectiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServiceLevelObjectiveRequest, CreateServiceLevelObjectiveResponse>()
                            .withOperationName("CreateServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateServiceLevelObjectiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createServiceLevelObjectiveRequest));
            CompletableFuture<CreateServiceLevelObjectiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified service level objective.
     * </p>
     *
     * @param deleteServiceLevelObjectiveRequest
     * @return A Java Future containing the result of the DeleteServiceLevelObjective operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.DeleteServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/DeleteServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceLevelObjectiveResponse> deleteServiceLevelObjective(
            DeleteServiceLevelObjectiveRequest deleteServiceLevelObjectiveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteServiceLevelObjective");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteServiceLevelObjectiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceLevelObjectiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceLevelObjectiveRequest, DeleteServiceLevelObjectiveResponse>()
                            .withOperationName("DeleteServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteServiceLevelObjectiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceLevelObjectiveRequest));
            CompletableFuture<DeleteServiceLevelObjectiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a service discovered by Application Signals.
     * </p>
     *
     * @param getServiceRequest
     * @return A Java Future containing the result of the GetService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.GetService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/GetService"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceResponse> getService(GetServiceRequest getServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceRequest, GetServiceResponse>().withOperationName("GetService")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceRequest));
            CompletableFuture<GetServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about one SLO created in the account.
     * </p>
     *
     * @param getServiceLevelObjectiveRequest
     * @return A Java Future containing the result of the GetServiceLevelObjective operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.GetServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/GetServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceLevelObjectiveResponse> getServiceLevelObjective(
            GetServiceLevelObjectiveRequest getServiceLevelObjectiveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceLevelObjective");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceLevelObjectiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceLevelObjectiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceLevelObjectiveRequest, GetServiceLevelObjectiveResponse>()
                            .withOperationName("GetServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetServiceLevelObjectiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceLevelObjectiveRequest));
            CompletableFuture<GetServiceLevelObjectiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of service dependencies of the service that you specify. A dependency is an infrastructure
     * component that an operation of this service connects with. Dependencies can include Amazon Web Services services,
     * Amazon Web Services resources, and third-party services.
     * </p>
     *
     * @param listServiceDependenciesRequest
     * @return A Java Future containing the result of the ListServiceDependencies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListServiceDependencies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceDependencies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceDependenciesResponse> listServiceDependencies(
            ListServiceDependenciesRequest listServiceDependenciesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceDependenciesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceDependenciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceDependencies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceDependenciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceDependenciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceDependenciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceDependenciesRequest, ListServiceDependenciesResponse>()
                            .withOperationName("ListServiceDependencies").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceDependenciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceDependenciesRequest));
            CompletableFuture<ListServiceDependenciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of dependents that invoked the specified service during the provided time range. Dependents
     * include other services, CloudWatch Synthetics canaries, and clients that are instrumented with CloudWatch RUM app
     * monitors.
     * </p>
     *
     * @param listServiceDependentsRequest
     * @return A Java Future containing the result of the ListServiceDependents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListServiceDependents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceDependents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceDependentsResponse> listServiceDependents(
            ListServiceDependentsRequest listServiceDependentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceDependentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceDependentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceDependents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceDependentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceDependentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceDependentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceDependentsRequest, ListServiceDependentsResponse>()
                            .withOperationName("ListServiceDependents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceDependentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceDependentsRequest));
            CompletableFuture<ListServiceDependentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of SLOs created in this account.
     * </p>
     *
     * @param listServiceLevelObjectivesRequest
     * @return A Java Future containing the result of the ListServiceLevelObjectives operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListServiceLevelObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceLevelObjectives"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceLevelObjectivesResponse> listServiceLevelObjectives(
            ListServiceLevelObjectivesRequest listServiceLevelObjectivesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceLevelObjectivesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceLevelObjectivesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceLevelObjectives");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceLevelObjectivesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceLevelObjectivesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceLevelObjectivesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceLevelObjectivesRequest, ListServiceLevelObjectivesResponse>()
                            .withOperationName("ListServiceLevelObjectives").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceLevelObjectivesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceLevelObjectivesRequest));
            CompletableFuture<ListServiceLevelObjectivesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the <i>operations</i> of this service that have been discovered by Application Signals. Only
     * the operations that were invoked during the specified time range are returned.
     * </p>
     *
     * @param listServiceOperationsRequest
     * @return A Java Future containing the result of the ListServiceOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListServiceOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServiceOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServiceOperationsResponse> listServiceOperations(
            ListServiceOperationsRequest listServiceOperationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServiceOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServiceOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServiceOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServiceOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListServiceOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServiceOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServiceOperationsRequest, ListServiceOperationsResponse>()
                            .withOperationName("ListServiceOperations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServiceOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServiceOperationsRequest));
            CompletableFuture<ListServiceOperationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of services that have been discovered by Application Signals. A service represents a minimum
     * logical and transactional unit that completes a business function. Services are discovered through Application
     * Signals instrumentation.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListServices"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesRequest));
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch resource. Tags can be assigned to service level objectives.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables this Amazon Web Services account to be able to use CloudWatch Application Signals by creating the
     * <i>AWSServiceRoleForCloudWatchApplicationSignals</i> service-linked role. This service- linked role has the
     * following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>xray:GetServiceGraph</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:StartQuery</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>logs:GetQueryResults</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:GetMetricData</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cloudwatch:ListMetrics</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>tag:GetResources</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>autoscaling:DescribeAutoScalingGroups</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * After completing this step, you still need to instrument your Java and Python applications to send data to
     * Application Signals. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Application-Signals-Enable.html">
     * Enabling Application Signals</a>.
     * </p>
     *
     * @param startDiscoveryRequest
     * @return A Java Future containing the result of the StartDiscovery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>AccessDeniedException You don't have sufficient permissions to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.StartDiscovery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/StartDiscovery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartDiscoveryResponse> startDiscovery(StartDiscoveryRequest startDiscoveryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDiscoveryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDiscoveryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDiscovery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDiscoveryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDiscoveryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDiscoveryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDiscoveryRequest, StartDiscoveryResponse>()
                            .withOperationName("StartDiscovery").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartDiscoveryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startDiscoveryRequest));
            CompletableFuture<StartDiscoveryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch resource, such as a service level
     * objective.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to Amazon Web Services and are interpreted strictly as strings of
     * characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with an alarm that already has tags. If you specify a new tag key
     * for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key that
     * is already associated with the alarm, the new tag value that you specify replaces the previous value for that
     * tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a CloudWatch resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>ServiceQuotaExceededException This request exceeds a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing service level objective (SLO). If you omit parameters, the previous values of those
     * parameters are retained.
     * </p>
     * <p>
     * You cannot change from a period-based SLO to a request-based SLO, or change from a request-based SLO to a
     * period-based SLO.
     * </p>
     *
     * @param updateServiceLevelObjectiveRequest
     * @return A Java Future containing the result of the UpdateServiceLevelObjective operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The resource is not valid.</li>
     *         <li>ResourceNotFoundException Resource not found.</li>
     *         <li>ThrottlingException The request was throttled because of quota limits.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ApplicationSignalsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ApplicationSignalsAsyncClient.UpdateServiceLevelObjective
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/application-signals-2024-04-15/UpdateServiceLevelObjective"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceLevelObjectiveResponse> updateServiceLevelObjective(
            UpdateServiceLevelObjectiveRequest updateServiceLevelObjectiveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceLevelObjectiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceLevelObjectiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Application Signals");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceLevelObjective");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceLevelObjectiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceLevelObjectiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceLevelObjectiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceLevelObjectiveRequest, UpdateServiceLevelObjectiveResponse>()
                            .withOperationName("UpdateServiceLevelObjective").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceLevelObjectiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceLevelObjectiveRequest));
            CompletableFuture<UpdateServiceLevelObjectiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ApplicationSignalsServiceClientConfiguration serviceClientConfiguration() {
        return new ApplicationSignalsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ApplicationSignalsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        ApplicationSignalsServiceClientConfigurationBuilder serviceConfigBuilder = new ApplicationSignalsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
