/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationsignals.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure containing information about one service level objective (SLO) that has been created in Application
 * Signals. Creating SLOs can help you ensure your services are performing to the level that you expect. SLOs help you
 * set and track a specific target level for the reliability and availability of your applications and services. Each
 * SLO uses a service level indicator (SLI), which is a key performance metric, to calculate how much underperformance
 * can be tolerated before the goal that you set for the SLO is not achieved.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ServiceLevelObjective implements SdkPojo, Serializable,
        ToCopyableBuilder<ServiceLevelObjective.Builder, ServiceLevelObjective> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(ServiceLevelObjective::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ServiceLevelObjective::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ServiceLevelObjective::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(ServiceLevelObjective::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(ServiceLevelObjective::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final SdkField<ServiceLevelIndicator> SLI_FIELD = SdkField
            .<ServiceLevelIndicator> builder(MarshallingType.SDK_POJO).memberName("Sli")
            .getter(getter(ServiceLevelObjective::sli)).setter(setter(Builder::sli)).constructor(ServiceLevelIndicator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sli").build()).build();

    private static final SdkField<RequestBasedServiceLevelIndicator> REQUEST_BASED_SLI_FIELD = SdkField
            .<RequestBasedServiceLevelIndicator> builder(MarshallingType.SDK_POJO).memberName("RequestBasedSli")
            .getter(getter(ServiceLevelObjective::requestBasedSli)).setter(setter(Builder::requestBasedSli))
            .constructor(RequestBasedServiceLevelIndicator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestBasedSli").build()).build();

    private static final SdkField<String> EVALUATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvaluationType").getter(getter(ServiceLevelObjective::evaluationTypeAsString))
            .setter(setter(Builder::evaluationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationType").build()).build();

    private static final SdkField<Goal> GOAL_FIELD = SdkField.<Goal> builder(MarshallingType.SDK_POJO).memberName("Goal")
            .getter(getter(ServiceLevelObjective::goal)).setter(setter(Builder::goal)).constructor(Goal::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Goal").build()).build();

    private static final SdkField<List<BurnRateConfiguration>> BURN_RATE_CONFIGURATIONS_FIELD = SdkField
            .<List<BurnRateConfiguration>> builder(MarshallingType.LIST)
            .memberName("BurnRateConfigurations")
            .getter(getter(ServiceLevelObjective::burnRateConfigurations))
            .setter(setter(Builder::burnRateConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BurnRateConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BurnRateConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(BurnRateConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            DESCRIPTION_FIELD, CREATED_TIME_FIELD, LAST_UPDATED_TIME_FIELD, SLI_FIELD, REQUEST_BASED_SLI_FIELD,
            EVALUATION_TYPE_FIELD, GOAL_FIELD, BURN_RATE_CONFIGURATIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String description;

    private final Instant createdTime;

    private final Instant lastUpdatedTime;

    private final ServiceLevelIndicator sli;

    private final RequestBasedServiceLevelIndicator requestBasedSli;

    private final String evaluationType;

    private final Goal goal;

    private final List<BurnRateConfiguration> burnRateConfigurations;

    private ServiceLevelObjective(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.description = builder.description;
        this.createdTime = builder.createdTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.sli = builder.sli;
        this.requestBasedSli = builder.requestBasedSli;
        this.evaluationType = builder.evaluationType;
        this.goal = builder.goal;
        this.burnRateConfigurations = builder.burnRateConfigurations;
    }

    /**
     * <p>
     * The ARN of this SLO.
     * </p>
     * 
     * @return The ARN of this SLO.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of this SLO.
     * </p>
     * 
     * @return The name of this SLO.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description that you created for this SLO.
     * </p>
     * 
     * @return The description that you created for this SLO.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The date and time that this SLO was created. When used in a raw HTTP Query API, it is formatted as
     * <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
     * </p>
     * 
     * @return The date and time that this SLO was created. When used in a raw HTTP Query API, it is formatted as
     *         <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The time that this SLO was most recently updated. When used in a raw HTTP Query API, it is formatted as
     * <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
     * </p>
     * 
     * @return The time that this SLO was most recently updated. When used in a raw HTTP Query API, it is formatted as
     *         <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * A structure containing information about the performance metric that this SLO monitors, if this is a period-based
     * SLO.
     * </p>
     * 
     * @return A structure containing information about the performance metric that this SLO monitors, if this is a
     *         period-based SLO.
     */
    public final ServiceLevelIndicator sli() {
        return sli;
    }

    /**
     * <p>
     * A structure containing information about the performance metric that this SLO monitors, if this is a
     * request-based SLO.
     * </p>
     * 
     * @return A structure containing information about the performance metric that this SLO monitors, if this is a
     *         request-based SLO.
     */
    public final RequestBasedServiceLevelIndicator requestBasedSli() {
        return requestBasedSli;
    }

    /**
     * <p>
     * Displays whether this is a period-based SLO or a request-based SLO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #evaluationType}
     * will return {@link EvaluationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #evaluationTypeAsString}.
     * </p>
     * 
     * @return Displays whether this is a period-based SLO or a request-based SLO.
     * @see EvaluationType
     */
    public final EvaluationType evaluationType() {
        return EvaluationType.fromValue(evaluationType);
    }

    /**
     * <p>
     * Displays whether this is a period-based SLO or a request-based SLO.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #evaluationType}
     * will return {@link EvaluationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #evaluationTypeAsString}.
     * </p>
     * 
     * @return Displays whether this is a period-based SLO or a request-based SLO.
     * @see EvaluationType
     */
    public final String evaluationTypeAsString() {
        return evaluationType;
    }

    /**
     * Returns the value of the Goal property for this object.
     * 
     * @return The value of the Goal property for this object.
     */
    public final Goal goal() {
        return goal;
    }

    /**
     * For responses, this returns true if the service returned a value for the BurnRateConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasBurnRateConfigurations() {
        return burnRateConfigurations != null && !(burnRateConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Each object in this array defines the length of the look-back window used to calculate one burn rate metric for
     * this SLO. The burn rate measures how fast the service is consuming the error budget, relative to the attainment
     * goal of the SLO.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBurnRateConfigurations} method.
     * </p>
     * 
     * @return Each object in this array defines the length of the look-back window used to calculate one burn rate
     *         metric for this SLO. The burn rate measures how fast the service is consuming the error budget, relative
     *         to the attainment goal of the SLO.
     */
    public final List<BurnRateConfiguration> burnRateConfigurations() {
        return burnRateConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(sli());
        hashCode = 31 * hashCode + Objects.hashCode(requestBasedSli());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(goal());
        hashCode = 31 * hashCode + Objects.hashCode(hasBurnRateConfigurations() ? burnRateConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ServiceLevelObjective)) {
            return false;
        }
        ServiceLevelObjective other = (ServiceLevelObjective) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime()) && Objects.equals(sli(), other.sli())
                && Objects.equals(requestBasedSli(), other.requestBasedSli())
                && Objects.equals(evaluationTypeAsString(), other.evaluationTypeAsString())
                && Objects.equals(goal(), other.goal()) && hasBurnRateConfigurations() == other.hasBurnRateConfigurations()
                && Objects.equals(burnRateConfigurations(), other.burnRateConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ServiceLevelObjective").add("Arn", arn()).add("Name", name()).add("Description", description())
                .add("CreatedTime", createdTime()).add("LastUpdatedTime", lastUpdatedTime()).add("Sli", sli())
                .add("RequestBasedSli", requestBasedSli()).add("EvaluationType", evaluationTypeAsString()).add("Goal", goal())
                .add("BurnRateConfigurations", hasBurnRateConfigurations() ? burnRateConfigurations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "Sli":
            return Optional.ofNullable(clazz.cast(sli()));
        case "RequestBasedSli":
            return Optional.ofNullable(clazz.cast(requestBasedSli()));
        case "EvaluationType":
            return Optional.ofNullable(clazz.cast(evaluationTypeAsString()));
        case "Goal":
            return Optional.ofNullable(clazz.cast(goal()));
        case "BurnRateConfigurations":
            return Optional.ofNullable(clazz.cast(burnRateConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Arn", ARN_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("Sli", SLI_FIELD);
        map.put("RequestBasedSli", REQUEST_BASED_SLI_FIELD);
        map.put("EvaluationType", EVALUATION_TYPE_FIELD);
        map.put("Goal", GOAL_FIELD);
        map.put("BurnRateConfigurations", BURN_RATE_CONFIGURATIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ServiceLevelObjective, T> g) {
        return obj -> g.apply((ServiceLevelObjective) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ServiceLevelObjective> {
        /**
         * <p>
         * The ARN of this SLO.
         * </p>
         * 
         * @param arn
         *        The ARN of this SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of this SLO.
         * </p>
         * 
         * @param name
         *        The name of this SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description that you created for this SLO.
         * </p>
         * 
         * @param description
         *        The description that you created for this SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The date and time that this SLO was created. When used in a raw HTTP Query API, it is formatted as
         * <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
         * </p>
         * 
         * @param createdTime
         *        The date and time that this SLO was created. When used in a raw HTTP Query API, it is formatted as
         *        <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The time that this SLO was most recently updated. When used in a raw HTTP Query API, it is formatted as
         * <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time that this SLO was most recently updated. When used in a raw HTTP Query API, it is formatted
         *        as <code>yyyy-MM-dd'T'HH:mm:ss</code>. For example, <code>2019-07-01T23:59:59</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * A structure containing information about the performance metric that this SLO monitors, if this is a
         * period-based SLO.
         * </p>
         * 
         * @param sli
         *        A structure containing information about the performance metric that this SLO monitors, if this is a
         *        period-based SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sli(ServiceLevelIndicator sli);

        /**
         * <p>
         * A structure containing information about the performance metric that this SLO monitors, if this is a
         * period-based SLO.
         * </p>
         * This is a convenience method that creates an instance of the {@link ServiceLevelIndicator.Builder} avoiding
         * the need to create one manually via {@link ServiceLevelIndicator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ServiceLevelIndicator.Builder#build()} is called immediately and
         * its result is passed to {@link #sli(ServiceLevelIndicator)}.
         * 
         * @param sli
         *        a consumer that will call methods on {@link ServiceLevelIndicator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sli(ServiceLevelIndicator)
         */
        default Builder sli(Consumer<ServiceLevelIndicator.Builder> sli) {
            return sli(ServiceLevelIndicator.builder().applyMutation(sli).build());
        }

        /**
         * <p>
         * A structure containing information about the performance metric that this SLO monitors, if this is a
         * request-based SLO.
         * </p>
         * 
         * @param requestBasedSli
         *        A structure containing information about the performance metric that this SLO monitors, if this is a
         *        request-based SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestBasedSli(RequestBasedServiceLevelIndicator requestBasedSli);

        /**
         * <p>
         * A structure containing information about the performance metric that this SLO monitors, if this is a
         * request-based SLO.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link RequestBasedServiceLevelIndicator.Builder} avoiding the need to create one manually via
         * {@link RequestBasedServiceLevelIndicator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RequestBasedServiceLevelIndicator.Builder#build()} is called
         * immediately and its result is passed to {@link #requestBasedSli(RequestBasedServiceLevelIndicator)}.
         * 
         * @param requestBasedSli
         *        a consumer that will call methods on {@link RequestBasedServiceLevelIndicator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requestBasedSli(RequestBasedServiceLevelIndicator)
         */
        default Builder requestBasedSli(Consumer<RequestBasedServiceLevelIndicator.Builder> requestBasedSli) {
            return requestBasedSli(RequestBasedServiceLevelIndicator.builder().applyMutation(requestBasedSli).build());
        }

        /**
         * <p>
         * Displays whether this is a period-based SLO or a request-based SLO.
         * </p>
         * 
         * @param evaluationType
         *        Displays whether this is a period-based SLO or a request-based SLO.
         * @see EvaluationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EvaluationType
         */
        Builder evaluationType(String evaluationType);

        /**
         * <p>
         * Displays whether this is a period-based SLO or a request-based SLO.
         * </p>
         * 
         * @param evaluationType
         *        Displays whether this is a period-based SLO or a request-based SLO.
         * @see EvaluationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EvaluationType
         */
        Builder evaluationType(EvaluationType evaluationType);

        /**
         * Sets the value of the Goal property for this object.
         *
         * @param goal
         *        The new value for the Goal property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder goal(Goal goal);

        /**
         * Sets the value of the Goal property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Goal.Builder} avoiding the need to create
         * one manually via {@link Goal#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Goal.Builder#build()} is called immediately and its result is
         * passed to {@link #goal(Goal)}.
         * 
         * @param goal
         *        a consumer that will call methods on {@link Goal.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #goal(Goal)
         */
        default Builder goal(Consumer<Goal.Builder> goal) {
            return goal(Goal.builder().applyMutation(goal).build());
        }

        /**
         * <p>
         * Each object in this array defines the length of the look-back window used to calculate one burn rate metric
         * for this SLO. The burn rate measures how fast the service is consuming the error budget, relative to the
         * attainment goal of the SLO.
         * </p>
         * 
         * @param burnRateConfigurations
         *        Each object in this array defines the length of the look-back window used to calculate one burn rate
         *        metric for this SLO. The burn rate measures how fast the service is consuming the error budget,
         *        relative to the attainment goal of the SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder burnRateConfigurations(Collection<BurnRateConfiguration> burnRateConfigurations);

        /**
         * <p>
         * Each object in this array defines the length of the look-back window used to calculate one burn rate metric
         * for this SLO. The burn rate measures how fast the service is consuming the error budget, relative to the
         * attainment goal of the SLO.
         * </p>
         * 
         * @param burnRateConfigurations
         *        Each object in this array defines the length of the look-back window used to calculate one burn rate
         *        metric for this SLO. The burn rate measures how fast the service is consuming the error budget,
         *        relative to the attainment goal of the SLO.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder burnRateConfigurations(BurnRateConfiguration... burnRateConfigurations);

        /**
         * <p>
         * Each object in this array defines the length of the look-back window used to calculate one burn rate metric
         * for this SLO. The burn rate measures how fast the service is consuming the error budget, relative to the
         * attainment goal of the SLO.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationsignals.model.BurnRateConfiguration.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.applicationsignals.model.BurnRateConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationsignals.model.BurnRateConfiguration.Builder#build()} is
         * called immediately and its result is passed to {@link #burnRateConfigurations(List<BurnRateConfiguration>)}.
         * 
         * @param burnRateConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationsignals.model.BurnRateConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #burnRateConfigurations(java.util.Collection<BurnRateConfiguration>)
         */
        Builder burnRateConfigurations(Consumer<BurnRateConfiguration.Builder>... burnRateConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String description;

        private Instant createdTime;

        private Instant lastUpdatedTime;

        private ServiceLevelIndicator sli;

        private RequestBasedServiceLevelIndicator requestBasedSli;

        private String evaluationType;

        private Goal goal;

        private List<BurnRateConfiguration> burnRateConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ServiceLevelObjective model) {
            arn(model.arn);
            name(model.name);
            description(model.description);
            createdTime(model.createdTime);
            lastUpdatedTime(model.lastUpdatedTime);
            sli(model.sli);
            requestBasedSli(model.requestBasedSli);
            evaluationType(model.evaluationType);
            goal(model.goal);
            burnRateConfigurations(model.burnRateConfigurations);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final ServiceLevelIndicator.Builder getSli() {
            return sli != null ? sli.toBuilder() : null;
        }

        public final void setSli(ServiceLevelIndicator.BuilderImpl sli) {
            this.sli = sli != null ? sli.build() : null;
        }

        @Override
        public final Builder sli(ServiceLevelIndicator sli) {
            this.sli = sli;
            return this;
        }

        public final RequestBasedServiceLevelIndicator.Builder getRequestBasedSli() {
            return requestBasedSli != null ? requestBasedSli.toBuilder() : null;
        }

        public final void setRequestBasedSli(RequestBasedServiceLevelIndicator.BuilderImpl requestBasedSli) {
            this.requestBasedSli = requestBasedSli != null ? requestBasedSli.build() : null;
        }

        @Override
        public final Builder requestBasedSli(RequestBasedServiceLevelIndicator requestBasedSli) {
            this.requestBasedSli = requestBasedSli;
            return this;
        }

        public final String getEvaluationType() {
            return evaluationType;
        }

        public final void setEvaluationType(String evaluationType) {
            this.evaluationType = evaluationType;
        }

        @Override
        public final Builder evaluationType(String evaluationType) {
            this.evaluationType = evaluationType;
            return this;
        }

        @Override
        public final Builder evaluationType(EvaluationType evaluationType) {
            this.evaluationType(evaluationType == null ? null : evaluationType.toString());
            return this;
        }

        public final Goal.Builder getGoal() {
            return goal != null ? goal.toBuilder() : null;
        }

        public final void setGoal(Goal.BuilderImpl goal) {
            this.goal = goal != null ? goal.build() : null;
        }

        @Override
        public final Builder goal(Goal goal) {
            this.goal = goal;
            return this;
        }

        public final List<BurnRateConfiguration.Builder> getBurnRateConfigurations() {
            List<BurnRateConfiguration.Builder> result = BurnRateConfigurationsCopier.copyToBuilder(this.burnRateConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBurnRateConfigurations(Collection<BurnRateConfiguration.BuilderImpl> burnRateConfigurations) {
            this.burnRateConfigurations = BurnRateConfigurationsCopier.copyFromBuilder(burnRateConfigurations);
        }

        @Override
        public final Builder burnRateConfigurations(Collection<BurnRateConfiguration> burnRateConfigurations) {
            this.burnRateConfigurations = BurnRateConfigurationsCopier.copy(burnRateConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder burnRateConfigurations(BurnRateConfiguration... burnRateConfigurations) {
            burnRateConfigurations(Arrays.asList(burnRateConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder burnRateConfigurations(Consumer<BurnRateConfiguration.Builder>... burnRateConfigurations) {
            burnRateConfigurations(Stream.of(burnRateConfigurations)
                    .map(c -> BurnRateConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ServiceLevelObjective build() {
            return new ServiceLevelObjective(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
